=head1 NAME

AnyEvent - provide framework for multiple event loops

EV, Event, Coro::EV, Coro::Event, Glib, Tk, Perl - various supported event loops

=head1 SYNOPSIS

   use AnyEvent;

   my $w = AnyEvent->io (fh => $fh, poll => "r|w", cb => sub {
      ...
   });

   my $w = AnyEvent->timer (after => $seconds, cb => sub {
      ...
   });

   my $w = AnyEvent->condvar; # stores wether a condition was flagged
   $w->wait; # enters "main loop" till $condvar gets ->broadcast
   $w->broadcast; # wake up current and all future wait's

=head1 WHY YOU SHOULD USE THIS MODULE (OR NOT)

Glib, POE, IO::Async, Event... CPAN offers event models by the dozen
nowadays. So what is different about AnyEvent?

Executive Summary: AnyEvent is I<compatible>, AnyEvent is I<free of
policy> and AnyEvent is I<small and efficient>.

First and foremost, I<AnyEvent is not an event model> itself, it only
interfaces to whatever event model the main program happens to use in a
pragmatic way. For event models and certain classes of immortals alike,
the statement "there can only be one" is a bitter reality, and AnyEvent
helps hiding the differences.

The goal of AnyEvent is to offer module authors the ability to do event
programming (waiting for I/O or timer events) without subscribing to a
religion, a way of living, and most importantly: without forcing your
module users into the same thing by forcing them to use the same event
model you use.

For modules like POE or IO::Async (which is actually doing all I/O
I<synchronously>...), using them in your module is like joining a
cult: After you joined, you are dependent on them and you cannot use
anything else, as it is simply incompatible to everything that isn't
itself.

AnyEvent + POE works fine. AnyEvent + Glib works fine. AnyEvent + Tk
works fine etc. etc. but none of these work together with the rest: POE
+ IO::Async? no go. Tk + Event? no go. If your module uses one of
those, every user of your module has to use it, too. If your module
uses AnyEvent, it works transparently with all event models it supports
(including stuff like POE and IO::Async).

In addition of being free of having to use I<the one and only true event
model>, AnyEvent also is free of bloat and policy: with POE or similar
modules, you get an enourmous amount of code and strict rules you have
to follow. AnyEvent, on the other hand, is lean and to the point by only
offering the functionality that is useful, in as thin as a wrapper as
technically possible.

Of course, if you want lots of policy (this can arguably be somewhat
useful) and you want to force your users to use the one and only event
model, you should I<not> use this module.


=head1 DESCRIPTION

L<AnyEvent> provides an identical interface to multiple event loops. This
allows module authors to utilise an event loop without forcing module
users to use the same event loop (as only a single event loop can coexist
peacefully at any one time).

The interface itself is vaguely similar but not identical to the Event
module.

On the first call of any method, the module tries to detect the currently
loaded event loop by probing wether any of the following modules is
loaded: L<Coro::EV>, L<Coro::Event>, L<EV>, L<Event>, L<Glib>, L<Tk>. The
first one found is used. If none are found, the module tries to load these
modules in the order given. The first one that could be successfully
loaded will be used. If still none could be found, AnyEvent will fall back
to a pure-perl event loop, which is also not very efficient.

Because AnyEvent first checks for modules that are already loaded, loading
an Event model explicitly before first using AnyEvent will likely make
that model the default. For example:

   use Tk;
   use AnyEvent;

   # .. AnyEvent will likely default to Tk

The pure-perl implementation of AnyEvent is called
C<AnyEvent::Impl::Perl>. Like other event modules you can load it
explicitly.

=head1 WATCHERS

AnyEvent has the central concept of a I<watcher>, which is an object that
stores relevant data for each kind of event you are waiting for, such as
the callback to call, the filehandle to watch, etc.

These watchers are normal Perl objects with normal Perl lifetime. After
creating a watcher it will immediately "watch" for events and invoke
the callback. To disable the watcher you have to destroy it (e.g. by
setting the variable that stores it to C<undef> or otherwise deleting all
references to it).

All watchers are created by calling a method on the C<AnyEvent> class.

=head2 IO WATCHERS

You can create I/O watcher by calling the C<< AnyEvent->io >> method with
the following mandatory arguments:

C<fh> the Perl I<filehandle> (not filedescriptor) to watch for
events. C<poll> must be a string that is either C<r> or C<w>, that creates
a watcher waiting for "r"eadable or "w"ritable events. C<cb> the callback
to invoke everytime the filehandle becomes ready.

Filehandles will be kept alive, so as long as the watcher exists, the
filehandle exists, too.

Example:

   # wait for readability of STDIN, then read a line and disable the watcher
   my $w; $w = AnyEvent->io (fh => \*STDIN, poll => 'r', cb => sub {
      chomp (my $input = <STDIN>);
      warn "read: $input\n";
      undef $w;
   });

=head2 TIME WATCHERS

You can create a time watcher by calling the C<< AnyEvent->timer >>
method with the following mandatory arguments:

C<after> after how many seconds (fractions are supported) should the timer
activate. C<cb> the callback to invoke.

The timer callback will be invoked at most once: if you want a repeating
timer you have to create a new watcher (this is a limitation by both Tk
and Glib).

Example:

   # fire an event after 7.7 seconds
   my $w = AnyEvent->timer (after => 7.7, cb => sub {
      warn "timeout\n";
   });

   # to cancel the timer:
   undef $w;

=head2 CONDITION WATCHERS

Condition watchers can be created by calling the C<< AnyEvent->condvar >>
method without any arguments.

A condition watcher watches for a condition - precisely that the C<<
->broadcast >> method has been called.

Note that condition watchers recurse into the event loop - if you have
two watchers that call C<< ->wait >> in a round-robbin fashion, you
lose. Therefore, condition watchers are good to export to your caller, but
you should avoid making a blocking wait, at least in callbacks, as this
usually asks for trouble.

The watcher has only two methods:

=over 4

=item $cv->wait

Wait (blocking if necessary) until the C<< ->broadcast >> method has been
called on c<$cv>, while servicing other watchers normally.

You can only wait once on a condition - additional calls will return
immediately.

Not all event models support a blocking wait - some die in that case
(programs might want to do that so they stay interactive), so I<if you
are using this from a module, never require a blocking wait>, but let the
caller decide wether the call will block or not (for example, by coupling
condition variables with some kind of request results and supporting
callbacks so the caller knows that getting the result will not block,
while still suppporting blocking waits if the caller so desires).

Another reason I<never> to C<< ->wait >> in a module is that you cannot
sensibly have two C<< ->wait >>'s in parallel, as that would require
multiple interpreters or coroutines/threads, none of which C<AnyEvent>
can supply (the coroutine-aware backends C<Coro::EV> and C<Coro::Event>
explicitly support concurrent C<< ->wait >>'s from different coroutines,
however).

=item $cv->broadcast

Flag the condition as ready - a running C<< ->wait >> and all further
calls to C<wait> will return after this method has been called. If nobody
is waiting the broadcast will be remembered..

Example:

   # wait till the result is ready
   my $result_ready = AnyEvent->condvar;

   # do something such as adding a timer
   # or socket watcher the calls $result_ready->broadcast
   # when the "result" is ready.

   $result_ready->wait;

=back

=head2 SIGNAL WATCHERS

You can listen for signals using a signal watcher, C<signal> is the signal
I<name> without any C<SIG> prefix. Multiple signals events can be clumped
together into one callback invocation, and callback invocation might or
might not be asynchronous.

These watchers might use C<%SIG>, so programs overwriting those signals
directly will likely not work correctly.

Example: exit on SIGINT

   my $w = AnyEvent->signal (signal => "INT", cb => sub { exit 1 });

=head2 CHILD PROCESS WATCHERS

You can also listen for the status of a child process specified by the
C<pid> argument (or any child if the pid argument is 0). The watcher will
trigger as often as status change for the child are received. This works
by installing a signal handler for C<SIGCHLD>. The callback will be called with
the pid and exit status (as returned by waitpid).

Example: wait for pid 1333

  my $w = AnyEvent->child (pid => 1333, cb => sub { warn "exit status $?" });

=head1 GLOBALS

=over 4

=item $AnyEvent::MODEL

Contains C<undef> until the first watcher is being created. Then it
contains the event model that is being used, which is the name of the
Perl class implementing the model. This class is usually one of the
C<AnyEvent::Impl:xxx> modules, but can be any other class in the case
AnyEvent has been extended at runtime (e.g. in I<rxvt-unicode>).

The known classes so far are:

   AnyEvent::Impl::CoroEV    based on Coro::EV, best choice.
   AnyEvent::Impl::CoroEvent based on Coro::Event, second best choice.
   AnyEvent::Impl::EV        based on EV (an interface to libev, also best choice).
   AnyEvent::Impl::Event     based on Event, also second best choice :)
   AnyEvent::Impl::Glib      based on Glib, third-best choice.
   AnyEvent::Impl::Tk        based on Tk, very bad choice.
   AnyEvent::Impl::Perl      pure-perl implementation, inefficient but portable.

=item AnyEvent::detect

Returns C<$AnyEvent::MODEL>, forcing autodetection of the event model if
necessary. You should only call this function right before you would have
created an AnyEvent watcher anyway, that is, very late at runtime.

=back

=head1 WHAT TO DO IN A MODULE

As a module author, you should "use AnyEvent" and call AnyEvent methods
freely, but you should not load a specific event module or rely on it.

Be careful when you create watchers in the module body - Anyevent will
decide which event module to use as soon as the first method is called, so
by calling AnyEvent in your module body you force the user of your module
to load the event module first.

=head1 WHAT TO DO IN THE MAIN PROGRAM

There will always be a single main program - the only place that should
dictate which event model to use.

If it doesn't care, it can just "use AnyEvent" and use it itself, or not
do anything special and let AnyEvent decide which implementation to chose.

If the main program relies on a specific event model (for example, in Gtk2
programs you have to rely on either Glib or Glib::Event), you should load
it before loading AnyEvent or any module that uses it, generally, as early
as possible. The reason is that modules might create watchers when they
are loaded, and AnyEvent will decide on the event model to use as soon as
it creates watchers, and it might chose the wrong one unless you load the
correct one yourself.

You can chose to use a rather inefficient pure-perl implementation by
loading the C<AnyEvent::Impl::Perl> module, but letting AnyEvent chose is
generally better.

=cut

package AnyEvent;

no warnings;
use strict;

use Carp;

our $VERSION = '3.1';
our $MODEL;

our $AUTOLOAD;
our @ISA;

our $verbose = $ENV{PERL_ANYEVENT_VERBOSE}*1;

our @REGISTRY;

my @models = (
   [Coro::EV::             => AnyEvent::Impl::CoroEV::],
   [Coro::Event::          => AnyEvent::Impl::CoroEvent::],
   [EV::                   => AnyEvent::Impl::EV::],
   [Event::                => AnyEvent::Impl::Event::],
   [Glib::                 => AnyEvent::Impl::Glib::],
   [Tk::                   => AnyEvent::Impl::Tk::],
   [AnyEvent::Impl::Perl:: => AnyEvent::Impl::Perl::],
);

our %method = map +($_ => 1), qw(io timer condvar broadcast wait signal one_event DESTROY);

sub detect() {
   unless ($MODEL) {
      no strict 'refs';

      # check for already loaded models
      for (@REGISTRY, @models) {
         my ($package, $model) = @$_;
         if (${"$package\::VERSION"} > 0) {
            if (eval "require $model") {
               $MODEL = $model;
               warn "AnyEvent: found model '$model', using it.\n" if $verbose > 1;
               last;
            }
         }
      }

      unless ($MODEL) {
         # try to load a model

         for (@REGISTRY, @models) {
            my ($package, $model) = @$_;
            if (eval "require $package"
                and ${"$package\::VERSION"} > 0
                and eval "require $model") {
               $MODEL = $model;
               warn "AnyEvent: autoprobed and loaded model '$model', using it.\n" if $verbose > 1;
               last;
            }
         }

         $MODEL
           or die "No event module selected for AnyEvent and autodetect failed. Install any one of these modules: EV (or Coro+EV), Event (or Coro+Event), Glib or Tk.";
      }

      unshift @ISA, $MODEL;
      push @{"$MODEL\::ISA"}, "AnyEvent::Base";
   }

   $MODEL
}

sub AUTOLOAD {
   (my $func = $AUTOLOAD) =~ s/.*://;

   $method{$func}
      or croak "$func: not a valid method for AnyEvent objects";

   detect unless $MODEL;

   my $class = shift;
   $class->$func (@_);
}

package AnyEvent::Base;

# default implementation for ->condvar, ->wait, ->broadcast

sub condvar {
   bless \my $flag, "AnyEvent::Base::CondVar"
}

sub AnyEvent::Base::CondVar::broadcast {
   ${$_[0]}++;
}

sub AnyEvent::Base::CondVar::wait {
   AnyEvent->one_event while !${$_[0]};
}

# default implementation for ->signal

our %SIG_CB;

sub signal {
   my (undef, %arg) = @_;

   my $signal = uc $arg{signal}
      or Carp::croak "required option 'signal' is missing";

   $SIG_CB{$signal}{$arg{cb}} = $arg{cb};
   $SIG{$signal} ||= sub {
      $_->() for values %{ $SIG_CB{$signal} || {} };
   };

   bless [$signal, $arg{cb}], "AnyEvent::Base::Signal"
}

sub AnyEvent::Base::Signal::DESTROY {
   my ($signal, $cb) = @{$_[0]};

   delete $SIG_CB{$signal}{$cb};

   $SIG{$signal} = 'DEFAULT' unless keys %{ $SIG_CB{$signal} };
}

# default implementation for ->child

our %PID_CB;
our $CHLD_W;
our $CHLD_DELAY_W;
our $PID_IDLE;
our $WNOHANG;

sub _child_wait {
   while (0 < (my $pid = waitpid -1, $WNOHANG)) {
      $_->($pid, $?) for (values %{ $PID_CB{$pid} || {} }),
                         (values %{ $PID_CB{0}    || {} });
   }

   undef $PID_IDLE;
}

sub _sigchld {
   # make sure we deliver these changes "synchronous" with the event loop.
   $CHLD_DELAY_W ||= AnyEvent->timer (after => 0, cb => sub {
      undef $CHLD_DELAY_W;
      &_child_wait;
   });
}

sub child {
   my (undef, %arg) = @_;

   defined (my $pid = $arg{pid} + 0)
      or Carp::croak "required option 'pid' is missing";

   $PID_CB{$pid}{$arg{cb}} = $arg{cb};

   unless ($WNOHANG) {
      $WNOHANG = eval { require POSIX; &POSIX::WNOHANG } || 1;
   }

   unless ($CHLD_W) {
      $CHLD_W = AnyEvent->signal (signal => 'CHLD', cb => \&_sigchld);
      # child could be a zombie already, so make at least one round
      &_sigchld;
   }

   bless [$pid, $arg{cb}], "AnyEvent::Base::Child"
}

sub AnyEvent::Base::Child::DESTROY {
   my ($pid, $cb) = @{$_[0]};

   delete $PID_CB{$pid}{$cb};
   delete $PID_CB{$pid} unless keys %{ $PID_CB{$pid} };

   undef $CHLD_W unless keys %PID_CB;
}

=head1 SUPPLYING YOUR OWN EVENT MODEL INTERFACE

If you need to support another event library which isn't directly
supported by AnyEvent, you can supply your own interface to it by
pushing, before the first watcher gets created, the package name of
the event module and the package name of the interface to use onto
C<@AnyEvent::REGISTRY>. You can do that before and even without loading
AnyEvent.

Example:

   push @AnyEvent::REGISTRY, [urxvt => urxvt::anyevent::];

This tells AnyEvent to (literally) use the C<urxvt::anyevent::>
package/class when it finds the C<urxvt> package/module is loaded. When
AnyEvent is loaded and asked to find a suitable event model, it will
first check for the presence of urxvt.

The class should provide implementations for all watcher types (see
L<AnyEvent::Impl::Event> (source code), L<AnyEvent::Impl::Glib>
(Source code) and so on for actual examples, use C<perldoc -m
AnyEvent::Impl::Glib> to see the sources).

The above isn't fictitious, the I<rxvt-unicode> (a.k.a. urxvt)
uses the above line as-is. An interface isn't included in AnyEvent
because it doesn't make sense outside the embedded interpreter inside
I<rxvt-unicode>, and it is updated and maintained as part of the
I<rxvt-unicode> distribution.

I<rxvt-unicode> also cheats a bit by not providing blocking access to
condition variables: code blocking while waiting for a condition will
C<die>. This still works with most modules/usages, and blocking calls must
not be in an interactive application, so it makes sense.

=head1 ENVIRONMENT VARIABLES

The following environment variables are used by this module:

C<PERL_ANYEVENT_VERBOSE> when set to C<2> or higher, reports which event
model gets used.

=head1 EXAMPLE

The following program uses an io watcher to read data from stdin, a timer
to display a message once per second, and a condvar to exit the program
when the user enters quit:

   use AnyEvent;

   my $cv = AnyEvent->condvar;

   my $io_watcher = AnyEvent->io (fh => \*STDIN, poll => 'r', cb => sub {
      warn "io event <$_[0]>\n";   # will always output <r>
      chomp (my $input = <STDIN>); # read a line
      warn "read: $input\n";       # output what has been read
      $cv->broadcast if $input =~ /^q/i; # quit program if /^q/i
   });

   my $time_watcher; # can only be used once

   sub new_timer {
      $timer = AnyEvent->timer (after => 1, cb => sub {
         warn "timeout\n"; # print 'timeout' about every second
         &new_timer; # and restart the time
      });
   }

   new_timer; # create first timer

   $cv->wait; # wait until user enters /^q/i

=head1 REAL-WORLD EXAMPLE

Consider the L<Net::FCP> module. It features (among others) the following
API calls, which are to freenet what HTTP GET requests are to http:

   my $data = $fcp->client_get ($url); # blocks

   my $transaction = $fcp->txn_client_get ($url); # does not block
   $transaction->cb ( sub { ... } ); # set optional result callback
   my $data = $transaction->result; # possibly blocks

The C<client_get> method works like C<LWP::Simple::get>: it requests the
given URL and waits till the data has arrived. It is defined to be:

   sub client_get { $_[0]->txn_client_get ($_[1])->result }

And in fact is automatically generated. This is the blocking API of
L<Net::FCP>, and it works as simple as in any other, similar, module.

More complicated is C<txn_client_get>: It only creates a transaction
(completion, result, ...) object and initiates the transaction.

   my $txn = bless { }, Net::FCP::Txn::;

It also creates a condition variable that is used to signal the completion
of the request:

   $txn->{finished} = AnyAvent->condvar;

It then creates a socket in non-blocking mode.

   socket $txn->{fh}, ...;
   fcntl $txn->{fh}, F_SETFL, O_NONBLOCK;
   connect $txn->{fh}, ...
      and !$!{EWOULDBLOCK}
      and !$!{EINPROGRESS}
      and Carp::croak "unable to connect: $!\n";

Then it creates a write-watcher which gets called whenever an error occurs
or the connection succeeds:

   $txn->{w} = AnyEvent->io (fh => $txn->{fh}, poll => 'w', cb => sub { $txn->fh_ready_w });

And returns this transaction object. The C<fh_ready_w> callback gets
called as soon as the event loop detects that the socket is ready for
writing.

The C<fh_ready_w> method makes the socket blocking again, writes the
request data and replaces the watcher by a read watcher (waiting for reply
data). The actual code is more complicated, but that doesn't matter for
this example:

   fcntl $txn->{fh}, F_SETFL, 0;
   syswrite $txn->{fh}, $txn->{request}
      or die "connection or write error";
   $txn->{w} = AnyEvent->io (fh => $txn->{fh}, poll => 'r', cb => sub { $txn->fh_ready_r });

Again, C<fh_ready_r> waits till all data has arrived, and then stores the
result and signals any possible waiters that the request ahs finished:

   sysread $txn->{fh}, $txn->{buf}, length $txn->{$buf};

   if (end-of-file or data complete) {
     $txn->{result} = $txn->{buf};
     $txn->{finished}->broadcast;
     $txb->{cb}->($txn) of $txn->{cb}; # also call callback
   }

The C<result> method, finally, just waits for the finished signal (if the
request was already finished, it doesn't wait, of course, and returns the
data:

   $txn->{finished}->wait;
   return $txn->{result};

The actual code goes further and collects all errors (C<die>s, exceptions)
that occured during request processing. The C<result> method detects
wether an exception as thrown (it is stored inside the $txn object)
and just throws the exception, which means connection errors and other
problems get reported tot he code that tries to use the result, not in a
random callback.

All of this enables the following usage styles:

1. Blocking:

   my $data = $fcp->client_get ($url);

2. Blocking, but running in parallel:

   my @datas = map $_->result,
                  map $fcp->txn_client_get ($_),
                     @urls;

Both blocking examples work without the module user having to know
anything about events.

3a. Event-based in a main program, using any supported event module:

   use EV;

   $fcp->txn_client_get ($url)->cb (sub {
      my $txn = shift;
      my $data = $txn->result;
      ...
   });

   EV::loop;

3b. The module user could use AnyEvent, too:

   use AnyEvent;

   my $quit = AnyEvent->condvar;

   $fcp->txn_client_get ($url)->cb (sub {
      ...
      $quit->broadcast;
   });

   $quit->wait;

=head1 SEE ALSO

Event modules: L<Coro::EV>, L<EV>, L<EV::Glib>, L<Glib::EV>,
L<Coro::Event>, L<Event>, L<Glib::Event>, L<Glib>, L<Coro>, L<Tk>.

Implementations: L<AnyEvent::Impl::CoroEV>, L<AnyEvent::Impl::EV>,
L<AnyEvent::Impl::CoroEvent>, L<AnyEvent::Impl::Event>,
L<AnyEvent::Impl::Glib>, L<AnyEvent::Impl::Tk>, L<AnyEvent::Impl::Perl>.

Nontrivial usage examples: L<Net::FCP>, L<Net::XMPP2>.

=head1

=cut

1

