package Class::Value;

use strict;
use warnings;
use Error ':try';
use Class::Value::DefaultNotify;
use Error::Hierarchy::Container;


our $VERSION = '0.01';


use base qw(Class::Accessor::Complex Class::Accessor::Constructor);


# Use strings for overloading sub names so they're interpreted as method names
# and the methods can be overridden in subclasses while continuing to work as
# expected. The overloaded operations have been chosen so that as much as
# necessary can be autogenerated (see "perldoc overload": MAGIC
# AUTOGENERATION). Subclasses are free to provide custom overloads of
# autogenerated methods, of course.

use overload
    '+'     => 'add',
    '-'     => 'subtract',
    '*'     => 'multiply',
    '/'     => 'divide',
    '**'    => 'power',
    '%'     => 'modulo',

    cmp     => 'str_cmp',
    '<=>'   => 'num_cmp',

    '<<'    => 'bit_shift_left',
    '>>'    => 'bit_shift_right',
    '&'     => 'bit_and',
    '|'     => 'bit_or',
    '^'     => 'bit_xor',
    '~'     => 'bit_not',

    'atan2' => 'atan2',
    'cos'   => 'cos',
    'sin'   => 'sin',
    'exp'   => 'exp',
    'log'   => 'log',
    'sqrt'  => 'sqrt',
    'int'   => 'int',

    '""'    => 'stringify',
    '<>'    => 'iterate';


__PACKAGE__
    ->mk_constructor_with_dirty
    ->mk_scalar_accessors(qw(notify_delegate))
    ->mk_object_accessors(
        'Error::Hierarchy::Container' => 'exception_container'
    );


use constant UNHYGIENIC => ( qw/value/ );


# our(), not a static boolean method, so it can be local()'ed.

our $SkipChecks           = 0;
our $SkipNormalizations   = 0;
our $SkipDirtying         = 0;
our $ThrowSingleException = 0;


sub skip_checks {
    our $SkipChecks;
    return $SkipChecks if @_ == 1;
    $SkipChecks = $_[1];
}


sub skip_normalizations {
    our $SkipNormalizations;
    return $SkipNormalizations if @_ == 1;
    $SkipNormalizations = $_[1];
}


sub skip_dirtying {
    our $SkipDirtying;
    return $SkipDirtying if @_ == 1;
    $SkipDirtying = $_[1];
}


sub throw_single_exception {
    our $ThrowSingleException;
    return $ThrowSingleException if @_ == 1;
    $ThrowSingleException = $_[1];
}


# to temporarily skip checks, use
#
#   {
#     local $Class::Value::SkipChecks = 1;
#     ...
#   }
#
# or
#
#   Class::Value->skip_checks(1);
#
# but then you have to remember to reset it again with
#
#   Class::Value->skip_checks(0);

# Every value object gets the same notify delegate object

use constant DEFAULTS => (
    notify_delegate => (our $DELEGATE ||= Class::Value::DefaultNotify->new),
);

use constant FIRST_CONSTRUCTOR_ARGS => ('notify_delegate');

sub MUNGE_CONSTRUCTOR_ARGS {
    my ($self, @args) = @_;

    if (@args == 1 && ref($args[0]) eq 'HASH') {
        @args = %{ $_[0] };
    } elsif (@args % 2) {
        # odd number of args
        @args = (value => @args);
    }

    if (UNIVERSAL::isa($self, 'Class::Framework::Storable')) {
        @args =
            $self->Class::Framework::Storable::MUNGE_CONSTRUCTOR_ARGS(@args);
    }

    @args;

    # return %{ $_[0] }    if @_ == 1 && ref($_[0]) eq 'HASH';
    # return (value => @_) if @_ % 2;      # odd number of args
    # return @_;
}


sub init {}


# value() is the method to use when getting or setting values. It includes the
# checks for well-formedness and validity. If you absolutely need to set an
# unvalidated value, use set_value() for that. The value stored by set_value()
# could be anything - a scalar, array, hash, coderef, another object etc.
# Specific value objects could also accept various input and decode it into
# the underlying components.
#
# Consider, for example, a date value class that internally stores year, month
# and day components (using accessors generated by Class::Accessor::*). The
# public interface for the value object would still be value(); outside code
# never needs to know what's happening behind the scenes. However, outside
# code could call year(), month() and day() on the value object and get at the
# components.
#
# The date value class would override set_value() to parse the input, split it
# into the components. It would also override is_well_formed_value() and/or
# is_valid_value() to do some checking. And it would override get_value to
# return the components joined up into a date string. Thus, stringify() would
# continue to work as expected.

sub value {
    my $self = shift;
    return $self->get_value unless @_;
    my $value = shift;

    # run_checks() returns normalized value; check even undefined values -
    # individual value objects have to decide whether undef is an acceptable
    # value for them.

    if (our $SkipChecks) {

        # When skipping checks, we still try to normalize the value, unless
        # told not to. If we can't normalize it (i.e., normalize_value()
        # returns undef), we use the value we were given. Don't try to
        # normalize an undefined value lest some normalize_value method checks
        # via assert_defined.
        #
        # Skipping even normalization is useful if you want to purposefully
        # set an denormalized value and later check whether run_checks()
        # properly normalizes it.

        our $SkipNormalizations;
        if (defined($value) && !$SkipNormalizations) {
            my $normalized = $self->normalize_value($value);
            $value = $normalized if defined $normalized;
        }
    } else {
        $value = $self->run_checks($value);
    }

    $self->set_value($value);
    $self->clear_dirty if our $SkipDirtying;
    $value;   # return for convenience
}


# Subclasses might want to override this if they don't use a custom notify
# delegate but choose to throw a fixed exception.

sub send_notify_value_not_wellformed {
    my ($self, $value) = @_;
    $self->notify_delegate->notify_value_not_wellformed(ref($self), $value);
}


sub send_notify_value_invalid {
    my ($self, $value) = @_;
    $self->notify_delegate->notify_value_invalid(ref($self), $value);
}


sub send_notify_value_normalized {
    my ($self, $value, $normalized) = @_;
    $self->notify_delegate->
        notify_value_normalized(ref($self), $value, $normalized);
}


sub get_value { $_[0]->{_value}         }
sub set_value { $_[0]->{_value} = $_[1] }


sub is_defined {
    my $self = shift;
    defined $self->get_value;
}


sub is_well_formed {
    my $self = shift;
    $self->is_well_formed_value(@_ ? shift : $self->value);
}


sub is_well_formed_value { 1 }


sub is_valid {
    my $self = shift;
    $self->is_valid_value(@_ ? shift : $self->value);
}


sub is_valid_value {
    my ($self, $value) = @_;

    # value can be undef
    return 1 unless defined $value;

    my $normalized = $self->normalize_value($value);
    return 0 unless defined $normalized;
    $self->is_valid_normalized_value($normalized);
}


sub is_valid_normalized_value {
    my ($self, $normalized) = @_;
    defined $normalized;
}


sub normalize_value {
    my ($self, $value) = @_;
    $value;
}


sub check {
    my $self = shift;
    my $value = @_ ? shift : $self->value;
    $self->is_well_formed($value) && $self->is_valid($value);
}


sub run_checks {
    my $self = shift;

    $self->exception_container->items_clear;
    my $value = @_ ? shift : $self->value;
    $self->is_well_formed($value) ||
        $self->send_notify_value_not_wellformed($value);

    $self->is_valid($value) || $self->send_notify_value_invalid($value);

    my $normalized = $self->normalize($value);
    if (defined($value) && defined($normalized) && ($value ne $normalized)) {
        $self->send_notify_value_normalized($value, $normalized);
    }

    if (my $count = $self->exception_container->items_count) {
        if ($count == 1 && our $ThrowSingleException) {
            $self->exception_container->items->[0]->throw;
        } else {
            $self->exception_container->throw;
        }
    }
    $normalized;
}


sub run_checks_with_exception_container {
    my $self = shift;
    my $exception_container = shift;
    my $value = @_ ? shift : $self->value;
    try {
        $self->run_checks($value);
    } catch Error with {
        $exception_container->items_set_push($_[0]);
    };

    # We only needed to fill the value object's exception container during
    # run_checks; now the exceptions have wandered into the exception
    # container that was passed to us, we don't need the value object's
    # exception container anymore.

    $self->exception_container->items_clear;
}


sub normalize {
    my ($self, $value) = @_;
    my $normalized = $self->normalize_value($value);
    if (defined $value) {
        if (defined $normalized) {
            if ($value ne $normalized) {
                $self->send_notify_value_normalized($value, $normalized);
            }
        } else {
            # can't normalize; treat as invalid value
            $self->send_notify_value_invalid($value);
        }
    }
    $normalized;
}


# have the Class::Value be restrictive with respect to operations on the
# value; subclasses can then define certain operations.

for my $op (qw/add subtract multiply divide power modulo num_cmp
                bit_shift_left bit_shift_right bit_and bit_or bit_xor bit_not
                atan2 cos sin exp log sqrt int iterate
            /) {
    no strict 'refs';
    *{$op} = sub {
        require Class::Value::Exception::UnsupportedOperation;
        throw Class::Value::Exception::UnsupportedOperation(
            ref     => ref($_[0]),
            value   => $_[0],
            opname  => $op,
        );
    };
}


sub str_cmp {
    sprintf("%s", ($_[0] || ''))
              cmp
    sprintf("%s", ($_[1] || ''))
}


sub stringify { $_[0]->value }


sub comparable {
    my $self = shift;
    my $value = $self->value;

    # Convert the value into a string, because eq_or_diff seems to make a
    # difference between strings and numbers.

    defined $value ? "$value" : '';
}


1;


__END__

=head1 NAME

Class::Value - foobar

=head1 SYNOPSIS

None yet (see below).

=head1 DESCRIPTION

None yet. This is an early release; fully functional, but undocumented. The
next release will have more documentation.

=head1 TAGS

If you talk about this module in blogs, on del.icio.us or anywhere else,
please use the C<classvalue> tag.

=head1 BUGS AND LIMITATIONS

No bugs have been reported.

Please report any bugs or feature requests to
C<bug-class-value@rt.cpan.org>, or through the web interface at
L<http://rt.cpan.org>.

=head1 INSTALLATION

See perlmodinstall for information and options on installing Perl modules.

=head1 AVAILABILITY

The latest version of this module is available from the Comprehensive Perl
Archive Network (CPAN). Visit <http://www.perl.com/CPAN/> to find a CPAN
site near you. Or see <http://www.perl.com/CPAN/authors/id/M/MA/MARCEL/>.

=head1 AUTHOR

Marcel GrE<uuml>nauer, C<< <marcel@cpan.org> >>

=head1 COPYRIGHT AND LICENSE

Copyright 2007 by Marcel GrE<uuml>nauer

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

