#!perl

use warnings;
use strict;
use lib q(lib);
use Benchmark;
use NetAddr::IP;
use Test::More;

my $count = 1;
my $text = do { local $/; <DATA> };

plan tests => 1 + 2 * $count + 3;

use_ok("Net::ISC::DHCPd::Config");

my $time = timeit($count, sub {
    my $config = Net::ISC::DHCPd::Config->new;

    is(ref $config, "Net::ISC::DHCPd::Config", "config object constructed");

    $config->add_keyvalue(
        name => 'ddns-update-style',
        value => 'none',
    );
    $config->add_optionspace(
        name => 'foo-enc',
        prefix => 'foo',
        code => 122,
        options => [
            {
                name => 'bar',
                code => 1,
                value => 'ip-address',
            },
        ],
    );
    $config->add_function(
        name => "commit",
        body => "    set leasetime = encode-int(lease-time, 32);\n",
    );
    $config->add_subnet(
        address => NetAddr::IP->new('10.0.0.96/27'),
        options => [
            {
                name => 'routers',
                value => '10.0.0.97',
            },
        ],
        pools => [
            {
                ranges => [
                    {
                        upper => NetAddr::IP->new("10.0.0.116"),
                        lower => NetAddr::IP->new("10.0.0.126"),
                    },
                ],
            },
        ],
    );
    $config->add_host(
        name => 'foo',
        keyvalues => [
            {
                name => 'fixed-address',
                value => '10.19.83.102',
            },
        ],
        filenames => [
            {
                file => 'pxelinux.0',
            },
        ],
    );

    #print $config->generate;

    is($config->generate, $text, "config generated");
});

diag($count .": " .timestr($time));

{
    my $config = Net::ISC::DHCPd::Config->new;
    my $include_file = 't/data/foo-included.conf';
    my $include_text;

    {
        open my $INCLUDE, '<', $include_file or BAIL_OUT $!;
        local $/;
        $include_text = <$INCLUDE>;
        $include_text =~ s/^\n+//m;
        $include_text =~ s/\n+$//m;
        $include_text .= "\n";
    }

    $config->add_include(file => $include_file);

    is($config->generate, qq(include "$include_file";\n), 'include file generated');

    $config->includes->[0]->generate_with_include(1);
    like($config->generate, qr{forgot to parse}, 'include file content cannot be generated before parsed');

    $config->includes->[0]->parse;
    is($config->generate, "$include_text", 'include file content generated');
}

__DATA__
ddns-update-style none;
option space foo;
    option foo.bar code 1 = ip-address;
option foo-enc code 122 = encapsulate foo;
on commit {
    set leasetime = encode-int(lease-time, 32);
}
subnet 10.0.0.96 netmask 255.255.255.224 {
    option routers 10.0.0.97;
    pool {
        range 10.0.0.126 10.0.0.116;
    }
}
host foo {
    fixed-address 10.19.83.102;
    filename pxelinux.0;
}
