# Verify Mersenne Twister algorithm
# Test standalone PRNG - srand(\&sub), get_/set_seed() and get_/set_state()
# Test OO interface - seeding from file, $obj->get_/set_seed() and
#                                               $obj->get_/set_state()
# Test PRNG 'cloning'

use strict;
use warnings;

use Test::More 'no_plan';
use Config;

my @WARN;
BEGIN {
    # Warning signal handler
    $SIG{__WARN__} = sub { push(@WARN, @_); };

    use_ok('Math::Random::MT::Auto', qw(rand irand srand get_seed set_seed
                                        get_state set_state :!auto));
};

can_ok('Math::Random::MT::Auto', qw(rand irand srand get_seed set_seed
                                    get_state set_state));
can_ok('main', qw(rand irand srand get_seed set_seed
                                    get_state set_state));

# Known test values for irand()
my @base_rint = ($Config{'uvsize'} == 8) ?
  (  # 64-bit randoms
     7266447313870364031,  4946485549665804864, 16945909448695747420, 16394063075524226720,  4873882236456199058,
    14877448043947020171,  6740343660852211943, 13857871200353263164,  5249110015610582907, 10205081126064480383,
     1235879089597390050, 17320312680810499042, 16489141110565194782,  8942268601720066061, 13520575722002588570,
    14226945236717732373,  9383926873555417063, 15690281668532552105, 11510704754157191257, 15864264574919463609,
     6489677788245343319,  5112602299894754389, 10828930062652518694, 15942305434158995996, 15445717675088218264,
     4764500002345775851, 14673753115101942098,   236502320419669032, 13670483975188204088, 14931360615268175698,
     8904234204977263924, 12836915408046564963, 12120302420213647524, 15755110976537356441,  5405758943702519480,
    10951858968426898805, 17251681303478610375,  4144140664012008120, 18286145806977825275, 13075804672185204371,
    10831805955733617705,  6172975950399619139, 12837097014497293886, 12903857913610213846,   560691676108914154,
     1074659097419704618, 14266121283820281686, 11696403736022963346, 13383246710985227247,  7132746073714321322,
    10608108217231874211,  9027884570906061560, 12893913769120703138, 15675160838921962454,  2511068401785704737,
    14483183001716371453,  3774730664208216065,  5083371700846102796,  9583498264570933637, 17119870085051257224,
     5217910858257235075, 10612176809475689857,  1924700483125896976,  7171619684536160599, 10949279256701751503,
    15596196964072664893, 14097948002655599357,   615821766635933047,  5636498760852923045, 17618792803942051220,
      580805356741162327,   425267967796817241,  8381470634608387938, 13212228678420887626, 16993060308636741960,
      957923366004347591,  6210242862396777185,  1012818702180800310, 15299383925974515757, 17501832009465945633,
    17453794942891241229, 15807805462076484491,  8407189590930420827,   974125122787311712,  1861591264068118966,
      997568339582634050, 18046771844467391493, 17981867688435687790,  3809841506498447207,  9460108917638135678,
    16172980638639374310,   958022432077424298,  4393365126459778813, 13408683141069553686, 13900005529547645957,
    15773550354402817866, 16475327524349230602,  6260298154874769264, 12224576659776460914,  6405294864092763507,
     7585484664713203306,  5187641382818981381, 12435998400285353380, 13554353441017344755,   646091557254529188,
    11393747116974949255, 16797249248413342857, 15713519023537495495, 12823504709579858843,  4738086532119935073,
     4429068783387643752,   585582692562183870,  1048280754023674130,  6788940719869959076, 11670856244972073775,
     2488756775360218862,  2061695363573180185,  6884655301895085032,  3566345954323888697, 12784319933059041817,
     4772468691551857254,  6864898938209826895,  7198730565322227090,  2452224231472687253, 13424792606032445807,
    10827695224855383989, 11016608897122070904, 14683280565151378358,  7077866519618824360, 17487079941198422333,
     3956319990205097495,  5804870313319323478,  8017203611194497730,  3310931575584983808,  5009341981771541845,
     6930001938490791874, 14415278059151389495, 11001114762641844083,  6715939435439735925,   411419160297131328,
     4522402260441335284,  3381955501804126859, 15935778656111987797,  4345051260540166684, 13978444093099579683,
     9219789505504949817,  9245142924137529075, 11628184459157386459,  7242398879359936370,  8511401943157540109,
    11948130810477009827,  6865450671488705049, 13965005347172621081, 15956599226522058336,  7737868921014130584,
     2107342503741411693, 15818996300425101108, 16399939197527488760, 13971145494081508107,  3910681448359868691,
     4249175367970221090,  9735751321242454020, 12418107929362160460,   241792245481991138,  5806488997649497146,
    10724207982663648949,  1121862814449214435,  1326996977123564236,  4902706567834759475, 12782714623891689967,
     7306216312942796257, 15681656478863766664,   957364844878149318,  5651946387216554503,  8197027112357634782,
     6302075516351125977, 13454588464089597862, 15638309200463515550, 10116604639722073476, 12052913535387714920,
     2889379661594013754, 15383926144832314187,  7841953313015471731, 17310575136995821873,  9820021961316981626,
    15319619724109527290, 15349724127275899898, 10511508162402504492,  6289553862380300393, 15046218882019267110,
    11772020174577005930,  3537640779967351792,  6801855569284252424, 17687268231192623388, 12968358613633237218,
     1429775571144180123, 10427377732172208413, 12155566091986788996, 16465954421598296115, 12710429690464359999,
     9547226351541565595, 12156624891403410342,  2985938688676214686, 18066917785985010959,  5975570403614438776,
    11541343163022500560, 11115388652389704592,  9499328389494710074,  9247163036769651820,  3688303938005101774,
     2210483654336887556, 15458161910089693228,  6558785204455557683,  1288373156735958118, 18433986059948829624,
     3435082195390932486, 16822351800343061990,  3120532877336962310, 16681785111062885568,  7835551710041302304,
     2612798015018627203, 15083279177152657491,  6591467229462292195, 10592706450534565444,  7438147750787157163,
      323186165595851698,  7444710627467609883,  8473714411329896576,  2782675857700189492,  3383567662400128329,
     3200233909833521327, 12897601280285604448,  3612068790453735040,  8324209243736219497, 15789570356497723463,
     1083312926512215996,  4797349136059339390,  5556729349871544986, 18266943104929747076,  1620389818516182276,
      172225355691600141,  3034352936522087096,  1266779576738385285,  3906668377244742888,  6961783143042492788,
    17159706887321247572,  4676208075243319061, 10315634697142985816, 13435140047933251189,   716076639492622016,
    13847954035438697558,  7195811275139178570, 10815312636510328870,  6214164734784158515, 16412194511839921544,
     3862249798930641332,  1005482699535576005,  4644542796609371301, 17600091057367987283,  4209958422564632034,
     5419285945389823940, 11453701547564354601,  9951588026679380114,  7425168333159839689,  8436306210125134906,
    11216615872596820107,  3681345096403933680,  5770016989916553752, 11102855936150871733, 11187980892339693935,
      396336430216428875,  6384853777489155236,  7551613839184151117, 16527062023276943109, 13429850429024956898,
     9901753960477271766,  9731501992702612259,  5217575797614661659, 10311708346636548706, 15111747519735330483,
     4353415295139137513,  1845293119018433391, 11952006873430493561,  3531972641585683893, 16852246477648409827,
    15956854822143321380, 12314609993579474774, 16763911684844598963, 16392145690385382634,  1545507136970403756,
    17771199061862790062, 12121348462972638971, 12613068545148305776,   954203144844315208,  1257976447679270605,
     3664184785462160180,  2747964788443845091, 15895917007470512307, 15552935765724302120, 16366915862261682626,
     8385468783684865323, 10745343827145102946,  2485742734157099909,   916246281077683950, 15214206653637466707,
    12895483149474345798,  1079510114301747843, 10718876134480663664,  1259990987526807294,  8326303777037206221,
    14104661172014248293, 15531278677382192198,  3874303698666230242,  3611366553819264523,  1358753803061653874,
     1552102816982246938, 14492630642488100979, 15001394966632908727,  2273140352787320862, 17843678642369606172,
     2903980458593894032, 16971437123015263604, 12969653681729206264,  3593636458822318001,  9719758956915223015,
     7437601263394568346,  3327758049015164431, 17851524109089292731, 14769614194455139039,  8017093497335662337,
    12026985381690317404,   739616144640253634, 15535375191850690266,  2418267053891303448, 15314073759564095878,
    10333316143274529509, 16565481511572123421, 16317667579273275294, 13991958187675987741,  3753596784796798785,
     9078249094693663275,  8459506356724650587, 12579909555010529099,  7827737296967050903,  5489801927693999341,
    10995988997350541459, 14721747867313883304,  7915884580303296560,  4105766302083365910, 12455549072515054554,
    13602111324515032467,  5205971628932290989,  5034622965420036444,  9134927878875794005, 11319873529597990213,
    14815445109496752058,  2266601052460299470,  5696993487088103383,  6540200741841280242,  6631495948031875490,
     5328340585170897740, 17897267040961463930,  9030000260502624168, 14285709137129830926, 12854071997824681544,
    15408328651008978682,  1063314403033437073, 13765209628446252802,   242013711116865605,  4772374239432528212,
     2515855479965038648,  5872624715703151235, 14237704570091006662,   678604024776645862, 12329607334079533339,
    17570877682732917020,  2695443415284373666,  4312672841405514468,  6454343485137106900,  8425658828390111343,
    16335501385875554899,  5551095603809016713, 11781094401885925035,  9395557946368382509,  9765123360948816956,
    18107191819981188154, 16049267500594757404, 16349966108299794199,  1040405303135858246,  2366386386131378192,
      223761048139910454, 15375217587047847934, 15231693398695187454, 12916726640254571028,  8878036960829635584,
     1626201782473074365,  5758998126998248293, 18077917959300292758, 10585588923088536745, 15072345664541731497,
     3559348759319842667, 12744591691872202375,  2388494115860283059,  6414691845696331748,  3069528498807764495,
     8737958486926519702, 18059264986425101074,  3139684427605102737, 12378931902986734693,   410666675039477949,
    12139894855769838924,  5780722552400398675,  7039346665375142557,  3020733445712569008,  2612305843503943561,
    13651771214166527665, 16478681918975800939,   566088527565499576,  4715785502295754870,  6957318344287196220,
    11645756868405128885, 13139951104358618000, 17650948583490040612, 18168787973649736637,  5486282999836125542,
     6122201977153895166, 17324241605502052782, 10063523107521105867, 17537430712468011382, 10828407533637104262,
    10294139354198325113, 12557151830240236401, 16673044307512640231, 10918020421896090419, 11077531235278014145,
     5499571814940871256,  2334252435740638702, 18177461912527387031,  2000007376901262542,  7968425560071444214,
     1472650787501520648,  3115849849651526279,  7980970700139577536, 12153253535907642097,  8109716914843248719,
     3154976533165008908,  5553369513523832559, 10345792701798576501,  3677445364544507875, 10637177623943913351,
     7380255087060498096, 14479400372337014801, 15381362583330700960,   204531043189704802, 13699106540959723942,
     3817903465872254783, 10972364467110284934,  2701394334530963810,  2931625600749229147, 16428252083632828910,
    11873166501966812913,  5566810080537233762,  7840617383807795056, 10699413880206684652, 18259119259617231436,
    10332714341486317526, 10137911902863059694,   669146221352346842,  8373571610024623455, 10620002450820868661,
    12220730820779815970,  5902974968095412898,  7931010481705150841, 16413777368097063650, 11273457888324769727,
    13719113891065284171,  8327795098009702553, 10333342364827584837,  6202832891413866653,  9137034567886143162,
    14514450826524340059,   473610156015331016,   813689571029117640, 13776316799690285717, 10429708855338427756,
     8995290140880620858,  2320123852041754384,  8082864073645003641,  6961777411740398590, 10008644283003991179,
     3239064015890722333, 16762634970725218787, 16467281536733948427, 10563290046315192938,  5108560603794851559,
    15121667220761532906, 14155440077372845941, 10050536352394623377, 15474881667376037792,  3448088038819200619,
     3692020001240358871,  6444847992258394902,  8687650838094264665,  3028124591188972359, 16945232313401161629,
    15547830510283682816,  3982930188609442149, 14270781928849894661, 13768475593433447867, 13815150225221307677,
     8502397232429564693,   718377350715476994,  7459266877697905475,  8353375565171101521,  7807281661994435472,
    16924127046922196149, 10157812396471387805,  2519858716882670232,  7384148884750265792,  8077153156180046901,
     3499231286164597752,  2700106282881469611, 14679824700835879737, 14188324938219126828,  3016120398601032793,
    10858152824243889420,  9412371965669250534,  4857522662584941069,   984331743838900386,  4094160040294753142,
     2368635764350388458, 15101240511397838657, 15584415763303953578,  7831857200208015446,  1952643641639729063,
     4184323302594028609, 16795120381104846695,  3541559381538365280, 15408472870896842474,  5628362450757896366,
    16277348886873708846, 12437047172652330846, 10172715019035948149,  1999700669649752791,  6217957085626135027,
    11220551167830336823, 16478747645632411810,  5437280487207382147, 11382378739613087836, 15866932785489521505,
     5502694314775516684, 16440179278067648435, 15510104554374162846, 15722061259110909195, 10760687291786964354,
    10736868329920212671,  4166148127664495614, 14303518358120527892,  9122250801678898571, 10028508179936801946,
      216630713752669403, 10655207865433859491,  4041437116174699233,  6280982262534375348,   297501356638818866,
    13976146806363377485, 13752396481560145603, 11472199956603637419, 16393728429143900496, 14752844047515986640,
     1524477318846038424,  6596889774254235440,  1591982099532234960,  8065146456116391065,  3964696017750868345,
    17040425970526664920, 11511165586176539991,  3443401252003315103, 16314977947073778249, 16860120454903458341,
     5370503221561340846, 15362920279125264094,  2822458124714999779, 14575378304387898337,  9689406052675046032,
     2872149351415175149, 13019620945255883050, 14929026760148695825,  8503417349692327218,  9677798905341573754,
      828949921821462483, 16110482368362750196, 15794218816553655671, 14942910774764855088, 12026350906243760195,
    13610867176871462505, 18324536557697872582,  2658962269666727629,   327225403251576027,  9207535177029277544,
     8744129291351887858,  6129603385168921503, 18385497655031085907, 13024478718952333892, 14547683159720717167,
     5932119629366981711,   325385464632594563,  3559879386019806291,  6629264948665231298, 14358245326238118181,
    15662449672706340765, 13975503159145803297,  3609534220891499022,  4224273587485638227,  9274084767162416370,
    13156843921244091998, 18284750575626858789, 14664767920489118779, 11292057742031803221, 13919998707305829132,
    14473305049457001422,  9696877879685767807,  1406758246007973837,  2429517644459056881, 14361215588101587430,
    11386164476149757528, 10474116023593331839,  2921165656527786564, 15604610369733358953, 12955027028676000544,
    10314281035410779907,  3167047178514709947,  1088721329408346700, 17930425515478182741,  7466411836095405617,
    15534027454610690575, 10879629128927506091, 11502219301371200635, 13915106894453889418,  4226784327815861027,
    12335222183627106346,  3648499746356007767, 18441388887898023393, 18117929843327093625,  4237736098094830438,
    14229123019768296655,  3930112058127932690, 12663879236019645778,  9281161952002617309,  4978473890680876319,
      845759387067546611,  1386164484606776333,  8008554770639925512, 11159581016793288971, 18065390393740782906,
    17647985458967631018,  9092379465737744314,  2914678236848656327,  4376066698447630270, 16057186499919087528,
     3031333261848790078,  2926746602873431597,  7931945763526885287,   147649915388326849, 15801792398814946230,
     5265900391686545347, 16173686275871890830,  7562781050481886043,  5853506575839330404, 14957980734704564792,
    10944286556353523404,  1783009880614150597,  9529762028588888983,   822992871011696119,  2130074274744257510,
     8000279549284809219,  3514744284158856431,   128770032569293263,  3737367602618100572, 16364836605077998543,
      783266423471782696,  4569418252658970391, 11093950688157406886, 14888808512267628166,  4217786261273670948,
    17047486076688645713, 14133826721458860485, 17539744882220127106, 12394675039129853905,  5757634999463277090,
     9621947619435861331,  1182210208559436772, 14603391040490913939, 17481976703660945893, 14063388816234683976,
     2046622692581829572,  8294969799792017441,  5293778434844788058, 17976364049306763808,   399482430848083948,
    16495545010129798933, 15241340958282367519,   989828753826900814, 17616558773874893537,  2471817920909589004,
    11764082277667899978,  9618755269550400950,  1240014743757147125,  1887649378641563002,  1842982574728131416,
    13243531042427194002,  7688268125537013927,  3080422097287486736,  2562894809975407783, 12428984115620094788,
     1355581933694478148,  9895969242586224966,  8628445623963160889,  4298916726468199239, 12773165416305557280,
     5240726258301567487,  4975412836403427561,  1842172398579595303,  7812151462958058676, 17974510987263071769,
    14980707022065991200, 18294903201142729875, 12911672684850242753,  8979482998667235743, 16808468362384462073,
     5981317232108359798, 12373702800369335100, 16119707581920094765,  2782738549717633602, 15454155188515389391,
    16495638000603654629, 16348757069342790497,  7769562861984504567, 17504300515449231559,  5557710032938318996,
    11846125204788401203, 13957316349928882624,  2738350683717432043, 15738068448047700954,  6224714837294524999,
     6081930777706411111, 11366312928059597928,  4355315799925031482, 12393324728734964015, 15277140291994338591,
     1406052433297386355, 15859448364509213398,  1672805458341158435,  2926095111610982994, 11056431822276774455,
    12083767323511977430,  3296968762229741153, 12312076899982286460, 17769284994682227273, 15349428916826953443,
     1056147296359223910, 18305757538706977431,  6214378374180465222, 14279648441175008454, 17791306410319136644,
      956593013486324072,  2921235772936241950, 10002890515925652606, 10399654693663712506,  6446247931049971441,
     6380465770144534958, 11439178472613251620, 10131486500045494660,  3692642123868351947, 10972816599561388940,
     4931112976348785580,  8213967169213816566, 15336469859637867841, 15026830342847689383,  7524668622380765825,
    17309937346758783807,   372780684412666438,  5642417144539399955, 18303842993081194577, 11085303253831702827,
    15658163165983586950,  8517521928922081563, 16091186344159989860, 17614656488010863910,  4736067146481515156,
    13449945221374241354, 17755469346196579408, 13300502638545717375,  6611828134763118043, 14177591906740276597,
     9340430243077460347,  7499765399826404087,  3409518087967832469,  9013253864026602045,  4444307427984430192,
     3729283608700519712, 13642048880719588383, 16486557958022946240,  2996465014991157904, 10020049344596426576,
    12302485648009883778,  8492591321344423126, 17407986443716172520, 10530482934957373052, 15740662350540828750,
     1790629986901049436,  6305948377669917188, 15092985352503125323,   928505047232899787, 14404651977039851607,
     7564177565277805597,  3411236815351677870,  7752718145953236134, 12315979971311483798, 12477729506691004724,
    14654956300924793305,  6689803038918974388,  1540738812233000153, 13508351811701989957, 15864432023192136053,
     7990997967273843917,  7424300239290765161,    39585249496300263,  3877436595063283319, 10710642254398044448,
     4653804418844456375,  1232267496410380283,  3690525514009038824, 15459770765077428485, 13240346522153894145,
     5674964360688390624, 16973644653010587289, 15924280764204855206, 15196708627253442662, 17596174821341373274,
    16196745023027393691,  6980050627399795351, 17582264380857746637, 18170372407506856324, 12108126025631005514,
    15687749089493373169,  5814107289258228434,  9381977959648494876, 15895601183088112734, 16267869075651604263,
    15228381979765852785, 11949618678312581999,  4545324791131029438,   582725409406225185, 15282520250746126790,
    14758446535973412711,  7605613563088071833,  1111140641057375915,  5364843095234852245,   218335432181198977,
     4891472444796201742,  4564628942836375772, 15500501278323817088,  4913946328556108657,  2684786251736694229,
    12090498456116310122,  5310885782157038567,  5032788439854011923, 12627401038822728242, 11869662610126430929,
    17650156853043540226, 12126672500118808436, 10437658933435653256, 13133995470637873311,  4601324715591152820,
     1874350460376708372,  5808688626286061164, 13777088437302430376,  5018451954762213522,  2588296738534474754,
     5503414509154170711,  5230497186769951796, 13261090710400573914,  8515217303152165705, 11074538219737365303,
    15481562385740613213, 12705484409881007350, 14221931471178549498, 12905633420087112297, 17337759164357146506,
    14081997515778175224, 17384320185513122939,  7131793076779216692, 17483217190312403109,   900692047897995877,
    14723287313048560400,  6132094372965340305,  7572797575350925726, 12725160700431903514,   380860122911632449,
     1900504978569024571,  8423729759529914138,  7305587201606052334, 12446871355267313320,  4615812356515386206,
     3361817115406652303, 17690418922000878428, 14632214537567910559,  2709702289926174775,  3459675155951086144,
     7788364399926538150, 16043992474431955950, 15830963823784930267,  4216893617835797954,   538159724689093771,
    16029152738918251363, 14444848757576686696, 12941757045272633696, 10900480525147953314, 12547307449905859302,
    16001571796892398181,   407942194622690676, 13873235372903944444, 18071603799493008777,  1015646077646778622,
     9387605808959554815, 11566702442022019410,  7061722181092883183,  2629032108249254109,  5271820053177594520,
    12640880742139693547, 10098688629735675775,  5716304472850923064,  3312674502353063071,  7295926377425759633,
      833281439103466115, 16316743519466861667,  9912050326606348167, 11651133878100804242, 18026798122431692459,
     6157758321723692663,  4856021830695749349,  7074321707293278978, 10748097797809573561,  2949954440753264783,
     9813922580940661152,  9949237950172138336, 15643982711269455885, 16078663425810239127, 12508044395364228880,
    12920301578340189344, 15368071871011048915,  1610400750626363239, 11994736084146033126,  6042574085746186088,
     4154587549267685807, 15915752367312946034,  1191196620621769193,   467437822242538360,  2836463788873877488,
    10476401302029164984,  1716169985450737419,  5327734953288310341,  3994170067185955262,   884431883768190063,
    11019001754831208284, 14322807384384895215,   161011537360955545,  1466223959660131656,  5227048585229497539,
    12410731857504225031,  2142243279080761103, 17682826799106851430,  1792612570704179953, 14727410295243056025,
     1459567192481221274,  5669760721687603135, 17507918443756456845, 10354471145847018200, 10362475129248202288,
    13143844410150939443,  6861184673150072028, 18396524361124732580,   543906666394301875, 12476817828199026728,
    11853496871128122868, 12747674713108891748,  7986179867749890282,  9158195177777627533,  2217320706811118570,
     8631389005200569973,  5538133061362648855,  3369942850878700758,  7813559982698427184,   509051590411815948,
    10197035660403006684, 13004818533162292132,  9831652587047067687,  7619315254749630976,   994412663058993407
  ) :
  (  # 32-bit randoms
    1067595299,  955945823,  477289528, 4107218783, 4228976476,
    3344332714, 3355579695,  227628506,  810200273, 2591290167,
    2560260675, 3242736208,  646746669, 1479517882, 4245472273,
    1143372638, 3863670494, 3221021970, 1773610557, 1138697238,
    1421897700, 1269916527, 2859934041, 1764463362, 3874892047,
    3965319921,   72549643, 2383988930, 2600218693, 3237492380,
    2792901476,  725331109,  605841842,  271258942,  715137098,
    3297999536, 1322965544, 4229579109, 1395091102, 3735697720,
    2101727825, 3730287744, 2950434330, 1661921839, 2895579582,
    2370511479, 1004092106, 2247096681, 2111242379, 3237345263,
    4082424759,  219785033, 2454039889, 3709582971,  835606218,
    2411949883, 2735205030,  756421180, 2175209704, 1873865952,
    2762534237, 4161807854, 3351099340,  181129879, 3269891896,
     776029799, 2218161979, 3001745796, 1866825872, 2133627728,
      34862734, 1191934573, 3102311354, 2916517763, 1012402762,
    2184831317, 4257399449, 2899497138, 3818095062, 3030756734,
    1282161629,  420003642, 2326421477, 2741455717, 1278020671,
    3744179621,  271777016, 2626330018, 2560563991, 3055977700,
    4233527566, 1228397661, 3595579322, 1077915006, 2395931898,
    1851927286, 3013683506, 1999971931, 3006888962, 1049781534,
    1488758959, 3491776230,  104418065, 2448267297, 3075614115,
    3872332600,  891912190, 3936547759, 2269180963, 2633455084,
    1047636807, 2604612377, 2709305729, 1952216715,  207593580,
    2849898034,  670771757, 2210471108,  467711165,  263046873,
    3569667915, 1042291111, 3863517079, 1464270005, 2758321352,
    3790799816, 2301278724, 3106281430,    7974801, 2792461636,
     555991332,  621766759, 1322453093,  853629228,  686962251,
    1455120532,  957753161, 1802033300, 1021534190, 3486047311,
    1902128914, 3701138056, 4176424663, 1795608698,  560858864,
    3737752754, 3141170998, 1553553385, 3367807274,  711546358,
    2475125503,  262969859,  251416325, 2980076994, 1806565895,
     969527843, 3529327173, 2736343040, 2987196734, 1649016367,
    2206175811, 3048174801, 3662503553, 3138851612, 2660143804,
    1663017612, 1816683231,  411916003, 3887461314, 2347044079,
    1015311755, 1203592432, 2170947766, 2569420716,  813872093,
    1105387678, 1431142475,  220570551, 4243632715, 4179591855,
    2607469131, 3090613241,  282341803, 1734241730, 1391822177,
    1001254810,  827927915, 1886687171, 3935097347, 2631788714,
    3905163266,  110554195, 2447955646, 3717202975, 3304793075,
    3739614479, 3059127468,  953919171, 2590123714, 1132511021,
    3795593679, 2788030429,  982155079, 3472349556,  859942552,
    2681007391, 2299624053,  647443547,  233600422,  608168955,
    3689327453, 1849778220, 1608438222, 3968158357, 2692977776,
    2851872572,  246750393, 3582818628, 3329652309, 4036366910,
    1012970930,  950780808, 3959768744, 2538550045,  191422718,
    2658142375, 3276369011, 2927737484, 1234200027, 1920815603,
    3536074689, 1535612501, 2184142071, 3276955054,  428488088,
    2378411984, 4059769550, 3913744741, 2732139246,   64369859,
    3755670074,  842839565, 2819894466, 2414718973, 1010060670,
    1839715346, 2410311136,  152774329, 3485009480, 4102101512,
    2852724304,  879944024, 1785007662, 2748284463, 1354768064,
    3267784736, 2269127717, 3001240761, 3179796763,  895723219,
     865924942, 4291570937,   89355264, 1471026971, 4114180745,
    3201939751, 2867476999, 2460866060, 3603874571, 2238880432,
    3308416168, 2072246611, 2755653839, 3773737248, 1709066580,
    4282731467, 2746170170, 2832568330,  433439009, 3175778732,
      26248366, 2551382801,  183214346, 3893339516, 1928168445,
    1337157619, 3429096554, 3275170900, 1782047316, 4264403756,
    1876594403, 4289659572, 3223834894, 1728705513, 4068244734,
    2867840287, 1147798696,  302879820, 1730407747, 1923824407,
    1180597908, 1569786639,  198796327,  560793173, 2107345620,
    2705990316, 3448772106, 3678374155,  758635715,  884524671,
     486356516, 1774865603, 3881226226, 2635213607, 1181121587,
    1508809820, 3178988241, 1594193633, 1235154121,  326117244,
    2304031425,  937054774, 2687415945, 3192389340, 2003740439,
    1823766188, 2759543402,   10067710, 1533252662, 4132494984,
      82378136,  420615890, 3467563163,  541562091, 3535949864,
    2277319197, 3330822853, 3215654174, 4113831979, 4204996991,
    2162248333, 3255093522, 2219088909, 2978279037,  255818579,
    2859348628, 3097280311, 2569721123, 1861951120, 2907080079,
    2719467166,  998319094, 2521935127, 2404125338,  259456032,
    2086860995, 1839848496, 1893547357, 2527997525, 1489393124,
    2860855349,   76448234, 2264934035,  744914583, 2586791259,
    1385380501,   66529922, 1819103258, 1899300332, 2098173828,
    1793831094,  276463159,  360132945, 4178212058,  595015228,
     177071838, 2800080290, 1573557746, 1548998935,  378454223,
    1460534296, 1116274283, 3112385063, 3709761796,  827999348,
    3580042847, 1913901014,  614021289, 4278528023, 1905177404,
      45407939, 3298183234, 1184848810, 3644926330, 3923635459,
    1627046213, 3677876759,  969772772, 1160524753, 1522441192,
     452369933, 1527502551,  832490847, 1003299676, 1071381111,
    2891255476,  973747308, 4086897108, 1847554542, 3895651598,
    2227820339, 1621250941, 2881344691, 3583565821, 3510404498,
     849362119,  862871471,  797858058, 2867774932, 2821282612,
    3272403146, 3997979905,  209178708, 1805135652,    6783381,
    2823361423,  792580494, 4263749770,  776439581, 3798193823,
    2853444094, 2729507474, 1071873341, 1329010206, 1289336450,
    3327680758, 2011491779,   80157208,  922428856, 1158943220,
    1667230961, 2461022820, 2608845159,  387516115, 3345351910,
    1495629111, 4098154157, 3156649613, 3525698599, 4134908037,
     446713264, 2137537399, 3617403512,  813966752, 1157943946,
    3734692965, 1680301658, 3180398473, 3509854711, 2228114612,
    1008102291,  486805123,  863791847, 3189125290, 1050308116,
    3777341526, 4291726501,  844061465, 1347461791, 2826481581,
     745465012, 2055805750, 4260209475, 2386693097, 2980646741,
     447229436, 2077782664, 1232942813, 4023002732, 1399011509,
    3140569849, 2579909222, 3794857471,  900758066, 2887199683,
    1720257997, 3367494931, 2668921229,  955539029, 3818726432,
    1105704962, 3889207255, 2277369307, 2746484505, 1761846513,
    2413916784, 2685127085, 4240257943, 1166726899, 4215215715,
    3082092067, 3960461946, 1663304043, 2087473241, 4162589986,
    2507310778, 1579665506,  767234210,  970676017,  492207530,
    1441679602, 1314785090, 3262202570, 3417091742, 1561989210,
    3011406780, 1146609202, 3262321040, 1374872171, 1634688712,
    1280458888, 2230023982,  419323804, 3262899800,   39783310,
    1641619040, 1700368658, 2207946628, 2571300939, 2424079766,
     780290914, 2715195096, 3390957695,  163151474, 2309534542,
    1860018424,  555755123,  280320104, 1604831083, 2713022383,
    1728987441, 3639955502,  623065489, 3828630947, 4275479050,
    3516347383, 2343951195, 2430677756,  635534992, 3868699749,
     808442435, 3070644069, 4282166003, 2093181383, 2023555632,
    1568662086, 3422372620, 4134522350, 3016979543, 3259320234,
    2888030729, 3185253876, 4258779643, 1267304371, 1022517473,
     815943045,  929020012, 2995251018, 3371283296, 3608029049,
    2018485115,  122123397, 2810669150, 1411365618, 1238391329,
    1186786476, 3155969091, 2242941310, 1765554882,  279121160,
    4279838515, 1641578514, 3796324015,   13351065,  103516986,
    1609694427,  551411743, 2493771609, 1316337047, 3932650856,
    4189700203,  463397996, 2937735066, 1855616529, 2626847990,
      55091862, 3823351211,  753448970, 4045045500, 1274127772,
    1124182256,   92039808, 2126345552,  425973257,  386287896,
    2589870191, 1987762798, 4084826973, 2172456685, 3366583455,
    3602966653, 2378803535, 2901764433, 3716929006, 3710159000,
    2653449155, 3469742630, 3096444476, 3932564653, 2595257433,
     318974657, 3146202484,  853571438,  144400272, 3768408841,
     782634401, 2161109003,  570039522, 1886241521,   14249488,
    2230804228, 1604941699, 3928713335, 3921942509, 2155806892,
     134366254,  430507376, 1924011722,  276713377,  196481886,
    3614810992, 1610021185, 1785757066,  851346168, 3761148643,
    2918835642, 3364422385, 3012284466, 3735958851, 2643153892,
    3778608231, 1164289832,  205853021, 2876112231, 3503398282,
    3078397001, 3472037921, 1748894853, 2740861475,  316056182,
    1660426908,  168885906,  956005527, 3984354789,  566521563,
    1001109523, 1216710575, 2952284757, 3834433081, 3842608301,
    2467352408, 3974441264, 3256601745, 1409353924, 1329904859,
    2307560293, 3125217879, 3622920184, 3832785684, 3882365951,
    2308537115, 2659155028, 1450441945, 3532257603, 3186324194,
    1225603425, 1124246549,  175808705, 3009142319, 2796710159,
    3651990107,  160762750, 1902254979, 1698648476, 1134980669,
     497144426, 3302689335, 4057485630, 3603530763, 4087252587,
     427812652,  286876201,  823134128, 1627554964, 3745564327,
    2589226092, 4202024494,   62878473, 3275585894, 3987124064,
    2791777159, 1916869511, 2585861905, 1375038919, 1403421920,
      60249114, 3811870450, 3021498009, 2612993202,  528933105,
    2757361321, 3341402964, 2621861700,  273128190, 4015252178,
    3094781002, 1621621288, 2337611177, 1796718448, 1258965619,
    4241913140, 2138560392, 3022190223, 4174180924,  450094611,
    3274724580,  617150026, 2704660665, 1469700689, 1341616587,
     356715071, 1188789960, 2278869135, 1766569160, 2795896635,
      57824704, 2893496380, 1235723989, 1630694347, 3927960522,
     428891364, 1814070806, 2287999787, 4125941184, 3968103889,
    3548724050, 1025597707, 1404281500, 2002212197,   92429143,
    2313943944, 2403086080, 3006180634, 3561981764, 1671860914,
    1768520622, 1803542985,  844848113, 3006139921, 1410888995,
    1157749833, 2125704913, 1789979528, 1799263423,  741157179,
    2405862309,  767040434, 2655241390, 3663420179, 2172009096,
    2511931187, 1680542666,  231857466, 1154981000,  157168255,
    1454112128, 3505872099, 1929775046, 2309422350, 2143329496,
    2960716902,  407610648, 2938108129, 2581749599,  538837155,
    2342628867,  430543915,  740188568, 1937713272, 3315215132,
    2085587024, 4030765687,  766054429, 3517641839,  689721775,
    1294158986, 1753287754, 4202601348, 1974852792,   33459103,
    3568087535, 3144677435, 1686130825, 4134943013, 3005738435,
    3599293386,  426570142,  754104406, 3660892564, 1964545167,
     829466833,  821587464, 1746693036, 1006492428, 1595312919,
    1256599985, 1024482560, 1897312280, 2902903201,  691790057,
    1037515867, 3176831208, 1968401055, 2173506824, 1089055278,
    1748401123, 2941380082,  968412354, 1818753861, 2973200866,
    3875951774, 1119354008, 3988604139, 1647155589, 2232450826,
    3486058011, 3655784043, 3759258462,  847163678, 1082052057,
     989516446, 2871541755, 3196311070, 3929963078,  658187585,
    3664944641, 2175149170, 2203709147, 2756014689, 2456473919,
    3890267390, 1293787864, 2830347984, 3059280931, 4158802520,
    1561677400, 2586570938,  783570352, 1355506163,   31495586,
    3789437343, 3340549429, 2092501630,  896419368,  671715824,
    3530450081, 3603554138, 1055991716, 3442308219, 1499434728,
    3130288473, 3639507000,   17769680, 2259741420,  487032199,
    4227143402, 3693771256, 1880482820, 3924810796,  381462353,
    4017855991, 2452034943, 2736680833, 2209866385, 2128986379,
     437874044,  595759426,  641721026, 1636065708, 3899136933,
     629879088, 3591174506,  351984326, 2638783544, 2348444281,
    2341604660, 2123933692,  143443325, 1525942256,  364660499,
     599149312,  939093251, 1523003209,  106601097,  376589484,
    1346282236, 1297387043,  764598052, 3741218111,  933457002,
    1886424424, 3219631016,  525405256, 3014235619,  323149677,
    2038881721, 4100129043, 2851715101, 2984028078, 1888574695,
    2014194741, 3515193880, 4180573530, 3461824363, 2641995497,
    3179230245, 2902294983, 2217320456, 4040852155, 1784656905,
    3311906931,   87498458, 2752971818, 2635474297, 2831215366,
    3682231106, 2920043893, 3772929704, 2816374944,  309949752,
    2383758854,  154870719,  385111597, 1191604312, 1840700563,
     872191186, 2925548701, 1310412747, 2102066999, 1504727249,
    3574298750, 1191230036, 3330575266, 3180292097, 3539347721,
     681369118, 3305125752, 3648233597,  950049240, 4173257693,
    1760124957,  512151405,  681175196,  580563018, 1169662867,
    4015033554, 2687781101,  699691603, 2673494188, 1137221356,
     123599888,  472658308, 1053598179, 1012713758, 3481064843,
    3759461013, 3981457956, 3830587662, 1877191791, 3650996736,
     988064871, 3515461600, 4089077232, 2225147448, 1249609188,
    2643151863, 3896204135, 2416995901, 1397735321, 3460025646);

# Known test values for rand()
my @base_doub = ($Config{'uvsize'} == 8) ?
    # 64-bit randoms
  qw/
    0.35252031 0.51052342 0.79771733 0.39300273 0.27216673
    0.72151068 0.43144703 0.38522290 0.20270676 0.58227313
    0.80812143 0.83767297 0.92401619 0.84065425 0.00852052
    0.13975395 0.35250930 0.71196972 0.14627395 0.17775331
    0.61046382 0.49623272 0.23292425 0.25038837 0.04380664
    0.43275994 0.74540936 0.33830700 0.68832616 0.68744230
    0.63626548 0.85932936 0.37089670 0.50756304 0.69925960
    0.83481025 0.09053196 0.09523253 0.17783108 0.78027239
    0.70071054 0.51879252 0.83027285 0.92895011 0.72144803
    0.18868644 0.83655674 0.20358945 0.99852143 0.88340103
    0.46729949 0.96993433 0.00162682 0.46829774 0.59080423
    0.54921999 0.42516462 0.54952196 0.99534722 0.04473888
    0.71139235 0.91881407 0.33781561 0.45746234 0.78292126
    0.69206723 0.66175448 0.07091147 0.18179208 0.38168454
    0.38819527 0.42452711 0.22732724 0.16191307 0.36842667
    0.13060083 0.68833248 0.60498705 0.19195304 0.26628584
    0.17030858 0.23892426 0.38430236 0.28034283 0.76069020
    0.21560653 0.78101667 0.90847812 0.06467974 0.18487868
    0.23570471 0.29475460 0.65563767 0.10066446 0.57272419
    0.88731391 0.60650995 0.96346079 0.32940100 0.29977746
    0.03798193 0.18026822 0.22402746 0.45480119 0.98114604
    0.25800668 0.94362433 0.17901062 0.36019313 0.45933644
    0.68309457 0.28175454 0.00774729 0.77054527 0.99723413
    0.59807532 0.10294164 0.32429228 0.54928986 0.18410980
    0.08441555 0.14230333 0.58892064 0.94030475 0.35378784
    0.77584320 0.71222448 0.83565208 0.47309248 0.23810761
    0.74408520 0.08891527 0.09729786 0.38377368 0.05092308
    0.69065638 0.10449489 0.45050670 0.92209534 0.80083714
    0.27902692 0.26897142 0.50650468 0.80111472 0.54590012
    0.96406097 0.63779553 0.81054357 0.75369248 0.47473037
    0.89100315 0.89395984 0.09985519 0.34087631 0.22293557
    0.24375510 0.31764191 0.04076993 0.06160830 0.41333434
    0.11883030 0.04548820 0.01008040 0.25336184 0.07325432
    0.49860151 0.07148695 0.89483338 0.87054457 0.15116809
    0.59650469 0.47487776 0.43490298 0.36684681 0.16470796
    0.76865078 0.42920071 0.20545481 0.87615922 0.80332404
    0.36462506 0.49571309 0.51904488 0.15534589 0.43719893
    0.16562157 0.37290862 0.91842631 0.21310523 0.87849154
    0.18532269 0.81713354 0.52182344 0.51845619 0.96261204
    0.18758718 0.68897600 0.61484764 0.46752993 0.05865458
    0.11614359 0.90386866 0.45781805 0.70649579 0.50917048
    0.21210656 0.97818608 0.00788342 0.61375222 0.67366318
    0.24197878 0.66177985 0.10463932 0.67390799 0.50025262
    0.88332650 0.77966851 0.13403622 0.54357114 0.97664854
    0.06540961 0.24013176 0.67234032 0.91347883 0.35486839
    0.87207865 0.43036581 0.23652488 0.81238450 0.72058432
    0.42239916 0.80265764 0.03552838 0.61939480 0.50972420
    0.21053832 0.59952743 0.36821802 0.45659617 0.12529468
    0.76941623 0.99878168 0.08602783 0.81825937 0.39350710
    0.86090923 0.36090230 0.75628888 0.45036982 0.44602266
    0.20595631 0.62241953 0.36777732 0.47523727 0.50248178
    0.73570362 0.48237781 0.45590948 0.73580783 0.96403851
    0.94586342 0.48819868 0.48102038 0.94618182 0.90279924
    0.78396650 0.85182389 0.92149394 0.32679198 0.83554856
    0.28320609 0.34598409 0.82090005 0.40177958 0.38888785
    0.77873931 0.23297931 0.75329335 0.30770340 0.71417540
    0.68939065 0.36577776 0.50784857 0.50928090 0.02552055
    0.85999075 0.26692089 0.01402799 0.67550392 0.48305605
    0.74608351 0.63408891 0.58904230 0.44337996 0.42174728
    0.74041679 0.72719148 0.19801992 0.66263633 0.10381594
    0.32818760 0.68369661 0.56076212 0.68681921 0.91616269
    0.39836106 0.39685027 0.97507945 0.91010563 0.27447360
    0.95538357 0.76758522 0.60091060 0.37734461 0.82948248
    0.06598078 0.50147615 0.08417763 0.18910044 0.51661735
    0.55011011 0.64888175 0.82986845 0.15126656 0.92649390
    0.25494941 0.73275293 0.94184393 0.84755226 0.45921936
    0.72934054 0.43722403 0.34305596 0.10827860 0.29026676
    0.01935431 0.46668573 0.83247509 0.26349603 0.01938542
    0.43222250 0.18109983 0.29337450 0.16721917 0.94751650
    0.67795254 0.56666228 0.20699452 0.23247262 0.19138610
    0.73495506 0.85893600 0.83411526 0.93689655 0.91804752
    0.99352333 0.03207550 0.28386071 0.48029543 0.18736013
    0.31736452 0.72542230 0.57530912 0.04229918 0.84798296
    0.21886935 0.98655615 0.52243102 0.22611020 0.42975741
    0.21726739 0.10912048 0.96684473 0.01092456 0.12461901
    0.57989070 0.39848707 0.06330277 0.62826828 0.01159081
    0.23157320 0.64690912 0.44876902 0.04463930 0.18933780
    0.21284518 0.61363480 0.67144845 0.38625586 0.75719122
    0.40361050 0.26708873 0.54534727 0.90174015 0.58654140
    0.44885346 0.35505544 0.65317830 0.26074572 0.39472912
    0.54366914 0.75020660 0.76113614 0.24595582 0.03941247
    0.60356153 0.23615721 0.01603475 0.72432457 0.39837424
    0.04195329 0.81561058 0.34208440 0.00513953 0.92826234
    0.11410393 0.86692030 0.25238726 0.98258626 0.53353856
    0.72269001 0.71850984 0.66829681 0.03540769 0.01676450
    0.23557835 0.78758497 0.85969589 0.14673207 0.28013860
    0.17796942 0.69924087 0.44663597 0.62112513 0.44079883
    0.48995231 0.18411497 0.18440877 0.74016388 0.28845694
    0.22969080 0.76851164 0.15551473 0.28980810 0.40906710
    0.47619039 0.72611392 0.55802939 0.69365597 0.85736313
    0.83343150 0.21324760 0.45327806 0.33053855 0.98198279
    0.53279389 0.76877035 0.20548656 0.37065042 0.59026910
    0.67418036 0.23585843 0.98156397 0.27849804 0.56198954
    0.68752287 0.30073445 0.69348664 0.72515585 0.40629047
    0.09320027 0.24334978 0.91407662 0.97226538 0.33904970
    0.01717092 0.60155725 0.03001652 0.50979706 0.80531036
    0.17450719 0.84984399 0.00498130 0.51636405 0.14080868
    0.62289701 0.07853030 0.70567541 0.79844050 0.63766566
    0.03559031 0.40994535 0.08423996 0.00389626 0.50608347
    0.19622681 0.90537903 0.75458034 0.75102094 0.81491673
    0.92925931 0.38074332 0.54817053 0.72593246 0.02146791
    0.57990460 0.87921074 0.59913886 0.66726893 0.24269154
    0.73344575 0.71826052 0.92313935 0.05212996 0.93771536
    0.69489385 0.57581887 0.48106155 0.06808800 0.33633940
    0.69142320 0.46566781 0.70654143 0.16541368 0.76257631
    0.82777900 0.62958327 0.34757935 0.10891487 0.79912728
    0.01156543 0.23111261 0.58535640 0.87461956 0.21723454
    0.80409615 0.33169686 0.72800785 0.31218099 0.13729737
    0.41637635 0.01234597 0.58313811 0.66746028 0.05105595
    0.14930937 0.56044864 0.76196851 0.98800104 0.37075949
    0.88740864 0.40697115 0.96598278 0.86013661 0.85386784
    0.23986516 0.39027464 0.59593927 0.00161530 0.31768197
    0.65702729 0.66461914 0.62937471 0.92120758 0.87578958
    0.37539860 0.59182615 0.12092214 0.55130437 0.86365117
    0.38725162 0.28757657 0.42803199 0.39014405 0.50253853
    0.85306128 0.92018995 0.71421618 0.54236780 0.96221157
    0.22956898 0.96519876 0.06694102 0.11915854 0.01354308
    0.24720070 0.71671739 0.00604305 0.65012352 0.71151390
    0.46616159 0.99228224 0.20684576 0.62941006 0.84535326
    0.30678993 0.55264568 0.50094784 0.39409122 0.15479416
    0.36536318 0.51925656 0.65567178 0.67255519 0.55089659
    0.42194295 0.27172413 0.79540954 0.71594806 0.88372598
    0.29179452 0.66411306 0.57064687 0.42494633 0.73389255
    0.12097313 0.53338622 0.38493233 0.79348021 0.01851341
    0.58594454 0.88396240 0.04410730 0.67419924 0.62770011
    0.64644200 0.40335135 0.17952644 0.55564678 0.56643922
    0.37715015 0.87092180 0.56726159 0.34011210 0.13661819
    0.11474177 0.93930097 0.48549077 0.28484289 0.13374371
    0.40966056 0.73662873 0.37355323 0.65216092 0.27372469
    0.56032082 0.14882684 0.95462890 0.17090266 0.92374766
    0.98368259 0.68448367 0.02872548 0.68598279 0.04601084
    0.17170501 0.08906644 0.23730372 0.02929037 0.38566261
    0.68957569 0.53021050 0.44200157 0.32085701 0.72520053
    0.17454174 0.19676599 0.88243877 0.87030228 0.15124486
    0.78670160 0.51731632 0.56674531 0.20910664 0.84962640
    0.05220467 0.91783159 0.19138968 0.68126378 0.79574471
    0.14910848 0.28030331 0.98067264 0.31263980 0.67448964
    0.69266650 0.40033551 0.22789781 0.78317066 0.55815261
    0.11247054 0.47337901 0.46310033 0.53192452 0.56164078
    0.41750378 0.43880622 0.69739327 0.11092778 0.18333765
    0.67222441 0.12789170 0.88316806 0.37891271 0.14935268
    0.64522185 0.93902079 0.62481092 0.21794927 0.71535266
    0.62169579 0.65147153 0.01411645 0.96413465 0.01021578
    0.50605180 0.51595053 0.03308040 0.01497870 0.07809658
    0.35743383 0.58079701 0.11785557 0.89568677 0.38793964
    0.37117709 0.13994133 0.11032813 0.99998594 0.06695042
    0.79774786 0.11093584 0.23879095 0.85918615 0.16109636
    0.63479696 0.75023359 0.29061187 0.53764772 0.30652318
    0.51387302 0.81620973 0.82433610 0.18302488 0.79048957
    0.07598187 0.27887732 0.37061042 0.36441016 0.93736882
    0.77480946 0.02269132 0.40309874 0.16427650 0.13969296
    0.57605029 0.00242426 0.56626691 0.84390990 0.87455806
    0.12321023 0.87561663 0.60431578 0.35880839 0.50426282
    0.50697689 0.06631164 0.14976092 0.89356018 0.91473662
    0.04235237 0.50073724 0.75969690 0.91743994 0.79352335
    0.58078351 0.91819984 0.53520520 0.18267367 0.05608828
    0.68315721 0.27264599 0.41245634 0.69706222 0.69666203
    0.08967342 0.64081905 0.22576796 0.69315628 0.53981640
    0.76059129 0.56712344 0.94318621 0.44081094 0.31699284
    0.29477911 0.80069824 0.28366921 0.96718081 0.85345644
    0.11681215 0.47600710 0.33448255 0.31217271 0.35469241
    0.59511650 0.49583692 0.48922303 0.20215259 0.60159380
    0.17882055 0.77601258 0.71020391 0.41833503 0.71522856
    0.87534517 0.43703394 0.43056077 0.64828071 0.43069441
    0.39356849 0.32063367 0.92788963 0.16878266 0.56762591
    0.56042446 0.84958464 0.79408949 0.08220340 0.13922856
    0.82529019 0.27134959 0.00278080 0.66192389 0.01782933
    0.95404763 0.50787645 0.85320521 0.83690362 0.83771227
    0.46268665 0.31716742 0.01716647 0.68264674 0.01789888
    0.30446846 0.14942271 0.26982182 0.74933947 0.50394161
    0.78444542 0.40009256 0.40333422 0.16627342 0.01898760
    0.04221829 0.77960213 0.66230976 0.56015996 0.49535426
    0.38536259 0.40406773 0.99930568 0.00857945 0.16158390
    0.64805163 0.20237524 0.59106326 0.76968277 0.96887042
    0.29264851 0.97373775 0.16767633 0.33014482 0.27426548
    0.10947014 0.75920652 0.37757457 0.13125207 0.00826451
    0.96684342 0.69362226 0.22763554 0.20717541 0.42112268
    0.22803038 0.33481806 0.14968742 0.71598558 0.55126711
    0.64518015 0.65170197 0.89103003 0.72728361 0.24485454
    0.09410780 0.79818029 0.54212409 0.17790462 0.64442619
    0.62193511 0.51193256 0.02848781 0.05719604 0.45795152
    0.03219332 0.28310254 0.85746127 0.64890240 0.20658356
    0.50946422 0.80432490 0.08354468 0.09222723 0.67455943
    0.44638771 0.76366629 0.99677267 0.89311242 0.11627279
    0.09181302 0.44767077 0.16448724 0.26005539 0.28670391
    0.52465703 0.43598116 0.41869096 0.98043420 0.01497272
    0.51791571 0.61825308 0.85503436 0.63025655 0.02719292
    0.09865668 0.30321729 0.56998039 0.14946350 0.64823918
    0.19931639 0.14623555 0.54169913 0.68944135 0.73551005
    0.46743658 0.04109096 0.26625801 0.09537298 0.98207890
    0.58109721 0.70793680 0.84379365 0.42774726 0.12653597
    0.08566633 0.53366781 0.33960092 0.11036831 0.84464510
    0.16493476 0.92493443 0.87640673 0.52727644 0.57181349
    0.65071340 0.00978637 0.31700693 0.69148222 0.85063311
    0.06781819 0.30794534 0.65541667 0.16400484 0.06886223
    0.96227205 0.09633060 0.34513153 0.31013900 0.78165882
    0.39583699 0.86327936 0.69269199 0.11016575 0.67358419
    0.81775427 0.50052824 0.30068582 0.16606837 0.62243724
    0.47863741 0.68796498 0.31526949 0.41180883 0.23022147
    0.82342139 0.83003381 0.53571829 0.41081533 0.48600142/
  :
    # 32-bit randoms
  qw/
    0.76275443 0.99000644 0.98670464 0.10143112 0.27933125
    0.69867227 0.94218740 0.03427201 0.78842173 0.28180608
    0.92179002 0.20785655 0.54534773 0.69644020 0.38107718
    0.23978165 0.65286910 0.07514568 0.22765211 0.94872929
    0.74557914 0.62664415 0.54708246 0.90959343 0.42043116
    0.86334511 0.19189126 0.14718544 0.70259889 0.63426346
    0.77408121 0.04531601 0.04605807 0.88595519 0.69398270
    0.05377184 0.61711170 0.05565708 0.10133577 0.41500776
    0.91810699 0.22320679 0.23353705 0.92871862 0.98897234
    0.19786706 0.80558809 0.06961067 0.55840445 0.90479405
    0.63288060 0.95009721 0.54948447 0.20645042 0.45000959
    0.87050869 0.70806991 0.19406895 0.79286390 0.49332866
    0.78483914 0.75145146 0.12341941 0.42030252 0.16728160
    0.59906494 0.37575460 0.97815160 0.39815952 0.43595080
    0.04952478 0.33917805 0.76509902 0.61034321 0.90654701
    0.92915732 0.85365931 0.18812377 0.65913428 0.28814566
    0.59476081 0.27835931 0.60722542 0.68310435 0.69387186
    0.03699800 0.65897714 0.17527003 0.02889304 0.86777366
    0.12352068 0.91439461 0.32022990 0.44445731 0.34903686
    0.74639273 0.65918367 0.92492794 0.31872642 0.77749724
    0.85413832 0.76385624 0.32744211 0.91326300 0.27458185
    0.22190155 0.19865383 0.31227402 0.85321225 0.84243342
    0.78544200 0.71854080 0.92503892 0.82703064 0.88306297
    0.47284073 0.70059042 0.48003761 0.38671694 0.60465770
    0.41747204 0.47163243 0.72750808 0.65830223 0.10955369
    0.64215401 0.23456345 0.95944940 0.72822249 0.40888451
    0.69980355 0.26677428 0.57333635 0.39791582 0.85377858
    0.76962816 0.72004885 0.90903087 0.51376506 0.37732665
    0.12691640 0.71249738 0.81217908 0.37037313 0.32772374
    0.14238259 0.05614811 0.74363008 0.39773267 0.94859135
    0.31452454 0.11730313 0.62962618 0.33334237 0.45547255
    0.10089665 0.56550662 0.60539371 0.16027624 0.13245301
    0.60959939 0.04671662 0.99356286 0.57660859 0.40269560
    0.45274629 0.06699735 0.85064246 0.87742744 0.54508392
    0.87242982 0.29321385 0.67660627 0.68230715 0.79052073
    0.48592054 0.25186266 0.93769755 0.28565487 0.47219067
    0.99054882 0.13155240 0.47110470 0.98556600 0.84397623
    0.12875246 0.90953202 0.49129015 0.23792727 0.79481194
    0.44337770 0.96564297 0.67749118 0.55684872 0.27286897
    0.79538393 0.61965356 0.22487929 0.02226018 0.49248200
    0.42247006 0.91797788 0.99250134 0.23449967 0.52531508
    0.10246337 0.78685622 0.34310922 0.89892996 0.40454552
    0.68608407 0.30752487 0.83601319 0.54956031 0.63777550
    0.82199797 0.24890696 0.48801123 0.48661910 0.51223987
    0.32969635 0.31075073 0.21393155 0.73453207 0.15565705
    0.58584522 0.28976728 0.97621478 0.61498701 0.23891470
    0.28518540 0.46809591 0.18371914 0.37597910 0.13492176
    0.66849449 0.82811466 0.56240330 0.37548956 0.27562998
    0.27521910 0.74096121 0.77176757 0.13748143 0.99747138
    0.92504502 0.09175241 0.21389176 0.21766512 0.31183245
    0.23271221 0.21207367 0.57903312 0.77523344 0.13242613
    0.31037988 0.01204835 0.71652949 0.84487594 0.14982178
    0.57423142 0.45677888 0.48420169 0.53465428 0.52667473
    0.46880526 0.49849733 0.05670710 0.79022476 0.03872047
    0.21697212 0.20443086 0.28949326 0.81678186 0.87629474
    0.92297064 0.27373097 0.84625273 0.51505586 0.00582792
    0.33295971 0.91848412 0.92537226 0.91760033 0.07541125
    0.71745848 0.61158698 0.00941650 0.03135554 0.71527471
    0.24821915 0.63636652 0.86159918 0.26450229 0.60160194
    0.35557725 0.24477500 0.07186456 0.51757096 0.62120362
    0.97981062 0.69954667 0.21065616 0.13382753 0.27693186
    0.59644095 0.71500764 0.04110751 0.95730081 0.91600724
    0.47704678 0.26183479 0.34706971 0.07545431 0.29398385
    0.93236070 0.60486023 0.48015011 0.08870451 0.45548581
    0.91872718 0.38142712 0.10668643 0.01397541 0.04520355
    0.93822273 0.18011940 0.57577277 0.91427606 0.30911399
    0.95853475 0.23611214 0.69619891 0.69601980 0.76765372
    0.58515930 0.49479057 0.11288752 0.97187699 0.32095365
    0.57563608 0.40760618 0.78703383 0.43261152 0.90877651
    0.84686346 0.10599030 0.72872803 0.19315490 0.66152912
    0.10210518 0.06257876 0.47950688 0.47062066 0.72701157
    0.48915116 0.66110261 0.60170685 0.24516994 0.12726050
    0.03451185 0.90864994 0.83494878 0.94800035 0.91035206
    0.14480751 0.88458997 0.53498312 0.15963215 0.55378627
    0.35171349 0.28719791 0.09097957 0.00667896 0.32309622
    0.87561479 0.42534520 0.91748977 0.73908457 0.41793223
    0.99279792 0.87908370 0.28458072 0.59132853 0.98672190
    0.28547393 0.09452165 0.89910674 0.53681109 0.37931425
    0.62683489 0.56609740 0.24801549 0.52948179 0.98328855
    0.66403523 0.55523786 0.75886666 0.84784685 0.86829981
    0.71448906 0.84670080 0.43922919 0.20771016 0.64157936
    0.25664246 0.73055695 0.86395782 0.65852932 0.99061803
    0.40280575 0.39146298 0.07291005 0.97200603 0.20555729
    0.59616495 0.08138254 0.45796388 0.33681125 0.33989127
    0.18717090 0.53545811 0.60550838 0.86520709 0.34290701
    0.72743276 0.73023855 0.34195926 0.65019733 0.02765254
    0.72575740 0.32709576 0.03420866 0.26061893 0.56997511
    0.28439072 0.84422744 0.77637570 0.55982168 0.06720327
    0.58449067 0.71657369 0.15819609 0.58042821 0.07947911
    0.40193792 0.11376012 0.88762938 0.67532159 0.71223735
    0.27829114 0.04806073 0.21144026 0.58830274 0.04140071
    0.43215628 0.12952729 0.94668759 0.87391019 0.98382450
    0.27750768 0.90849647 0.90962737 0.59269720 0.96102026
    0.49544979 0.32007095 0.62585546 0.03119821 0.85953001
    0.22017528 0.05834068 0.80731217 0.53799961 0.74166948
    0.77426600 0.43938444 0.54862081 0.58575513 0.15886492
    0.73214332 0.11649057 0.77463977 0.85788827 0.17061997
    0.66838056 0.96076133 0.07949296 0.68521946 0.89986254
    0.05667410 0.12741385 0.83470977 0.63969104 0.46612929
    0.10200126 0.01194925 0.10476340 0.90285217 0.31221221
    0.32980614 0.46041971 0.52024973 0.05425470 0.28330912
    0.60426543 0.00598243 0.97244013 0.21135841 0.78561597
    0.78428734 0.63422849 0.32909934 0.44771136 0.27380750
    0.14966697 0.18156268 0.65686758 0.28726350 0.97074787
    0.63676171 0.96649494 0.24526295 0.08297372 0.54257548
    0.03166785 0.33735355 0.15946671 0.02102971 0.46228045
    0.11892296 0.33408336 0.29875681 0.29847692 0.73767569
    0.02080745 0.62980060 0.08082293 0.22993106 0.25031439
    0.87787525 0.45150053 0.13673441 0.63407612 0.97907688
    0.52241942 0.50580158 0.06273902 0.05270283 0.77031811
    0.05113352 0.24393329 0.75036441 0.37436336 0.22877652
    0.59975358 0.85707591 0.88691457 0.85547165 0.36641027
    0.58720133 0.45462835 0.09243817 0.32981586 0.07820411
    0.25421519 0.36004706 0.60092307 0.46192412 0.36758683
    0.98424170 0.08019934 0.68594024 0.45826386 0.29962317
    0.79365413 0.89231296 0.49478547 0.87645944 0.23590734
    0.28106737 0.75026285 0.08136314 0.79582424 0.76010628
    0.82792971 0.27947652 0.72482861 0.82191216 0.46171689
    0.79189752 0.96043686 0.51609668 0.88995725 0.28998963
    0.55191845 0.03934737 0.83033700 0.49553013 0.98009549
    0.19017594 0.98347750 0.33452066 0.87144372 0.72106301
    0.71272114 0.71465963 0.88361677 0.85571283 0.73782329
    0.20920458 0.34855153 0.46766817 0.02780062 0.74898344
    0.03680650 0.44866557 0.77426312 0.91025891 0.25195236
    0.87319953 0.63265037 0.25552148 0.27422476 0.95217406
    0.39281839 0.66441573 0.09158900 0.94515992 0.07800798
    0.02507888 0.39901462 0.17382573 0.12141278 0.85502334
    0.19902911 0.02160210 0.44460522 0.14688742 0.68020336
    0.71323733 0.60922473 0.95400380 0.99611159 0.90897777
    0.41073520 0.66206647 0.32064685 0.62805003 0.50677209
    0.52690101 0.87473387 0.73918362 0.39826974 0.43683919
    0.80459118 0.32422684 0.01958019 0.95319576 0.98326137
    0.83931735 0.69060863 0.33671416 0.68062550 0.65152380
    0.33392969 0.03451730 0.95227244 0.68200635 0.85074171
    0.64721009 0.51234433 0.73402047 0.00969637 0.93835057
    0.80803854 0.31485260 0.20089527 0.01323282 0.59933780
    0.31584602 0.20209563 0.33754800 0.68604181 0.24443049
    0.19952227 0.78162632 0.10336988 0.11360736 0.23536740
    0.23262256 0.67803776 0.48749791 0.74658435 0.92156640
    0.56706407 0.36683221 0.99157136 0.23421374 0.45183767
    0.91609720 0.85573315 0.37706276 0.77042618 0.30891908
    0.40709595 0.06944866 0.61342849 0.88817388 0.58734506
    0.98711323 0.14744128 0.63242656 0.87704136 0.68347125
    0.84446569 0.43265239 0.25146321 0.04130111 0.34259839
    0.92697368 0.40878778 0.56990338 0.76204273 0.19820348
    0.66314909 0.02482844 0.06669207 0.50205581 0.26084093
    0.65139159 0.41650223 0.09733904 0.56344203 0.62651696
    0.67332139 0.58037374 0.47258086 0.21010758 0.05713135
    0.89390629 0.10781246 0.32037450 0.07628388 0.34227964
    0.42190597 0.58201860 0.77363549 0.49595133 0.86031236
    0.83906769 0.81098161 0.26694195 0.14215941 0.88210306
    0.53634237 0.12090720 0.82480459 0.75930318 0.31847147
    0.92768077 0.01037616 0.56201727 0.88107122 0.35925856
    0.85860762 0.61109408 0.70408301 0.58434977 0.92192494
    0.62667915 0.75988365 0.06858761 0.36156496 0.58057195
    0.13636150 0.57719713 0.59340255 0.63530602 0.22976282
    0.71915530 0.41162531 0.63979565 0.09931342 0.79344045
    0.10893790 0.84450224 0.23122236 0.99485593 0.73637397
    0.17276368 0.13357764 0.74965804 0.64991737 0.61990341
    0.41523170 0.05878239 0.05687301 0.05497131 0.42868366
    0.42571090 0.25810502 0.89642955 0.30439758 0.39310223
    0.11357431 0.04288255 0.23397550 0.11200634 0.85621396
    0.89733974 0.37508865 0.42077265 0.68597384 0.72781399
    0.19296476 0.61699087 0.31667128 0.67756410 0.00177323
    0.05725176 0.79474693 0.18885238 0.06724856 0.68193156
    0.42202167 0.22082041 0.28554673 0.64995708 0.87851940
    0.29124547 0.61009521 0.87374537 0.05743712 0.69902994
    0.81925115 0.45653873 0.37236821 0.31118709 0.52734307
    0.39672836 0.38185294 0.30163915 0.17374510 0.04913278
    0.90404879 0.25742801 0.58266467 0.97663209 0.79823377
    0.36437958 0.15206043 0.26529938 0.22690047 0.05839021
    0.84721160 0.18622435 0.37809403 0.55706977 0.49828704
    0.47659049 0.24289680 0.88477595 0.07807463 0.56245739
    0.73490635 0.21099431 0.13164942 0.75840044 0.66877037
    0.28988183 0.44046090 0.24967434 0.80048356 0.26029740
    0.30416821 0.64151867 0.52067892 0.12880774 0.85465381
    0.02690525 0.19149288 0.49630295 0.79682619 0.43566145
    0.00288078 0.81484193 0.03763639 0.68529083 0.01339574
    0.38405386 0.30537067 0.22994703 0.44000045 0.27217985
    0.53831243 0.02870435 0.86282045 0.61831306 0.09164956
    0.25609707 0.07445781 0.72185784 0.90058883 0.30070608
    0.94476583 0.56822213 0.21933909 0.96772793 0.80063440
    0.26307906 0.31183306 0.16501252 0.55436179 0.68562285
    0.23829083 0.86511559 0.57868991 0.81888344 0.20126869
    0.93172350 0.66028129 0.21786948 0.78515828 0.10262106
    0.35390326 0.79303876 0.63427924 0.90479631 0.31024934
    0.60635447 0.56198079 0.63573813 0.91854197 0.99701497
    0.83085849 0.31692291 0.01925964 0.97446405 0.98751283
    0.60944293 0.13751018 0.69519957 0.68956636 0.56969015
    0.46440193 0.88341765 0.36754434 0.89223647 0.39786427
    0.85055280 0.12749961 0.79452122 0.89449784 0.14567830
    0.45716830 0.74822309 0.28200437 0.42546044 0.17464886
    0.68308746 0.65496587 0.52935411 0.12736159 0.61523955
    0.81590528 0.63107864 0.39786553 0.20102294 0.53292914
    0.75485590 0.59847044 0.32861691 0.12125866 0.58917183
    0.07638293 0.86845380 0.29192617 0.03989733 0.52180460
    0.32503407 0.64071852 0.69516575 0.74254998 0.54587026
    0.48713246 0.32920155 0.08719954 0.63497059 0.54328459
    0.64178757 0.45583809 0.70694291 0.85212760 0.86074305
    0.33163422 0.85739792 0.59908488 0.74566046 0.72157152/;


### - Standalone PRNG - ###

# Set predetermined seed for verification test
my @base_seed = ($Config{'uvsize'} == 8)
                    ? (0x12345, 0x23456, 0x34567, 0x45678)
                    : (0x123, 0x234, 0x345, 0x456);
sub myseed
{
    my $seed = $_[0];
    my $need = $_[1];  # Ignored

    push(@$seed, @base_seed);
}
eval { srand(\&myseed); };
if (! ok(! $@, 'srand(\&sub) works')) {
    diag('srand(\&sub) died: ' . $@);
}


# Check for warnings
@WARN = grep { $_ !~ /Partial seed/ } @WARN;
if (! ok(! @WARN, 'Acquired seed data')) {
    diag('Seed warnings: ' . join(' | ', @WARN));
}
undef(@WARN);

# Fetch and verify seed
my @seed;
eval { @seed = get_seed(); };
if (! ok(! $@, 'get_seed() works')) {
    diag('get_seed() died: ' . $@);
}
is_deeply(\@base_seed, \@seed, 'Seed is correct');

# Create PRNG object using state of standalone PRNG
my $prng2;
eval {
    my @state = get_state();
    $prng2 = Math::Random::MT::Auto->new('STATE' => \@state);
};
if (! ok(! $@, 'SA cloning works')) {
    diag('SA cloning died: ' . $@);
}
isa_ok($prng2, 'Math::Random::MT::Auto');
can_ok($prng2, qw(rand irand gaussian exponential erlang poisson binomial
                  shuffle srand get_seed set_seed get_state set_state));

# Verify hidden 'init' subroutine
if ($] > 5.006) {
    eval { $prng2->_init({}); };
    if (my $e = OIO->caught()) {
        ok($e->error =~ /hidden/i, '->_init() hidden: ' . $e->error);
    } else {
        ok($@, '->_init() visible - this is bad');
    }
} else {
    ok(1, 'SKIPPED');
}

# Test for known output from PRNG for irand()
my @test_rint;
for (1 .. 500) {
    push(@test_rint, irand());
}

# Save the current PRNG state
my @state;
eval { @state = get_state(); };
ok(! $@, 'get_state() died: ' . $@);

# Reset the seed
eval { set_seed(\@seed); };
ok(! $@, 'set_seed() died: ' . $@);

# Test again with reset seed
my @test2_rint;
for (1 .. 1000) {
    push(@test2_rint, irand());
}
is_deeply(\@test2_rint, \@base_rint, '1000 ints');

# Restore previous state
eval { set_state(\@state); };
ok(! $@, 'set_state() died: ' . $@);

# Continue from previous state
for (501 .. 1000) {
    push(@test_rint, Math::Random::MT::Auto::irand());
}
is_deeply(\@test_rint, \@base_rint, '500 ints');

# Test for known output from PRNG
#  for rand() and Math::Random::MT::Auto::rand()
my @test_doub;
for (1 .. 500) {
    push(@test_doub, sprintf('%0.8f', rand()));
}
for (501 .. 1000) {
    push(@test_doub, sprintf('%0.8f', Math::Random::MT::Auto::rand()));
}
is_deeply(\@test_doub, \@base_doub, '1000 doubles');

### - OO INterface - ###

# Create binary seed file
if (open(FH, '>seed_data.tmp')) {
    if ($Config{'uvsize'} == 8) {
        print(FH pack('Q4', 0x12345, 0x23456, 0x34567, 0x45678));
    } else {
        print(FH pack('L4', 0x123, 0x234, 0x345, 0x456));
    }
    close(FH);
} else {
    diag('Failure writing seed to file');
}

# Create PRNG object
my $prng;
eval { $prng = Math::Random::MT::Auto->new('src' => 'seed_data.tmp'); };
unlink('seed_data.tmp');
if (! ok(! $@, '->new works')) {
    diag('->new died: ' . $@);
}
isa_ok($prng, 'Math::Random::MT::Auto');

# Check for warnings
@WARN = grep { $_ !~ /exhausted/  &&  $_ !~ /Partial seed/ } @WARN;
if (! ok(! @WARN, 'Acquired seed data')) {
    diag('Seed warnings: ' . join(' | ', @WARN));
}
undef(@WARN);

# Fetch and verify seed
eval { @seed = @{$prng->get_seed()}; };
if (! ok(! $@, '$prng->get_seed() works')) {
    diag('$prng->get_seed() died: ' . $@);
}
is_deeply(\@base_seed, \@seed, 'Seed is correct');


# Test for known output from PRNG for $prng->irand()
@test_rint = ();
for (1 .. 500) {
    push(@test_rint, $prng->irand());
}


# Save the current PRNG state
eval { @state = $prng->get_state(); };
ok(! $@, '$prng->get_state() died: ' . $@);

# Reset the seed
eval { $prng->set_seed(\@seed); };
ok(! $@, '$prng->set_seed() died: ' . $@);

# Test again with reset seed
@test2_rint = ();
for (1 .. 1000) {
    push(@test2_rint, $prng->irand());
}
is_deeply(\@test2_rint, \@base_rint, '1000 ints');

# Restore previous state
eval { $prng->set_state(\@state); };
ok(! $@, '$prng->set_state() died: ' . $@);


# Continue from previous state
for (501 .. 1000) {
    push(@test_rint, $prng->irand());
}
is_deeply(\@test_rint, \@base_rint, '1000 ints');

# Test for known output from PRNG for $prng->rand()
@test_doub = ();
for (1 .. 1000) {
    push(@test_doub, sprintf('%0.8f', $prng->rand()));
}
is_deeply(\@test_doub, \@base_doub, '1000 doubles');


### - Cloning Tests - ###

# Create PRNG object using state of another PRNG object
my $prng3;
eval { $prng3 = $prng2->clone(); };
unlink('seed_data.tmp');
if (! ok(! $@, '->clone() works')) {
    diag('->clone() died: ' . $@);
}
isa_ok($prng3, 'Math::Random::MT::Auto');

# Test OO copy of SA PRNG
@test_rint = ();
for (1 .. 1000) {
    push(@test_rint, $prng2->irand());
}
is_deeply(\@test_rint, \@base_rint, '1000 ints');

# Test OO copy of OO PRNG
@test_rint = ();
for (1 .. 1000) {
    push(@test_rint, $prng3->irand());
}
is_deeply(\@test_rint, \@base_rint, '1000 ints');


### Test 'readonly'

if ($] >= 5.008003) {
    eval{
        my $old = $$prng;
        $$prng = 99;
        $$prng = $old;   # Just in case
    };
    ok($@, "Readonly: $@");
} else {
    ok(1, 'SKIPPED');
}

# EOF
