# Before `make install' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl t/ExifTool.t'

######################### We start with some black magic to print on failure.

# Change "1..N" below to so that N matches last test number

BEGIN { $| = 1; print "1..18\n"; }
END {print "not ok 1\n" unless $loaded;}

# test 1: Load ExifTool
use Image::ExifTool 'ImageInfo';
$loaded = 1;
print "ok 1\n";

######################### End of black magic.

use t::TestLib;

my $testname = 'ExifTool';
my $testnum = 1;

# test 2: JPG file using name
{
    ++$testnum;
    my $info = ImageInfo('t/ExifTool.jpg');
    print 'not ' unless check($info, $testname, $testnum);
    print "ok $testnum\n";
}

# test 3: GIF file using data in memory
{
    ++$testnum;
    open(TESTFILE, 't/ExifTool.gif');
    binmode(TESTFILE);
    my $gifImage;
    read(TESTFILE, $gifImage, 100000);
    close(TESTFILE);
    my $info = ImageInfo(\$gifImage);
    print 'not ' unless check($info, $testname, $testnum);
    print "ok $testnum\n";
}

# test 4: TIFF file using file reference and ExifTool object with options
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
    $exifTool->Options(Duplicates => 1, Unknown => 1);
    open(TESTFILE, 't/ExifTool.tif');
    my $info = $exifTool->ImageInfo(\*TESTFILE);
    close(TESTFILE);
    print 'not ' unless check($exifTool, $info, $testname, $testnum);
    print "ok $testnum\n";
}

# test 5: test the Group option to extract EXIF info only
{
    ++$testnum;
    my $info = ImageInfo('t/ExifTool.jpg', {Group0 => 'EXIF'});
    print 'not ' unless check($info, $testname, $testnum);
    print "ok $testnum\n";
}

# test 6: extract specified tags only
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
# don't test DateFormat because strftime output is system dependent
#    $exifTool->Options(DateFormat => '%H:%M:%S %a. %b. %e, %Y');
    my @tags = ('CreateDate', 'DateTimeOriginal', 'ModifyDate');
    my $info = $exifTool->ImageInfo('t/ExifTool.jpg', \@tags);
    print 'not ' unless check($exifTool, $info, $testname, $testnum);
    print "ok $testnum\n";
}

# test 7: test the 4 different ways to exclude tags...
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
    $exifTool->Options(Exclude => 'ImageWidth');
    my @tagList = ( '-ImageHeight', '-Make' );
    my $info = $exifTool->ImageInfo('t/ExifTool.jpg', '-FileSize',
                        \@tagList, {Group0 => '-MakerNotes'});
    print 'not ' unless check($exifTool, $info, $testname, $testnum);
    print "ok $testnum\n";
}

# tests 8/9: test ExtractInfo(), GetInfo(), CombineInfo()
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
    $exifTool->Options(Duplicates => 0);  # don't allow duplicates
    $exifTool->ExtractInfo('t/ExifTool.jpg');
    my $info1 = $exifTool->GetInfo({Group0 => 'MakerNotes'});
    my $info2 = $exifTool->GetInfo({Group0 => 'EXIF'});
    my $info = $exifTool->CombineInfo($info1, $info2);
    print 'not ' unless check($exifTool, $info, $testname, $testnum);
    print "ok $testnum\n";

    # combine information in different order
    ++$testnum;
    $info = $exifTool->CombineInfo($info2, $info1);
    print 'not ' unless check($exifTool, $info, $testname, $testnum);
    print "ok $testnum\n";
}

# test 10: test group options across different families
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
    my $info = $exifTool->ImageInfo('t/ExifTool.jpg',
                    { Group1 => 'Canon', Group2 => '-Camera' });
    print 'not ' unless check($exifTool, $info, $testname, $testnum);
    print "ok $testnum\n";
}

# test 11/12: test ExtractInfo() and GetInfo()
# (uses output from test 6 for comparison)
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
# don't test DateFormat because strftime output is system dependent
#    $exifTool->Options(DateFormat => '%H:%M:%S %a. %b. %e, %Y');
    $exifTool->ExtractInfo('t/ExifTool.jpg');
    my @tags = ('createdate', 'datetimeoriginal', 'modifydate');
    my $info = $exifTool->GetInfo(\@tags);
    my $good = 1;
    my @expectedTags = ('CreateDate', 'DateTimeOriginal', 'ModifyDate');
    for (my $i=0; $i<scalar(@tags); ++$i) {
        $tags[$i] = $expectedTags[$i] or $good = 0;
    }
    print 'not ' unless $good;
    print "ok $testnum\n";

    ++$testnum;
    print 'not ' unless check($exifTool, $info, $testname, $testnum, 6);
    print "ok $testnum\n";
}

# tests 13/14: check precidence of tags extracted from groups
# (Note: these tests should produce the same output as 8/9,
#  so the .out files from tests 8/9 are used)
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
    $exifTool->Options(Duplicates => 0);  # don't allow duplicates
    my $info = $exifTool->ImageInfo('t/ExifTool.jpg',{Group0=>['MakerNotes','EXIF']});
    print 'not ' unless check($exifTool, $info, $testname, $testnum, 8);
    print "ok $testnum\n";

    # combine information in different order
    ++$testnum;
    $info = $exifTool->ImageInfo('t/ExifTool.jpg',{Group0=>['EXIF','MakerNotes']});
    print 'not ' unless check($exifTool, $info, $testname, $testnum, 9);
    print "ok $testnum\n";
}

# test 15/16/17: test GetGroups()
{
    ++$testnum;
    my $exifTool = new Image::ExifTool;
    $exifTool->ExtractInfo('t/ExifTool.jpg');
    my @groups = $exifTool->GetGroups(2);
    my $not;
    foreach ('Camera','ExifTool','Image','Time') {
        $_ eq shift @groups or $not = 1;
    }
    @groups and $not = 1;
    print 'not ' if $not;
    print "ok $testnum\n";
    
    ++$testnum;
    my $info = $exifTool->GetInfo({Group0 => 'EXIF'});
    @groups = $exifTool->GetGroups($info,0);
    print 'not ' unless @groups==1 and $groups[0] eq 'EXIF';
    print "ok $testnum\n";

    ++$testnum;
    my $testfile = "t/ExifTool_$testnum";
    open(TESTFILE,">$testfile.failed");
    my $oldSep = $/;   
    $/ = "\x0a";        # set input line separator
    $exifTool->ExtractInfo('t/ExifTool.jpg');
    my $family = 1;
    @groups = $exifTool->GetGroups($family);
    my $group;
    foreach $group (@groups) {
        next if $group eq 'ExifTool';
        print TESTFILE "---- $group ----\n";
        my $info = $exifTool->GetInfo({"Group$family" => $group});
        foreach (sort $exifTool->GetTagList($info)) {
            print TESTFILE "$_ : $$info{$_}\n";
        } 
    }
    $/ = $oldSep;       # restore input line separator
    close(TESTFILE);
    print 'not ' unless testCompare("$testfile.out","$testfile.failed",$testnum);
    print "ok $testnum\n";
}

# test 18: Test verbose output
{
    ++$testnum;
    my $testfile = "t/ExifTool_$testnum";
    my $ok = 1;
    my $skip = '';
    # capture verbose output by redirecting STDOUT
    if (open(TESTFILE,">&STDOUT") and open(STDOUT,">$testfile.tmp")) {
        ImageInfo('t/ExifTool.jpg', { Verbose => 3 });
        close(STDOUT);
        open(STDOUT,">&TESTFILE"); # restore original STDOUT
        # re-write output file to change newlines to be same as standard test file
        # (if I was a Perl guru, maybe I would know a better way to do this)
        open(TMPFILE,"$testfile.tmp");
        open(TESTFILE,">$testfile.failed");
        my $oldSep = $\;
        $\ = "\x0a";        # set output line separator
        while (<TMPFILE>) {
            chomp;          # remove existing newline
            print TESTFILE $_;  # re-write line using \x0a for newlines
        }
        $\ = $oldSep;       # restore output line separator
        close(TESTFILE);
        unlink("$testfile.tmp");
        $ok = testCompare("$testfile.out","$testfile.failed",$testnum);
    } else {
        # skip this test
        $skip = ' # Skip Can not redirect standard output to test verbose option';
    }
    print 'not ' unless $ok;
    print "ok $testnum$skip\n";
}

# end
