package IPC::ClearTool;

use strict;
use vars qw($VERSION @ISA @EXPORT_OK %EXPORT_TAGS);

BEGIN {
    if ($^O =~ /win32/i) {
	require Win32::OLE;
    }
}

use IPC::ChildSafe 2.34;
@EXPORT_OK = @IPC::ChildSafe::EXPORT_OK;
%EXPORT_TAGS = ( BehaviorMod => \@EXPORT_OK );
@ISA = q(IPC::ChildSafe);

# The current version and a way to access it.
$VERSION = "2.00"; sub version {$VERSION}

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;

    if ($^O =~ /win32/i) {
	return $class->SUPER::new(@_);
    }

    my $ct = join('/', $ENV{ATRIAHOME} || '/usr/atria', 'bin/cleartool');
    $ct = 'cleartool' unless -x $ct;
    my $chk = sub {
	my($r_stderr, $r_stdout) = @_;
	return int grep /Error:\s/, @$r_stderr;
    };
    my %params = ( QUIT => 'exit', CHK => $chk );
    my $self = $class->SUPER::new($ct, 'pwd -h', 'Usage: pwd', \%params);
    bless ($self, $class);
    return $self;
}

sub comment {
    my $self = shift;
    my $cmnt = shift;
    $self->stdin("$cmnt\n.");
    return $self;
}

sub chdir {
    my $self = shift;
    my $nwd = shift;
    if ($^O =~ /win32/i) {
	return $self->cmd(qq(cd $nwd));
    } else {
	if (!CORE::chdir($nwd)) {
	    warn "$nwd: $!\n";
	    return 0;
	}
	return $self->cmd(qq(cd "$nwd"));
    }
}
*cd = *chdir;

sub _open {
    my $self = shift;
    if ($^O =~ /win32/i) {
	$self->{IPC_CHILD} = Win32::OLE->new('ClearCase.ClearTool')
			|| die "Cannot create ClearCase.ClearTool object\n";
	Win32::OLE->Option(Warn => 0);
	return $self;
    }
    return $self->SUPER::_open(@_);
}

sub _puts {
    my $self = shift;
    if ($^O =~ /win32/i) {
	my $cmd = shift;
	my $dbg = $self->{DBGLEVEL} || 0;
	warn "+ -->> $cmd\n" if $dbg;
	my $out = $self->{IPC_CHILD}->CmdExec($cmd);
	my $error = int Win32::OLE->LastError;
	$self->{IPC_STATUS} = $error;
	# CmdExec always returns a scalar through Win32::OLE so
	# we have to split it in case it's really a list.
	my @stdout = $self->_fixup_COM_scalars($out) if $out;
	print map {"+ <<-- $_"} @stdout if @stdout && $dbg > 1;
	push(@{$self->{IPC_STDOUT}}, @stdout);
	push(@{$self->{IPC_STDERR}},
		$self->_fixup_COM_scalars(Win32::OLE->LastError)) if $error;
	return $self;
    }
    return $self->SUPER::_puts(@_);
}

sub finish {
    my $self = shift;
    if ($^O =~ /win32/i) {
	undef $self->{IPC_CHILD};
	return 0;
    }
    return $self->SUPER::finish(@_);
}

1;

__END__

=head1 NAME

IPC::ClearTool, ClearTool - run a bidirectional pipe to a cleartool process

=head1 SYNOPSIS

  use IPC::ClearTool;

  my $CT = IPC::ClearTool->new;
  $CT->cmd("pwv");
  $CT->cmd("lsview");
  $CT->cmd("lsvob -s");
  for ($CT->stdout) { print }
  $CT->finish;

=head1 ALTERNATE SYNOPSES

  use IPC::ClearTool;

  $rc = $CT->cmd("pwv");		# Assign return code to $rc

  $CT->notify;				# "notify mode" is default;
  $rc = $CT->cmd("pwv");		# same as above

  $CT->store;				# "Store mode" - hold stderr for
  $rc = $CT->cmd("pwv -X");		# later retrieval via $CT->stderr
  @errs = $CT->stderr;			# Retrieve it now

  $CT->ignore;				# Discard all stdout/stderr and
  $CT->cmd("pwv");			# ignore nonzero return codes

  $CT->cmd("ls foo@@");			# In void context, store stdout,
					# print stderr immediately,
					# exit on error.

  my %results = $CT->cmd("pwv");	# Place all results in %results,
					# available as:
					#   @{$results{stdout}}
					#   @{$results{stderr}}
					#   @{$results{status}}

  $CT->cmd();				# Clear all accumulators

  $CT->stdout;				# In void context, print stored output

=head1 DESCRIPTION

This module invokes the ClearCase 'cleartool' command as a child
process and opens pipes to its standard input, output, and standard
error. Cleartool commands may be sent "down the pipe" via the
$CT->cmd() method.  All stdout resulting from commands is stored in the
object and can be retrieved at any time via the $CT->stdout method. By
default, stderr from commands is sent directly to the real (parent's)
stderr but if the I<store> attribute is set as shown above, stderr will
accumulate just like stdout and must be retrieved via $CT->stderr.

If $CT->cmd is called in a void context it will exit on error unless
the I<ignore> attribute is set, in which case all output is thrown away
and error messages suppressed.  If called in a scalar context it
returns the exit status of the command.

When used with no arguments and in a void context, $CT->cmd simply
clears the stdout and stderr accumulators.

The $CT->stdout and $CT-stderr methods behave just like arrays; when
used in a scalar context they return the number of lines currently
stored.  When used in an array context they return, well, an array
containing all currently stored lines, and then clear the internal
stack.

The $CT->finish method ends the child process and returns its exit
status.

This is only a summary of the documentation. There are more advanced
methods for error detection, data return, etc. documented as part of
IPC::ChildSafe. Note that IPC::ClearTool is simply a small subclass of
ChildSafe; it provides the right defaults to ChildSafe's constructor
for running cleartool and adds a few ClearCase-specific methods. In all
other ways it's identical to ChildSafe, and all ChildSafe documentation
applies.

=head1 BUGS

=over 4

=item * Comments

Comments present a special problem. If a comment is prompted for, it
will likely hang the child process by interrupting the tag/eot
sequencing. So we prefer to pass comments on the command line with C<-c>.
Unfortunately, the quoting rules of cleartool are insufficient to allow
passing comments with embedded newlines using C<-c>. The result being
that there's no clean way to handle multi-line comments.

To work around this, a method C<$CT->comment> is provided which
registers a comment I<to be passed to the next C<$CT->cmd()> command>.
It's inserted into the stdin stream with a "\n.\n" appended.
The subsequent command must have a C<-cq> flag, e.g.:

    $CT->comment("Here's a\nmultiple line\ncomment");
    $CT->cmd("ci -cq foo.c");

If your script hangs and the comment for the last element checked in is
C<"pwd -h">, then you were burned by such a sync problem.

=item * UNIX/Win32 Semantics Skew

On UNIX, this module works by running cleartool as a child process.  On
Windows, the ClearCase Automation Library (a COM API) is used instead.
This provides the same interface B<but be warned that there's a subtle
semantic difference!> On UNIX you can send a setview command to the
child and it will run in the new view while the parent's environment
remains unchanged. On Windows there's no subprocess; thus the setview
would change the context of the "parent process".  The same applies to
chdir commands. It's unclear which behavior is "better" overall, but in
any case portable applications must take extra care in using such
stateful techniques.

As a partial remedy, a C<chdir> method is provided. This simply does
the C<cd> in both parent and child processes in an attempt to emulate
the in-process behavior. Emulating an in-process C<setview> is harder
because on UNIX, setview is implemented with a fork/chroot/exec
sequence so (a) it's hard to know how a single-process setview
I<should> behave and (b) I wouldn't know how to do it anyway,
especially lacking the privileges required by chroot(2). Of course
in most cases you could work around this by using C<chdir> to work
in view-extended space rather than a set view.

Of course, in some cases the ability to set the child process into a
different view or directory is a feature, and no attempt is made to
stop you from doing that.

=item * Win32::OLE Behavior with IClearTool

Due to the way Win32::OLE works, on Windows the results of each
command are passed back as a single string, possibly with embedded
newlines. For consistency, in a list context we split this back into
lines and return the list. However, it's possible for this to result in
a different line count for the same command in UNIX and NT, if one of
the lines would have had embedded newlines in it anyway.

=item * Win32/CC 3.2.1 COM Bug

The ClearCase/COM API wasn't documented until CC 4.0, but it's actually
present (at least the IClearTool interface, which is what
IPC::ClearTool uses) in 3.2.1 with one major bug - all output arrives
backwards! So we provide a class method

	IPC::ClearTool->cc_321_hack;

which, if called, will determine whether it's running on CC 3.2.1 and
if so reverse the order of output. This is called once at the start of the
program to set the state; it's a no-op if CC 4.0 or above is in use.

=back

=head1 AUTHOR

David Boyce dsb@world.std.com

=head1 SEE ALSO

perl(1), "perldoc IPC::ChildSafe"

=cut
