package Net::SSLLabs::Cert;

use strict;
use warnings;
use URI();

sub new {
    my ( $class, $json ) = @_;
    my $self = $json;
    bless $self, $class;
    return $self;
}

sub altNames {
    my ($self) = @_;
    return @{ $self->{altNames} };
}

sub issuerSubject {
    my ($self) = @_;
    return $self->{issuerSubject};
}

sub issues {
    my ($self) = @_;
    return $self->{issues};
}

sub ocspRevocationStatus {
    my ($self) = @_;
    return $self->{ocspRevocationStatus};
}

sub ocspURIs {
    my ($self) = @_;
    return map { URI->new($_) } @{ $self->{ocspURIs} };
}

sub revocationInfo {
    my ($self) = @_;
    return $self->{revocationInfo};
}

sub sgc {
    my ($self) = @_;
    return $self->{sgc} ? 1 : 0;
}

sub validationType {
    my ($self) = @_;
    return $self->{validationType};
}

sub sct {
    my ($self) = @_;
    return $self->{sct} ? 1 : 0;
}

sub sigAlg {
    my ($self) = @_;
    return $self->{sigAlg};
}

sub commonNames {
    my ($self) = @_;
    return map { URI->new($_) } @{ $self->{commonNames} };
}

sub crlURIs {
    my ($self) = @_;
    return map { URI->new($_) } @{ $self->{crlURIs} };
}

sub issuerLabel {
    my ($self) = @_;
    return $self->{issuerLabel};
}

sub subject {
    my ($self) = @_;
    return $self->{subject};
}

sub notBefore {
    my ($self) = @_;
    return $self->{notBefore};
}

sub revocationStatus {
    my ($self) = @_;
    return $self->{revocationStatus};
}

sub notAfter {
    my ($self) = @_;
    return $self->{notAfter};
}

sub crlRevocationStatus {
    my ($self) = @_;
    return $self->{crlRevocationStatus};
}

1;
__END__

=head1 NAME

Net::SSLLabs::Cert - Cert object

=head1 METHODS

=head2 new

a new I<Net::SSLLabs::Cert> object, accepts a JSON object as it's parameter

=head2 subject

certificate subject

=head2 commonNames

list of common names extracted from the subject

=head2 altNames

alternative names

=head2 notBefore

UNIX timestamp before which the certificate is not valid

=head2 notAfter

UNIX timestamp after which the certificate is not valid

=head2 issuerSubject

issuer subject

=head2 sigAlg

certificate signature algorithm

=head2 issuerLabel

issuer name

=head2 revocationInfo

a number that represents revocation information present in the certificate:

=over 2

=item bit 0 (1) - CRL information available

=item bit 1 (2) - OCSP information available

=back

=head2 crlURIs

list of CRL I<URI>s extracted from the certificate.

=head2 ocspURIs

list of OCSP I<URI>s extracted from the certificate

=head2 revocationStatus

a number that describes the revocation status of the certificate:

=over 2

=item 0 - not checked

=item 1 - certificate revoked

=item 2 - certificate not revoked

=item 3 - revocation check error

=item 4 - no revocation information

=item 5 - internal error

=back

=head2 crlRevocationStatus

same as revocationStatus, but only for the CRL information (if any).

=head2 ocspRevocationStatus

same as revocationStatus, but only for the OCSP information (if any).

=head2 sgc

Server Gated Cryptography support; integer:

=over 2

=item bit 1 (1) - Netscape SGC

=item bit 2 (2) - Microsoft SGC

=back

=head2 validationType

E for Extended Validation certificates; may be null if unable to determine

=head2 issues

list of certificate issues, one bit per issue:

=over 2

=item bit 0 (1) - no chain of trust

=item bit 1 (2) - not before

=item bit 2 (4) - not after

=item bit 3 (8) - hostname mismatch

=item bit 4 (16) - revoked

=item bit 5 (32) - bad common name

=item bit 6 (64) - self-signed

=item bit 7 (128) - blacklisted

=item bit 8 (256) - insecure signature

=back

=head2 sct

true if the certificate contains an embedded SCT; false otherwise.

