package HTTP::PublicKeyPins;

use 5.006;
use strict;
use warnings;
use Crypt::OpenSSL::X509();
use Crypt::OpenSSL::RSA();
use Digest();
use MIME::Base64();
use English qw( -no_match_vars );
use FileHandle();
use Exporter();
use Carp();
*import = \&Exporter::import;
our @EXPORT_OK = qw(
  pin_sha256
);
our %EXPORT_TAGS = ( 'all' => \@EXPORT_OK, );

sub _CERTIFICATE_HEADER_SIZE { return 32; }

our $VERSION = '0.05';

sub pin_sha256 {
    my ($path) = @_;
    my $handle = FileHandle->new($path)
      or Carp::croak("Failed to open $path for reading:$EXTENDED_OS_ERROR");
    read $handle, my $file_header, _CERTIFICATE_HEADER_SIZE()
      or Carp::croak("Failed to read from $path:$EXTENDED_OS_ERROR");
    if ( $file_header !~ /^[-]{5}BEGIN[ ]CERTIFICATE[-]{5}\r?\nMII/smx ) {
        Carp::croak("$path is not a PEM encoded SSL Certificate");
    }
    my $x509 = Crypt::OpenSSL::X509->new_from_file($path);
    my $string;
    if ( $x509->key_alg_name() eq 'rsaEncryption' ) {
        my $pubkey = Crypt::OpenSSL::RSA->new_public_key( $x509->pubkey() );
        $string = $pubkey->get_public_key_x509_string();
    }
    else {
        $string = $x509->pubkey();
    }
    $string =~ s/^[-]+BEGIN[^\n]+\n//smx;
    $string =~ s/^[-]+END[^\n]+\n//smx;
    my $der    = MIME::Base64::decode($string);
    my $digest = Digest->new('SHA-256');
    $digest->add($der);
    my $base64 = MIME::Base64::encode_base64( $digest->digest() );
    chomp $base64;
    return $base64;
}

1;    # End of HTTP::PublicKeyPins
__END__

=head1 NAME

HTTP::PublicKeyPins - Generate RFC 7469 HTTP Public Key Pin (HPKP) header values

=head1 VERSION

Version 0.05

=head1 SYNOPSIS

Make it more difficult for the bad guys to Man-In-The-Middle your users TLS sessions

    use HTTP::Headers();
    use HTTP::PublicKeyPins qw( pin_sha256 );

    ...
    my $h = HTTP::Headers->new();
    $h->header( 'Public-Key-Pins-Report-Only',
            'pin-sha256="'
          . pin_sha256('/etc/pki/tls/certs/example.pem')
          . '"; pin-sha256="'
          . pin_sha256('/etc/pki/tls/certs/backup.pem')
          . '"; report-uri="https://example.com/pkp-report.pl' );


=head1 DESCRIPTION

This module allows the calculation of RFC 7469 HTTP Public Key Pin header values. This can be used to verify your TLS session to a remote server has not been hit by a Man-In-The-Middle attack OR to instruct your users to ignore any TLS sessions to your web service that does not use your Public Key

=head1 EXPORT

=head2 pin_sha256

This function accepts the path to a certificate.  It will load the public key from the certificate and prepare the appropriate value for the pin_sha256 parameter of the Public-Key-Pins value.

=head1 SUBROUTINES/METHODS

None.  This module only has the one exported function.

=head1 DIAGNOSTICS
 
=over
 
=item C<< Failed to open %s for reading >>
 
Failed to open the supplied SSL Certificate file
 
=item C<< Failed to read from %s >>
 
Failed to read from the supplied SSL Certificate file

=item C<< %s is not a PEM encoded SSL Certificate >>
 
The supplied input file does not look like a SSL Certificate File. An SSL Certificate file has the following header

  -----BEGIN CERTIFICATE-----
  MII

=back
 
=head1 CONFIGURATION AND ENVIRONMENT
 
HTTP::PublicKeyPins requires no configuration files or environment variables.
 
=head1 DEPENDENCIES
 
HTTP::PublicKeyPins requires the following non-core modules
 
  Crypt::OpenSSL::X509;
  Crypt::OpenSSL::RSA;
  Digest
 
=head1 INCOMPATIBILITIES
 
None known.

=head1 SEE ALSO

L<RFC 7469 - Public Key Pinning Extension for HTTP|http://tools.ietf.org/html/rfc7469>

=head1 AUTHOR

David Dick, C<< <ddick at cpan.org> >>

=head1 BUGS AND LIMITATIONS
 
Please report any bugs or feature requests to C<bug-http-publickeypins at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=HTTP-PublicKeyPins>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc HTTP::PublicKeyPins


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=HTTP-PublicKeyPins>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/HTTP-PublicKeyPins>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/HTTP-PublicKeyPins>

=item * Search CPAN

L<http://search.cpan.org/dist/HTTP-PublicKeyPins/>

=back


=head1 LICENSE AND COPYRIGHT

Copyright 2015 David Dick.

This module is free software; you can redistribute it and/or modify it under
the same terms as Perl itself. See L<perlartistic|perlartistic>.  This program is
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.
