package OpenInteract::ResultsIterator;

# $Id: ResultsIterator.pm,v 1.6 2001/10/11 03:17:27 lachoy Exp $

use strict;
use Data::Dumper     qw( Dumper );
use SPOPS::Iterator  qw( ITER_IS_DONE );

@OpenInteract::ResultsIterator::ISA     = qw( SPOPS::Iterator );
$OpenInteract::ResultsIterator::VERSION = sprintf("%d.%02d", q$Revision: 1.6 $ =~ /(\d+)\.(\d+)/);

sub initialize {
    my ( $self, $p ) = @_;
    $self->{_SEARCH_RESULTS}    = $p->{results}{result_list};
    $self->{_SEARCH_EXTRA_NAME} = $p->{results}{extra_name};
    $self->{_SEARCH_COUNT}      = 1;
    $self->{_SEARCH_RAW_COUNT}  = 0;
    $self->{_SEARCH_OFFSET}     = $p->{min} || $p->{results}{min};
    $self->{_SEARCH_MAX}        = $p->{max} || $p->{results}{max};
}


sub fetch_object {
    my ( $self ) = @_;

    # Get the info for retrieving the object

    my $current_count = $self->{_SEARCH_RAW_COUNT};
    my $object_class = $self->{_SEARCH_RESULTS}->[ $current_count ]->{class};
    my $object_id    = $self->{_SEARCH_RESULTS}->[ $current_count ]->{id};

    return ITER_IS_DONE unless ( $object_class and $object_id );

    my $obj = eval { $object_class->fetch( $object_id,
                                           { skip_security => $self->{_SKIP_SECURITY} } ) };
    if ( $@ ) {
        $self->{_SEARCH_RAW_COUNT}++;
        return $self->fetch_object;
    }

    return ITER_IS_DONE unless ( $obj );

    # Using min/max and haven't reached it yet

    if ( $self->{_SEARCH_OFFSET} and 
         ( $self->{_SEARCH_COUNT} < $self->{_SEARCH_OFFSET} ) ) {
        $self->{_SEARCH_COUNT}++;
        $self->{_SEARCH_RAW_COUNT}++;
        return $self->fetch_object;
    }

    if ( $self->{_SEARCH_MAX} and
         ( $self->{_SEARCH_COUNT} > $self->{_SEARCH_MAX} ) ) {
        return ITER_IS_DONE;
    }

    # Ok, we've gone through all the necessary contortions -- we can
    # actually return the object. Finish up.

    if ( $self->{_SEARCH_EXTRA_NAME} ) {
        my $extra_info = $self->{_SEARCH_RESULTS}->[ $current_count ]->{extra};
        my $extra_count = 0;
        foreach my $name ( @{ $self->{_SEARCH_EXTRA_NAME} } ) {
            $obj->{"tmp_$name"} = $extra_info->[ $extra_count ];
            $extra_count++;
        }
    }

    $self->{_SEARCH_RAW_COUNT}++;
    $self->{_SEARCH_COUNT}++;

    return ( $obj, $self->{_SEARCH_COUNT} );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::ResultsIterator - Iterator to scroll through search results that are objects of different classes.

=head1 SYNOPSIS

 my $results = OpenInteract::ResultsManage->new({ search_id => $search_id });
 my $iter = $results->retrieve({ return => 'iterator' });
 while ( my $obj = $iter->get_next ) {
     print "Object is a ", ref $obj, " with ID ", $obj->id, "\n";
 }

=head1 DESCRIPTION

This class implements L<SPOPS::Iterator> so we can scroll through
search results one at a time.

=head1 METHODS

B<initialize>

B<fetch_object>

=head1 BUGS

None yet!

=head1 TO DO

Nothing known.

=head1 SEE ALSO

L<SPOPS::Iterator>

L<OpenInteract::SearchManage>

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
