package OpenInteract::Handler::Error;

# $Id: Error.pm,v 1.7 2001/07/13 14:48:51 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::Error::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::Error::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Error::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Error::default_method    = 'listing';
@OpenInteract::Handler::Error::forbidden_methods = ();
%OpenInteract::Handler::Error::security          = ( 
 listing  => SEC_LEVEL_READ,
 show     => SEC_LEVEL_READ,
 remove   => SEC_LEVEL_WRITE,
 notify   => SEC_LEVEL_READ,
);

use constant MAIN_SCRIPT => '/Error';

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;

    # See if we were passed any sort directives

    my $order = $apr->param( 'sort' ) || 'error_time';
    my $db_order = ( $order eq 'error_time' ) ? 'error_time DESC' : $order;

    # See if we were passed any filtering directives -- if so process
    # them so we can determine which error objects to display

    my $where = undef;
    my @value = ();
    my %filter = ();

    # See if they asked for a particular code

    if ( my $filter_code = $apr->param( 'filter_code' ) ) {
        $filter{code} = $filter_code;
        $where = ' AND code = ?';
        push @value, $filter_code;
    }

    # See if they asked for a particular type

    if ( my $filter_type = $apr->param( 'filter_type' ) ) {
        $filter{type} = $filter_type;
        $where = ' AND type = ?';
        push @value, $filter_type;
    }

    # Figure out the date stuff

    if ( my $filter_from_year = $apr->param( 'filter_date_from_year' ) ) {
        my $from_date = $class->read_date( 'filter_date_from' );
        ( $filter{from_year}, $filter{from_month}, $filter{from_day} ) = split /\-/, $from_date;
        $where .= ' AND error_time > ?';
        push @value, $from_date;
    } 
    if ( my $filter_to_year = $apr->param( 'filter_date_to_year' ) ) {
        my $to_date = $class->read_date( 'filter_date_to' );
        ( $filter{to_year}, $filter{to_month}, $filter{to_day} ) = split /\-/, $to_date;
        $where .= ' AND error_time < ?';
        push @value, $to_date;
    }
    $where =~ s/^ AND //;

    my $params = { main_script => MAIN_SCRIPT, 
                   sort        => $order, 
                   filterby    => \%filter, 
                   error_msg   => $p->{error_msg} };

    # Now actually retrieve the errors 

    $params->{error_iterator} = eval { $R->error_object->fetch_iterator({ 
                                                            order => $db_order,
                                                            where => $where,
                                                            value => \@value }) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $OpenInteract::Error::user_msg = 'Cannot retrieve list of errors.';
        $R->throw( { code => 403 } );
        $params->{error_msg} .= "Error retrieving list of errors $OpenInteract::Error::system_msg";
    }

    # Set the template and title and we're done.

    $R->{page}->{title} = 'Listing of Errors';
    return $R->template->handler( {}, $params, 
                                  { db      => 'error_list',
                                    package => 'base_error'  } );
}


sub show {
    my ( $class, $p ) = @_;
    my $params = { main_script => MAIN_SCRIPT };
    my $R = OpenInteract::Request->instance;
    my $error = $p->{error};
    $error ||= eval { $R->error_object->fetch( $R->apache->param( 'error_id' ) ) };
    if ( $@ ) {
        my $ei = SPOPS::Error->get;
        if ( $ei->{type} ne 'security' ) { 
            $R->throw( { code => 404 } );
            $params->{error_msg} = "Could not retrieve Error object. Error logged.";
        }
        else {
            $params->{error_msg} = "You do not have permission to view this error object.";
        }
    }
    local $Text::Wrap::columns = 60;
    $error->{system_msg} = Text::Wrap::wrap( '', '', $error->{system_msg} );
    $error->{user_msg}   = Text::Wrap::wrap( '', '', $error->{user_msg} );
    $R->{page}->{title} = 'Detail of Error';
    $params->{err} = $error;
    return $R->template->handler( {}, $params, 
                                  { db      => 'error_detail', 
                                    package => 'base_error' } );
}



sub remove {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my @error_id = $R->apache->param( 'error_id' );
    my ( @error_items );
    foreach my $eid ( @error_id ) {
        $R->DEBUG && $R->scrib( 1, "Removing error: $eid" );
        my $error = eval { $R->error_object->fetch( $eid ); };
        if ( $@ or ! $error ) {
            push @error_items, "Could not retrieve error $eid to remove: $SPOPS::Error::system_msg" if ( $@ );
            push @error_items, "Error $eid apparently does not exist, so it was not removed."       if ( ! $@ );;
            next;
        }
        eval { $error->remove; };   
        push @error_items, SPOPS::Error->get  if ( $@ );
    } 
    $p->{error_msg} = join "\n\n", map { $_->{system_msg} } @error_items if ( scalar @error_items );
    return $class->listing( $p );
}


sub notify {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my @error_id = $R->apache->param( 'error_id' );
    my $email    = $R->apache->param( 'email' );
    if ( ! $email or ! scalar @error_id ) {
        return '<h2 align="center">Error</h2>' .
               '<p>Error: Cannot notify anyone about an object when no ID/email is given.</p>';
    }
    my @error_list = ();
    foreach my $eid ( @error_id ) {
        my $error = eval { $R->error_object->fetch( $eid ) };
        push @error_list, $error    if ( $error );
    }
    if ( $R->error_object->notify({ email   => $email, 
                                    subject => 'Error object notification', 
                                    object  => \@error_list, 
                                    type    => 'error' } ) ) {
        return '<h2 align="center">Success!</h2>' .
               '<p>Notification sent properly!</p>';
    }
    return '<h2 align="center">Error</h2>' .
           '<p>Error sending email. Please check error logs!</p>';
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Error - Browse the error logs

=head1 SYNOPSIS

=head1 DESCRIPTION

This module has four simple handlers to deal with error messages
currently logged:

=over 4

=item *

B<list>: Display all messages, or those that fit the filter

=item *

B<show>: Display detail regarding a particular message

=item *

B<remove>: Delete one or more messages from the system

=item *

B<notify>: Send via email one or more messages

=back

=head1 TO DO

B<Page?>

Should we page results? 

B<Change initial page?>

Should the initial page be just the filter component so people can
browse by date without seeing ALL the errors?

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
