#!/usr/bin/perl

# $Id: add_objects_to_index.pl,v 1.3 2001/06/09 18:09:50 lachoy Exp $

# add_objects_to_index.pl -- Re-index a particular class.

use strict;
use Data::Dumper  qw( Dumper );
use Getopt::Long  qw( GetOptions );
use OpenInteract::Startup;

{
 my $usage = "Usage: $0 --website_dir=/path/to/my_site object-tag";
 
 # Get the options

 my ( $OPT_website_dir, $OPT_where );
 GetOptions( 'website_dir=s' => \$OPT_website_dir,
             'where=s'       => \$OPT_where );

 if ( ! $OPT_website_dir and $ENV{OIWEBSITE} ) {
   print "[oi_manage]: Using ($ENV{OIWEBSITE}) for 'website_dir'.\n";
   $OPT_website_dir = $ENV{OIWEBSITE};
 }

 unless ( -d $OPT_website_dir ) {
   die "$usage\n Parameter '--website_dir' or \$ENV{OIWEBSITE} must refer\n",
       "to an OpenInteract website directory!\n";
 }

 my $object_tag = shift;
 die "$usage\n" unless ( $object_tag );

 # Setup the OI environment

 my $R = OpenInteract::Startup->setup_static_environment( $OPT_website_dir );

 # Try to get the class corresponding to the object tag passed in

 my $obj_class = eval { $R->$object_tag() };
 if ( $@ or ! $obj_class ) {
   my $error_msg = $@ || 'none returned';
   die "Cannot retrieve objects without a class -- no match for ",
       "$object_tag. (Error: $error_msg)\n";
 }

 # Ensure the object class is currently being indexed

 unless ( $obj_class->isa( 'OpenInteract::FullText' ) ) {
   die "Failed! The class ($obj_class) corresponding to tag \n",
       "($object_tag) does not currently use the full-text indexing\n",
       "engine. Change the 'isa' tag for the object.\n";
 }

 my $CONFIG = $obj_class->CONFIG;
 my $ft_fields = $CONFIG->{fulltext_field};
 unless ( ref $ft_fields eq 'ARRAY' and scalar @{ $ft_fields } ) {
   die "Failed! You must define a list of fields to index in the\n",
       "'fulltext_field' key in your object configuration.\n";
 }

 # Retrieve all the objects -- but define a column group on the fly
 # for just the fields we need (sneaky)

 $CONFIG->{column_group}->{fulltext} ||= $ft_fields;

 my $obj_list = eval { $obj_class->fetch_group({ skip_security => 1,
                                                 where         => $OPT_where,
                                                 column_group  => 'fulltext' }) };
 if ( $@ ) {
   my $ei = SPOPS::Error->get;
   die "Fetch of objects failed!\n",
       "Simple error: $@\n",
       "Enlightening error: $ei->{system_msg}\n";
 }

 my $start_time = scalar localtime;
 print "Objects found in group: ", scalar @{ $obj_list }, "\n";
 print "Starting to index each object. This might take a while...\n";

 # Index each object

 my ( $count, $ok );
 $count = $ok = 0;
 foreach my $obj ( @{ $obj_list } ) {   
   $obj->reindex_object;
   $count++;
   print "($count) ", $obj->id;
   if ( $@ ) { 
     print " FAIL ($@)\n";
   }
   else {
     print " OK\n";
     $ok++;
   }
 }
 print "Done.\nObjects attempted/indexed: $count/$ok\n";
 print "Start: $start_time\n",
       "End:   ", scalar localtime, "\n";

 # Some DBD drivers complain about not explicitly disconnecting, which
 # worries people...

 $R->db->disconnect;
}
 
__END__

=pod

=head1 NAME

add_objects_to_index.pl - Reindex objects in a particular class

=head1 SYNOPSIS

 # 'news' is the label you use in your 'spops.perl' file for the
 # object -- e.g., 'user' for 'OpenInteract::User' objects or
 # 'sitetemplate' for 'OpenInteract::SiteTemplate' objects.

 $ perl add_objects_to_index.pl --website_dir=/home/httpd/www.myoisite.com news

OR (using a bash shell):

 $ export OIWEBSITE=/home/httpd/www.myoisite.com
 $ perl add_objects_to_index.pl news

 # Find objects matching only a particular criteria

 $ perl add_objects_to_index.pl --where="title like '%all your base%'"

=head1 DESCRIPTION

Cycles through every available object in the class given (or each
object matching a particular criteria) and calls 'reindex_object' on
it. Pretty simple.

Note that the '--where' option requires you to do value quoting
manually. If your clause fails, you will definitely hear about it.

=head1 BUGS 

This can take an amazingly long time to run on some databases. MySQL,
in particular, seems to have some performance issues with relatively
large index tables.

=head1 TO DO

B<Web interface>

This same function should be implemented via a web interface.

=head1 SEE ALSO

L<OpenInteract::FullText>

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
