package String::BooleanSimple; ## Gets the boolean representative of a string


our $VERSION='0.01';


use strict;

use vars qw(@ISA @EXPORT %EXPORT_TAGS $VERSION);
use Exporter; 

@ISA = qw(Exporter);

%EXPORT_TAGS = ( all => [qw(
                      boolean
                      is_true
                      is_false
                      isTrue  
                      isFalse
                )] ); 

Exporter::export_ok_tags('all'); 


# Some static functions returning a boolean value (0 or 1) of a string like
# 'true', 'false', etc.
#
# SYNOPSIS
# ========
# 
#  # imports all functions
#  use String::BooleanSimple ':all';
#
#  # imports only is_true()
#  use String::BooleanSimple qw(is_true);
#  
#
# Matches
# =======
# Supports these strings:
#
#
#   true yes active enabled on y ok positive 1 2 3 4 5 6 7 8 9
#
#   false no inactive disabled off n not ok negative 0
#
# If the string does not match, it causes an error. Whitespace at the beginning or end will be automatically removed.
#
# LICENSE
# =======   
# You can redistribute it and/or modify it under the conditions of LGPL.
# 
# AUTHOR
# ======
# Andreas Hernitscheck  ahernit(AT)cpan.org


# Returns 1 if the string matches to a positive pattern like "true".
sub is_true { # $boolean ($string)
  my $value = shift;
  my $ret = undef;

  $ret = boolean($value);

  return $ret;
}



# Returns 1 if the string matches to a negative pattern like "false".
sub is_false { # $boolean ($string)
  my $value = shift;
  my $ret = undef;

  $ret = 1 - boolean($value);

  return $ret;
}


# Returns 1 if the string matches to a postive pattern like "true".
# Returns 0 if the string matches to a negative pattern like "false".
sub boolean { # $boolean ($string)
  my $value = shift;
  my $ret = 2;

  # trim
  $value = _trim( $value );

  # lower case
  $value = lc( $value );

  if ( $value =~ m/^(true|yes|active|enabled|on|y|ok|positive|[1-9])$/ ){ $ret = 1 };

  if ( $value =~ m/^(false|no|inactive|disabled|off|n|negative|not ok|[0])$/ ){ $ret = 0 };

  if ($ret == 2) {
    die "String value \'$value\' does not match to a given true/false pattern.";
  }

  return $ret;
}


# trimming the string from whitespace
sub _trim {
  my $value = shift;

  $value =~ s/^\s*//s;
  $value =~ s/\s*$//s;
  
  return $value;
}


# If you like the java style, you may import that alias
sub isTrue{ # $boolean ($string)
  return is_true(@_);
}

# If you like the java style, you may import that alias
sub isFalse{ # $boolean ($string)
  return is_false(@_);
}


1;
#################### pod generated by Pod::Autopod - keep this line to make pod updates possible ####################

=head1 NAME

String::BooleanSimple - Gets the boolean representative of a string


=head1 SYNOPSIS


 # imports all functions
 use String::BooleanSimple ':all';

 # imports only is_true()
 use String::BooleanSimple qw(is_true);
 



=head1 DESCRIPTION

Some static functions returning a boolean value (0 or 1) of a string like
'true', 'false', etc.



=head1 REQUIRES

L<Exporter> 


=head1 METHODS

=head2 boolean

 my $boolean = boolean($string);

Returns 1 if the string matches to a postive pattern like "true".
Returns 0 if the string matches to a negative pattern like "false".


=head2 isFalse

 my $boolean = isFalse($string);

If you like the java style, you may import that alias


=head2 isTrue

 my $boolean = isTrue($string);

If you like the java style, you may import that alias


=head2 is_false

 my $boolean = is_false($string);

Returns 1 if the string matches to a negative pattern like "false".


=head2 is_true

 my $boolean = is_true($string);

Returns 1 if the string matches to a positive pattern like "true".



=head1 Matches

Supports these strings:


  true yes active enabled on y ok positive 1 2 3 4 5 6 7 8 9

  false no inactive disabled off n not ok negative 0

If the string does not match, it causes an error. Whitespace at the beginning or end will be automatically removed.



=head1 AUTHOR

Andreas Hernitscheck  ahernit(AT)cpan.org


=head1 LICENSE

You can redistribute it and/or modify it under the conditions of LGPL.



=cut

