use strict;
use warnings;

package Printer::ESCPOS;

# PODNAME: Printer::ESCPOS
# ABSTRACT: Interface for all thermal, dot-matrix and other receipt printers that support ESC-POS specification.
#
# This file is part of Printer-ESCPOS
#
# This software is copyright (c) 2015 by Shantanu Bhadoria.
#
# This is free software; you can redistribute it and/or modify it under
# the same terms as the Perl 5 programming language system itself.
#
our $VERSION = '0.012'; # VERSION

# Dependencies
use 5.010;
use Moo;
use Carp;
use Type::Tiny;
use aliased 'Printer::ESCPOS::Roles::Profile' => 'ESCPOSProfile';
use namespace::autoclean;

use Printer::ESCPOS::Connections::File;
use Printer::ESCPOS::Connections::Network;
use Printer::ESCPOS::Connections::Serial;
use Printer::ESCPOS::Connections::USB;


has driverType => (
    is       => 'ro',
    required => 1,
);


has profile => (
    is      => 'ro',
    default => 'Generic',
);


has deviceFilePath => ( is => 'ro', );


has deviceIP => ( is => 'ro', );


has devicePort => (
    is      => 'ro',
    default => '9100',
);


has baudrate => (
    is      => 'ro',
    default => 38400,
);


has serialOverUSB => (
    is      => 'ro',
    default => '1',
);


has vendorId => ( is => 'ro', );


has productId => ( is => 'ro', );


has endPoint => (
    is      => 'ro',
    default => 0x01,
);


has timeout => (
    is       => 'ro',
    required => 1,
    default  => 1000,
);

has _driver => (
    is       => 'lazy',
    init_arg => undef,
);

sub _build__driver {
    my ($self) = @_;

    if ( $self->driverType eq 'File' ) {
        return Printer::ESCPOS::Connections::File->new(
            deviceFilePath => $self->deviceFilePath, );
    }
    elsif ( $self->driverType eq 'Network' ) {
        return Printer::ESCPOS::Connections::Network->new(
            deviceIP   => $self->deviceIP,
            devicePort => $self->devicePort,
        );
    }
    elsif ( $self->driverType eq 'Serial' ) {
        return Printer::ESCPOS::Connections::Serial->new(
            deviceFilePath => $self->deviceFilePath,
            baudrate       => $self->baudrate,
            serialOverUSB  => $self->serialOverUSB,
        );
    }
    elsif ( $self->driverType eq 'USB' ) {
        return Printer::ESCPOS::Connections::USB->new(
            productId => $self->productId,
            vendorId  => $self->vendorId,
            endPoint  => $self->endPoint,
            timeout   => $self->timeout,
        );
    }
}


has printer => ( is => 'lazy', );

sub _build_printer {
    my ($self) = @_;

    my $base  = __PACKAGE__ . "::Profiles::";
    my $class = $base . $self->profile;

    Class::Load::load_class($class);
    unless ( $class->does(ESCPOSProfile) ) {
        confess
"Class ${class} in ${base} does not implement the Printer::ESCPOS::Roles::Profile Interface";
    }
    my $object = $class->new( driver => $self->_driver, );

    $object->init();

    return $object;
}

no Moo;
__PACKAGE__->meta->make_immutable;

1;

__END__

=pod

=head1 NAME

Printer::ESCPOS - Interface for all thermal, dot-matrix and other receipt printers that support ESC-POS specification.

=head1 VERSION

version 0.012

=head1 SYNOPSIS

If you are just starting up with POS RECEIPT Printers, you must first refer to L<Printer::ESCPOS::Manual> to get started.

Printer::ESCPOS provides four different types of printer connections to talk to a ESCPOS printer. 
Among these connection types 'Serial', 'Network', 'File' are already implemented in this module. 'USB' type using L<Device::USB> module is under development. In the meantime most of the printing tasks for USB connected printer can be achieved using the 'File' connection mode for usb connected ESCPOS printers.

     use Printer::ESCPOS;
 
     use Data::Dumper; # Just to get dumps of status functions.
 
 
     #########################################################
     BEGIN: Network Printer
     #########################################################
     # For Network Printers $port is 9100 in most cases but might differ depending on how 
     # you have configured your printer
     my $printer_id = '192.168.0.10';
     my $port       = '9100';
     my $device = Printer::ESCPOS->new(
         driverType => 'Network',
         deviceIp   => $printer_ip,
         devicePort => $port,
     );
     $device->printer->init(); # This calls the initialization functions for your printer.
 
     # These commands won't actually send anything to the printer but will store all the 
     # merged data including control codes to module buffer.
     $device->printer->printAreaWidth( nL => 0, nH => 1);
     $device->printer->text("Print Area Width Modified\n");
     $device->printer->printAreaWidth(); # Reset to default
     $device->printer->text("print area width reset\n");
     $device->printer->tab();
     $device->printer->underline(1);
     $device->printer->text("underline on\n");
     $device->printer->invert(1);
     $device->printer->text("Inverted Text\n");
     $device->printer->justification('right');
     $device->printer->text("Right Justified\n");
     $device->printer->upsideDown(1);
     $device->printer->text("Upside Down\n");
     $device->printer->cutPaper();
 
     $device->printer->print(); # Dispatch the above commands from module buffer to the Printer. 
                                # This command takes care of read text buffers for the printer.
 
 
     #########################################################
     BEGIN: Serial Printer
     #########################################################
     # Use the Serial mode for local printer connected on serial port(or a printer connected via 
     # a physical USB port in USB to Serial mode), check syslog(Usually under /var/log/syslog) 
     # for what device file was created for your printer when you connect it to your system(For 
     # plug and play printers).
     my $path = '/dev/ttyACM0';
     $device = Printer::ESCPOS->new(
         driverType     => 'Serial',
         deviceFilePath => $path,
     );
     $device->printer->init(); # This calls the initialization functions for your printer.
 
     say Dumper $device->printer->printerStatus();
     say Dumper $device->printer->offlineStatus();
     say Dumper $device->printer->errorStatus();
     say Dumper $device->printer->paperSensorStatus();
 
     $device->printer->bold(1);
     $device->printer->text("Bold Text\n");
     $device->printer->bold(0);
     $device->printer->text("Bold Text Off\n");
 
     $device->printer->print();
 
 
     #########################################################
     BEGIN: File(Direct to Device File) Printer
     #########################################################
     # A 'File' driver is similar to the 'Serial' driver in all functionality except that it 
     # doesn't support the status functions for the printer. i.e. you will not be able to use 
     # printerStatus, offlineStatus, errorStatus or paperSensorStatus functions
     my $path = '/dev/usb/lp0';
     $device = Printer::ESCPOS->new(
         driverType     => 'File',
         deviceFilePath => $path,
     );

=head1 DESCRIPTION

You can use this module for all your ESC-POS Printing needs. If some of your printer's functions are not included, you may extend this module by adding specialized funtions for your printer in it's own subclass. Refer to L<Printer::ESCPOS::Roles::Profile> and L<Printer::ESCPOS::Profiles::Generic>

=head1 ATTRIBUTES

=head2 driverType

"Required attribute". The driver type to use for your printer. This can be B<File>, B<Network> or B<Serial>. B<USB> driver is not implemented yet.
If you choose B<File> or B<Serial> driver, you must provide the I<deviceFilePath>, for B<Network> driver you must provide the I<printerIp> and I<printerPort>.

Network driver type:

    my $printer_id = '192.168.0.10';
    my $port       = '9100';
    my $device = Printer::ESCPOS->new(
        driverType => 'Network',
        deviceIp   => $printer_ip,
        devicePort => $port,
    );

Serial driver type:

    my $path = '/dev/ttyACM0';
    $device = Printer::ESCPOS->new(
        driverType     => 'Serial',
        deviceFilePath => $path,
    );

File driver type:

    my $path = '/dev/usb/lp0';
    $device = Printer::ESCPOS->new(
        driverType     => 'File',
        deviceFilePath => $path,
    );

=head2 profile

There are minor differences in ESC POS printers across different brands and models in terms of specifications and extra features. For using special features of a particular brand you may create a sub class in the name space Printer::ESCPOS::Profiles::* and load your profile here. I would recommend extending  Generic ( L<Printer::ESCPOS::Profiles::Generic> ).
Use the following classes as examples.
L<Printer::ESCPOS::Profiles::Generic>
L<Printer::ESCPOS::Profiles::SinocanPSeries>

Note that your driver class will have to implement the Printer::ESCPOS::Roles::Profile Interface. This is a Moose Role and can be included in your class with the following line.

    use Moose;
    with 'Printer::ESCPOS::Roles::Profile';

By default the generic profile is loaded but if you have written your own Printer::ESCPOS::Profile::* class and want to override the generic class pass the I<profile> Param during object creation.

    my $device = Printer::ESCPOS->new(
        driverType => 'Network',
        deviceIp   => $printer_ip,
        devicePort => $port,
        profile    => 'USERCUSTOM'
    );

The above $device object will use the Printer::ESCPOS::Profile::USERCUSTOM profile.

=head2 deviceFilePath

File path for UNIX device file. e.g. "/dev/ttyACM0" this is a mandatory parameter if you are using B<File> or B<Serial> I<driverType>.

=head2 deviceIP

Contains the IP address of the device when its a network printer. The module creates L<IO:Socket::INET> object to connect to the printer. This can be passed in the constructor.

=head2 devicePort

Contains the network port of the device when its a network printer. The module creates L<IO:Socket::INET> object to connect to the printer. This can be passed in the constructor.

=head2 baudrate

When used as a local serial device you can set the I<baudrate> of the printer too. Default (38400) will usually work, but not always. 

=head2 serialOverUSB

Set this value to 1 if you are connecting your printer using the USB Cable but it shows up as a serial device and you are using the B<Serial> driver.

=head2 vendorId

USB Printers VendorId. use lsusb command to get this value

=head2 productId

USB Printers product Id. use lsusb command to get this value

=head2 endPoint

USB endPoint to write to.

=head2 timeout

Timeout for bulk write functions for the USB printer.

=head2 printer

Use this attribute to send commands to the printer

    $device->printer->setFont('a');
    $device->printer->text("blah blah blah\n");

=head1 USAGE

Refer to the following manual to get started with L<Printer::ESCPOS>

=over

=item *

L<Printer::ESCPOS::Manual>

=back

=head2 Quick usage summary in steps:

=over

=item 1.

Create a device object $device by providing parameters for one of the supported printer types. Call $device-E<gt>printer-E<gt>init to initialize the printer.

=item 2.

call text() and other Text formatting functions on $device-E<gt>printer for the data to be sent to the printer. Make sure to end it all with a linefeed $device-E<gt>printer-E<gt>lf().

=item 3.

Then call the print() method to dispatch the sequences from the module buffer to the printer

=back

     $device->printer->print()

Note: While you may call print() after every single command code, this is not advisable as some printers tend to choke up if you send them too many print commands in quick succession. To avoid this, aggregate the data to be sent to the printer with text() and other text formatting functions and then send it all in one go using print() at the very end.

=head1 NOTES

=over

=item *

In Serial mode if the printer prints out garbled characters instead of proper text, try specifying the baudrate parameter when you create the printer object. The default baudrate is set at 38400

=back

     $device = Printer::ESCPOS->new(
         driverType     => 'Serial',
         deviceFilePath => $path,
         baudrate       => 9600,
     );

=over

=item *

For ESC-P codes refer the guide from Epson L<http://support.epson.ru/upload/library_file/14/esc-p.pdf>

=back

=for :stopwords cpan testmatrix url annocpan anno bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through github at 
L<https://github.com/shantanubhadoria/perl-printer-escpos/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/shantanubhadoria/perl-printer-escpos>

  git clone git://github.com/shantanubhadoria/perl-printer-escpos.git

=head1 AUTHOR

Shantanu Bhadoria <shantanu@cpan.org> L<https://www.shantanubhadoria.com>

=head1 CONTRIBUTOR

=for stopwords Shantanu Bhadoria

Shantanu Bhadoria <shantanu att cpan dott org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Shantanu Bhadoria.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
