/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_instance_token.c
 * @brief command to test /private/token POSTing
 * @author Martin Schanzenbach
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State of a "POST /instances/$ID/private/token" CMD.
 */
struct TokenInstanceState
{

  /**
   * Handle for a "POST token" request.
   */
  struct TALER_MERCHANT_InstanceTokenPostHandle *itph;

  /**
   * Handle for a "DELETE token" request.
   */
  struct TALER_MERCHANT_InstanceTokenDeleteHandle *itdh;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Base URL of the merchant serving the request.
   */
  const char *merchant_url;

  /**
   * ID of the instance to run GET for.
   */
  const char *instance_id;

  /**
   * The received token (if any).
   */
  char *token;

  /**
   * Desired scope. Can be NULL
   */
  const char *scope;

  /**
   * Desired duration.
   */
  struct GNUNET_TIME_Relative duration;

  /**
   * Refreshable?
   */
  bool refreshable;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

  /**
   * DELETE or POST.
   */
  unsigned int is_delete;

};

/**
 * Callback for a POST /instances/$ID/private/token operation.
 *
 * @param cls closure for this function
 * @param hr response being processed
 */
static void
token_instance_cb (void *cls,
                   const struct TALER_MERCHANT_HttpResponse *hr)
{
  struct TokenInstanceState *tis = cls;
  const char *scope;
  struct GNUNET_TIME_Timestamp duration;
  bool refreshable;
  const char *error_name;
  unsigned int error_line;


  tis->itph = NULL;
  if (tis->http_status != hr->http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                hr->http_status,
                (int) hr->ec,
                TALER_TESTING_interpreter_get_current_label (tis->is));
    TALER_TESTING_interpreter_fail (tis->is);
    return;
  }
  switch (hr->http_status)
  {
  case MHD_HTTP_NO_CONTENT:
    GNUNET_assert (GNUNET_YES == tis->is_delete);
    break;
  case MHD_HTTP_OK:
    {
      /* Get token */
      struct GNUNET_JSON_Specification spec[] = {
        GNUNET_JSON_spec_string_copy ("access_token",
                                      &tis->token),
        GNUNET_JSON_spec_string ("scope",
                                 &scope),
        GNUNET_JSON_spec_bool ("refreshable",
                               &refreshable),
        GNUNET_JSON_spec_timestamp ("expiration",
                                    &duration),
        GNUNET_JSON_spec_end ()
      };

      GNUNET_assert (GNUNET_NO == tis->is_delete);
      if (GNUNET_OK !=
          GNUNET_JSON_parse (hr->reply,
                             spec,
                             &error_name,
                             &error_line))
      {
        char *js;

        js = json_dumps (hr->reply,
                         JSON_INDENT (1));
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Parser failed on %s:%u for input `%s'\n",
                    error_name,
                    error_line,
                    js);
        free (js);
        TALER_TESTING_FAIL (tis->is);
      }
      break;
    }
  case MHD_HTTP_BAD_REQUEST:
    /* likely invalid auth value, we do not check client-side */
    break;
  case MHD_HTTP_FORBIDDEN:
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unhandled HTTP status %u (%d) returned from /private/token operation.\n",
                hr->http_status,
                hr->ec);
  }


  TALER_TESTING_interpreter_next (tis->is);
}


/**
 * set a token
 *
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
set_token_instance_run (void *cls,
                        const struct TALER_TESTING_Command *cmd,
                        struct TALER_TESTING_Interpreter *is)
{
  const char *token_job_label = cls;
  const char *token;
  const struct TALER_TESTING_Command *tok_cmd;
  struct GNUNET_CURL_Context *cctx;
  char *authorization;

  cctx = TALER_TESTING_interpreter_get_context (is);
  GNUNET_assert (NULL != cctx);
  tok_cmd = TALER_TESTING_interpreter_lookup_command (
    is,
    token_job_label);
  TALER_TESTING_get_trait_bearer_token (tok_cmd,
                                        &token);
  GNUNET_assert (NULL != token);

  GNUNET_asprintf (&authorization,
                   "%s: Bearer %s",
                   MHD_HTTP_HEADER_AUTHORIZATION,
                   token);
  GNUNET_assert (GNUNET_OK ==
                 GNUNET_CURL_append_header (cctx,
                                            authorization));
  GNUNET_free (authorization);
  TALER_TESTING_interpreter_next (is);
}


/**
 * Run the "token /instances/$ID" CMD.
 *
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
token_instance_run (void *cls,
                    const struct TALER_TESTING_Command *cmd,
                    struct TALER_TESTING_Interpreter *is)
{
  struct TokenInstanceState *tis = cls;

  tis->is = is;
  if (GNUNET_NO == tis->is_delete)
    tis->itph = TALER_MERCHANT_instance_token_post (
      TALER_TESTING_interpreter_get_context (is),
      tis->merchant_url,
      tis->instance_id,
      tis->scope,
      tis->duration,
      tis->refreshable,
      &token_instance_cb,
      tis);
  else
    tis->itdh = TALER_MERCHANT_instance_token_delete (
      TALER_TESTING_interpreter_get_context (is),
      tis->merchant_url,
      tis->instance_id,
      &token_instance_cb,
      tis);
  GNUNET_assert ((NULL != tis->itph) || (NULL != tis->itdh));
}


/**
 * Free the state of a "POST instance token" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
token_instance_cleanup (void *cls,
                        const struct TALER_TESTING_Command *cmd)
{
  struct TokenInstanceState *tis = cls;

  if (NULL != tis->itph)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "%s /instance/$ID/token operation did not complete\n",
                (GNUNET_NO == tis->is_delete) ? "DELETE" : "POST");
    if (GNUNET_NO == tis->is_delete)
      TALER_MERCHANT_instance_token_post_cancel (tis->itph);
    else
      TALER_MERCHANT_instance_token_delete_cancel (tis->itdh);
  }
  GNUNET_free (tis);
}


/**
 * Offer internal data to other commands.
 *
 * @param cls closure
 * @param[out] ret result (could be anything)
 * @param trait name of the trait
 * @param index index number of the object to extract.
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
token_instance_traits (void *cls,
                       const void **ret,
                       const char *trait,
                       unsigned int index)
{
  struct TokenInstanceState *ais = cls;
  struct TALER_TESTING_Trait traits[] = {
    TALER_TESTING_make_trait_bearer_token (ais->token),
    TALER_TESTING_trait_end ()
  };

  return TALER_TESTING_get_trait (traits,
                                  ret,
                                  trait,
                                  index);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_delete_instance_token (const char *label,
                                                  const char *merchant_url,
                                                  const char *instance_id,
                                                  unsigned int http_status)
{
  struct TokenInstanceState *tis;

  tis = GNUNET_new (struct TokenInstanceState);
  tis->merchant_url = merchant_url;
  tis->instance_id = instance_id;
  tis->is_delete = GNUNET_YES;
  tis->http_status = http_status;

  {
    struct TALER_TESTING_Command cmd = {
      .cls = tis,
      .label = label,
      .run = &token_instance_run,
      .cleanup = &token_instance_cleanup,
      .traits = &token_instance_traits
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_set_instance_token (const char *label,
                                               const char *token_job_label)
{
  {
    struct TALER_TESTING_Command cmd = {
      .cls = (void*) token_job_label, // FIXME scope
      .label = label,
      .run = &set_token_instance_run,
      .cleanup = NULL,
      .traits = NULL
    };

    return cmd;
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_post_instance_token (const char *label,
                                                const char *merchant_url,
                                                const char *instance_id,
                                                const char *scope,
                                                struct GNUNET_TIME_Relative
                                                duration,
                                                bool refreshable,
                                                unsigned int http_status)
{
  struct TokenInstanceState *tis;

  tis = GNUNET_new (struct TokenInstanceState);
  tis->merchant_url = merchant_url;
  tis->instance_id = instance_id;
  tis->scope = scope;
  tis->duration = duration;
  tis->refreshable = refreshable;
  tis->is_delete = GNUNET_NO;
  tis->http_status = http_status;

  {
    struct TALER_TESTING_Command cmd = {
      .cls = tis,
      .label = label,
      .run = &token_instance_run,
      .cleanup = &token_instance_cleanup,
      .traits = &token_instance_traits
    };

    return cmd;
  }
}


/* end of testing_api_cmd_token_instance.c */
