/* tag: primary applet source file
 *
 * Copyright (C) 2000  Andrew Chatham
 * Copyright (C) 2003  Charles Duffy
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <string.h>
#include <config.h>

#include "krb-util.h"
#include "ticket_applet.h"
#include "properties.h"
#include "renew.h"

#include "egg-screen-help.h"

#ifdef ENABLE_TRACE
#define TRACE(string) fprintf(stderr,"TRACE: %s\n", string)
#else
#define TRACE(string) ;
#endif

static void about_verb(BonoboUIComponent *uih, gpointer data, const char* path);
static void help_verb(BonoboUIComponent *uih, gpointer data, const char* path);
static void refresh_menu_verb(BonoboUIComponent *uih, gpointer data, const char *path);
static GObjectClass *parent_class;

const BonoboUIVerb ticket_menu_verbs[] = {
	BONOBO_UI_VERB ("TicketProperties", properties_verb),
	BONOBO_UI_VERB ("TicketRefresh", refresh_menu_verb),
	BONOBO_UI_VERB ("TicketHelp", help_verb),
	BONOBO_UI_VERB ("TicketAbout", about_verb),
	BONOBO_UI_VERB_END
};

static GType ticket_applet_get_type(void);

const gchar *description = "A panel applet which displays how much time you have left on your kerberos tickets and allows you to renew them.";

const gchar *copyright =  _("(c) 2000 Andrew Chatham; (c) 2003 Charles Duffy");
const gchar *authors[] = {"Andrew Chatham <andrew.chatham@duke.edu>",
       			  "Charles Duffy <cduffy@spamcop.net>",
			  NULL};

static void
renew_cb(GtkWidget *applet, gpointer data)
{
  TicketApplet * ad = (TicketApplet *) data;

  renew(ad);
}

static void
about_verb(BonoboUIComponent *uih, gpointer data, const char* path)
{
  static GtkWidget *about_box = NULL;
  
  if (about_box) {
    gdk_window_show(about_box->window);
    gdk_window_raise(about_box->window);
    return;
  }

  about_box = gnome_about_new( _("TicketApplet"), VERSION, copyright,
			       description, authors, NULL, NULL, NULL);
  gtk_signal_connect(GTK_OBJECT(about_box), "destroy",
		     GTK_SIGNAL_FUNC(gtk_widget_destroyed),
		     &about_box);
  gtk_widget_show(about_box);
  return;
}

/* stolen from show_help in the gnome2 fish applet */
static void
help_verb(BonoboUIComponent *uih, gpointer data, const char* path)
{
        GError *error = NULL;
	GtkWidget *ticket_applet_widget = GTK_WIDGET(data);
                                                                                   
        egg_help_display_desktop_on_screen (
                NULL, "ticket-applet-2-manual", "ticket-applet-2-manual", NULL,
                gtk_widget_get_screen (ticket_applet_widget),
                &error);
                                                                                   
        if (error) {
                GtkWidget *dialog;
                                                                                   
                dialog = gtk_message_dialog_new (
                                NULL,
                                GTK_DIALOG_DESTROY_WITH_PARENT,
                                GTK_MESSAGE_ERROR,
                                GTK_BUTTONS_CLOSE,
                                _("There was an error displaying help: %s"),
                                error->message);

                g_signal_connect (dialog, "response",
                                  G_CALLBACK (gtk_widget_destroy),
                                  NULL);

                gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
                gtk_window_set_screen (GTK_WINDOW (dialog),
                                       gtk_widget_get_screen (ticket_applet_widget));
                gtk_widget_show (dialog);
                g_error_free (error);
        }
}

static int refresh_timeout_cb(gpointer);

void change_timeout(TicketApplet *ad)
{
  gtk_timeout_remove(ad->timeout);
  ad->timeout = gtk_timeout_add(ad->refresh, refresh_timeout_cb, ad);
}

void
refresh(TicketApplet *ad)
{
  gchar *msg;

  change_timeout(ad);
  msg = krb_time_left_as_string();
  gtk_label_set_text(GTK_LABEL(ad->label), msg);
  free(msg);

}

static void 
refresh_menu_verb(BonoboUIComponent *uih, gpointer data, const char *path)
{
  refresh((TicketApplet *) data);
}

static int
refresh_timeout_cb(gpointer data)
{
  refresh((TicketApplet *) data);
  return 0;
}

//function pointer for g_list_foreach to free everything
static void
free_node(gpointer data, gpointer other)
{
  g_free(data);
}

static void
free_list(GList *list)
{
  if (list) {
    g_list_foreach(list, free_node, NULL);
    g_list_free(list);
  }
}

static void
init_values(TicketApplet *ad)
{
  GError* error;
  gpointer iterator;

  TRACE("init_values");

  // TODO: actually check error
  ad->user = panel_applet_gconf_get_string(
		  PANEL_APPLET(ad), USER_CONFIG_KEY, &error);
  ad->realm = panel_applet_gconf_get_string(
		  PANEL_APPLET(ad), REALM_CONFIG_KEY, &error);
  ad->refresh = panel_applet_gconf_get_int(
		  PANEL_APPLET(ad), REFRESH_CONFIG_KEY, &error);
  if(ad->refresh < MIN_REFRESH) ad->refresh = MIN_REFRESH;

  iterator = gnome_config_init_iterator(panel_applet_gconf_get_full_key(PANEL_APPLET(ad), COMMANDS_CONFIG_KEY));
  free_list(ad->cmds);
  ad->cmds = NULL;

  if (iterator != NULL) {
    gchar *key, *val;
    while ((iterator = gnome_config_iterator_next(iterator, 
						  &key, &val))) {
      g_free(key);
      ad->cmds = g_list_prepend(ad->cmds, val);
    }
  }

  ad->p_refresh = ad->refresh;

}

/* This is a hack around the fact that gtk+ doesn't
 * propogate button presses on button2/3.
 *
 * Stolen from the GNOME CD Player Applet.
 */
static gboolean
button_press_hack (GtkWidget      *widget,
                   GdkEventButton *event,
                   GtkWidget      *applet)
{
  if (event->button == 3 || event->button == 2) {
    gtk_propagate_event (applet, (GdkEvent *) event);
    return TRUE;
  }
  return FALSE;
}


static void
ticket_applet_setup (TicketApplet *ad)
{
  GtkWidget *frame;
  GtkWidget *button;
  GtkWidget *label;

  TRACE("ticket_applet_setup");

  panel_applet_setup_menu_from_file(PANEL_APPLET(ad), UI_DATA_DIR,
				    "GNOME_TicketApplet.xml",
			  	    NULL, ticket_menu_verbs, ad);

  frame = gtk_frame_new(NULL);
  button = gtk_button_new();
  label = gtk_label_new("???");

  ad->frame = frame;
  ad->button = button;
  ad->label = label;

  ad->cmds = NULL;

  gtk_container_add(GTK_CONTAINER(button), label);
  gtk_container_add(GTK_CONTAINER(frame), button);
  gtk_container_add(GTK_CONTAINER(PANEL_APPLET(ad)), frame);

  g_signal_connect(G_OBJECT(ad->button), "button_press_event", G_CALLBACK(button_press_hack), (gpointer)ad);
  g_signal_connect(G_OBJECT(ad->button), "clicked", G_CALLBACK(renew_cb), (gpointer)ad);

  init_values(ad);

  ad->timeout = gtk_timeout_add(ad->refresh, refresh_timeout_cb, ad);
  refresh(ad);

  gtk_widget_show_all(GTK_WIDGET(ad));
}

static gboolean
ticket_applet_fill (TicketApplet *ticket_applet)
{
	/* GError *error = NULL; */

	TRACE("ticket_applet_fill");

	panel_applet_add_preferences(PANEL_APPLET(ticket_applet), "/schemas/apps/ticket_applet/prefs", NULL);
	ticket_applet_setup(ticket_applet);
	/* setup_gconf(ticket_applet); */ /* TODO: load data from gconf */
	/* setup_ticket_widget(ticket_applet); */ /* TODO: connect signals for redraw events and such */
	return TRUE;
}

static gboolean
ticket_applet_factory (PanelApplet *applet,
                       const gchar *iid,
                       gpointer    data)
{
  gboolean retval = FALSE;
  TRACE("ticket_applet_factory");

  if(!strcmp(iid, "OAFIID:GNOME_TicketApplet"))
    retval = ticket_applet_fill(TICKET_APPLET(applet));
  return retval;
}

static void ticket_applet_destroy(GtkObject *object)
{
  /* TicketApplet *applet = (TicketApplet*) object; */
  TRACE("ticket_applet_destroy");

  /* TODO: use fish dereg code as sample */
  GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static void ticket_applet_class_init (TicketAppletClass *klass)
{
  /* PanelAppletClass *applet_class = (PanelAppletClass*) klass; */
  GtkObjectClass *gtkobject_class = (GtkObjectClass*) klass;

  TRACE("ticket_applet_class_init");
  parent_class = g_type_class_peek_parent(klass);
  /* applet_class->change_orient = ticket_applet_change_orient; */
  gtkobject_class->destroy = ticket_applet_destroy;
}

static void
ticket_applet_instance_init( TicketApplet *applet, TicketAppletClass *klass)
{
	TRACE("ticket_applet_instance_init");
	/* TODO: initialize applet->applet? */
	applet->frame = applet->button = applet->label = NULL;
	applet->timeout = 0;

	applet->p_box = NULL;
	applet->refresh = 0;
	applet->user = applet->realm = NULL;
	applet->cmds = NULL;

	applet->p_refresh = 0;
	applet->user_entry = NULL;
	applet->realm_entry = NULL;
	applet->cmd_list = NULL;

	applet->pass_entry = NULL;

	panel_applet_set_flags(PANEL_APPLET(applet), PANEL_APPLET_EXPAND_MINOR);
}

static GType
ticket_applet_get_type(void)
{
	static GType type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof(PanelAppletClass),
			NULL, NULL,
			(GClassInitFunc)ticket_applet_class_init,
			NULL, NULL,
			sizeof(TicketApplet),
			0,
			(GInstanceInitFunc)ticket_applet_instance_init,
			NULL
		};
		type = g_type_register_static (
			PANEL_TYPE_APPLET, "TicketApplet", &info, 0);
	}
	return type;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_TicketApplet_Factory",
			     ticket_applet_get_type(),
			     "Ticket",
			     "0.3.0",
			     ticket_applet_factory,
			     NULL)
