/*

  sshregex.h

  Author: Antti Huima <huima@ssh.fi>

  Copyright (c) 1999 SSH Communications Security, Finland
  All rights reserved.

  Created Tue Sep  7 11:44:00 1999.

  */

#ifndef SSH_REGEX_H_INCLUDED
#define SSH_REGEX_H_INCLUDED

#include "sshbuffer.h"

typedef struct ssh_rex_matcher *SshRegexMatcher;
typedef struct ssh_rex_global_context *SshRegexContext;

/* For the descriptions of the syntaxes, please see the separate
   file `REGEX-SYNTAX'. */
typedef enum {
  SSH_REGEX_SYNTAX_SSH,
  SSH_REGEX_SYNTAX_EGREP,
  SSH_REGEX_SYNTAX_ZSH_FILEGLOB
} SshRegexSyntax;

typedef struct {
  unsigned char *literal;
  size_t literal_len;
  int subexpr;
} SshRegexSubstitutionItem;

/* Create a global regex allocation context which must be passed to
   ssh_regex_create. */
SshRegexContext ssh_regex_create_context(void);

/* Free a regex context. When the context is freed, no more regular
   expressions created with this context should exist but they should
   have been freed by calling ssh_regex_free. */
void ssh_regex_free_context(SshRegexContext c);

/* Create a matcher that matches the regular expression `regex'.
   Returns NULL if `regex' is invalid.

   `syntax' chooses the syntax which is used to parse the regular
   expression. See the file REGEX-SYNTAX in the source directory for
   descriptions of the syntaxes. */
SshRegexMatcher ssh_regex_create(SshRegexContext context,
                                 const char *regex,
                                 SshRegexSyntax syntax);

/* Free a matcher and data associated with it. */
void ssh_regex_free(SshRegexMatcher matcher);

/* Try to match against the NUL-terminated string `data'. Return TRUE
   if match succeeded, otherwise FALSE. */
Boolean ssh_regex_match_cstr(SshRegexMatcher matcher,
                             const char *data);

/* As above, but do not treat NUL specially. */
Boolean ssh_regex_match(SshRegexMatcher matcher,
                        const unsigned char *data,
                        size_t data_len);

/* As above, but use an SshBuffer instead. */
Boolean ssh_regex_match_buffer(SshRegexMatcher matcher,
                               SshBuffer buffer);

/* Try to match against the NUL-terminated string `data'. Return TRUE
   if match succeeded or end of the string was reached before a match
   completed, otherwise FALSE. */
Boolean ssh_regex_match_cstr_prefix(SshRegexMatcher matcher,
                                    const char *data);

/* Access, after a succesful match, the `subexpr_num'th subexpression.
   `start_ptr' points to the start of the matched text and `match_len'
   is set to the length of the matched text. If the subexpression did
   not match, returns FALSE, otherwise TRUE.

   subexpr_num == 0 denotes the whole matched string.  The first
   parenthesized subexpression is obtained by setting subexpr_num ==
   1. */
Boolean ssh_regex_access_submatch(SshRegexMatcher matcher,
                                  int subexpr_num,
                                  int *index,
                                  size_t *match_len);

/* Get the `subexpr_num'th submatch as a NUL-terminated string. The
   string is allocated by the regex library and you should NOT free
   it. The string will be freed later, at least when `matcher' is
   freed. */
unsigned char *ssh_regex_get_submatch(SshRegexMatcher matcher,
                                      int subexpr_num);

/* `Substitution', basically creating a new string composed of some
   submatches. This is a convenience function. */   
unsigned char *ssh_regex_compose(SshRegexMatcher matcher,
                                 SshRegexSubstitutionItem *items,
                                 int num_items,
                                 size_t *length_return);
                                  

#endif /* SSH_REGEX_H_INCLUDED */
