/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)es_glue.c	1.7 96/01/24 Sun Microsystems"

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/xv_xrect.h>
#include <xview/notice.h>
#include <skip_conf.h>
#include "skiptool_ui.h"

#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_lib.h>

#include "es_glue.h"

#ifdef SYSV
#include <sys/systeminfo.h>
#include <sys/resource.h>
#endif

static Server_image			unknown, none, icon_v1, icon_v2, skip;
const char				*config = SKIP_DIR"/hosts";
char					ifname[STRSZ], *progname;
int					es_display_depth = 0;
es_system_t				*skip_acl = NULL;

static void
usage(char *prog)
{
	fprintf(stderr, "%s: <device>\n", prog);
	exit(1);
}


/*
 * display simple alert
 */
void
alert(Panel_item owner, char *s, int bell)
{
	Panel				panel;
	Xv_notice			notice; 

	if (owner == NULL) {
		panel = Skiptool_base_window->base_window;
	} else {
		panel = xv_get(owner, PANEL_PARENT_PANEL);
	}
	notice = xv_create(panel,
			NOTICE,
			XV_SHOW, TRUE,
			NOTICE_LOCK_SCREEN, TRUE,
			NOTICE_NO_BEEPING, bell ? FALSE : TRUE,
			NOTICE_MESSAGE_STRING, s, NULL,
			NULL);
	xv_destroy_safe(notice);
}

/*
 * clear left footer
 */
static void
clear_footer()
{

	(void) xv_set(Skiptool_base_window->base_window,
		FRAME_LEFT_FOOTER, "",
		NULL);
}

/*
 * set the frame left footer message
 */
void
set_footer(char *s, boolean_t bell)
{
	if (bell) {
		window_bell(Skiptool_base_window->base_window);
	}

	(void) xv_set(Skiptool_base_window->base_window,
		FRAME_LEFT_FOOTER, s,
		NULL);
	(void) start_timer("footer",  SKIP_MSG_WAIT, clear_footer, 0);
}


/*
 * return hostname component of access control information
 */
static char *
hostname_of(skip_param_t *params)
{
	struct hostent		*hp = NULL;

	hp = gethostbyaddr((char *) &params->ip_addr, 4, AF_INET);

	if (hp) {
		return (hp->h_name);
	} else {
		return (inet_ntoa(params->ip_addr));
	}
}

/*
 * return a glyph corresponding to access control entry
 */
static Xv_opaque
glyph_of(skip_param_t *params)
{
	switch (params->version) {
	case SKIP_NONE:
		return (none);

	case SKIP_V1:
		return (icon_v1);

	case SKIP_V2:
		return (icon_v2);
	}
	return (unknown);
}

/*
 * set an algorithm choice item to the specified value
 */
int
set_alg_selector(Xv_opaque item, char *algname)
{
	register int		i;
	register char		*setting;

	for (i = 0; i < skip_max_algs; i++) {
		setting = (char *) xv_get(item, PANEL_CHOICE_STRING, i);
		if (setting && !strcmp(setting, algname)) {
			xv_set(item, PANEL_VALUE, i, NULL);
			return (i);
		}
	}
	return (-1);
}

/*
 * build menu of name space identifiers
 */
void
build_nsid_menu(skiptool_hostwin_objects *win)
{
	register int	nsid;

	for (nsid = 0; nsid < skip_max_nsids; nsid++) {
		(void) xv_set(win->s_nsid,
			PANEL_CHOICE_STRING, nsid,
			skip_nsids[nsid].name,
			NULL);
		(void) xv_set(win->r_nsid,
			PANEL_CHOICE_STRING, nsid,
			skip_nsids[nsid].name,
			NULL);
	}
}

/*
 * build menu of available key/traffic algorithms for host window
 */
void
build_alg_menus(skiptool_hostwin_objects *win)
{
	register int		alg, kij_item = 0, kp_item = 0;
	unsigned char		version;

	version = ((int) xv_get(win->secure, PANEL_VALUE) == 2 ?
							SKIP_V1 : SKIP_V2);

	/*
	 * destroy existing menus
	 */
	xv_destroy(win->kij_alg);
	win->kij_alg = NULL;
	xv_destroy(win->kp_alg);
	win->kp_alg = NULL;
	notify_dispatch();

	/*
	 * get GUIDE to recreate them anew
	 */
	(void ) skiptool_hostwin_objects_initialize(
		win, Skiptool_base_window->base_window);

	for (alg = 1; alg < skip_max_algs; alg++) {
		if (skip_supported_kij_alg(alg, version)) {
			(void) xv_set(win->kij_alg,
				PANEL_CHOICE_STRING, kij_item,
				skip_kij_alg_to_name(alg, version),
				NULL);
			kij_item++;
		}
		if (skip_supported_kp_alg(alg, version)) {
			(void) xv_set(win->kp_alg,
				PANEL_CHOICE_STRING, kp_item,
				skip_kp_alg_to_name(alg, version),
				NULL);
			kp_item++;
		}
	}
	if (kij_item) {
		(void) xv_set(win->kij_alg,
			PANEL_VALUE, 0,
			XV_SHOW, TRUE,
			NULL
		);
	}
	if (kp_item) {
		(void) xv_set(win->kp_alg,
			PANEL_VALUE, 0,
			XV_SHOW, TRUE,
			NULL
		);
	}
	notify_dispatch();

	XFlush((Display *) 
		xv_get(win->hostwin, XV_DISPLAY)
	);
}

/*
 * make correct selectors available on menu
 */
void
selectors_available(skiptool_hostwin_objects *win)
{
	int		secure, version, r_nsid;

	version = (int) xv_get(win->secure, PANEL_VALUE);

	secure = version > 0;

	/*
	 * make settings available/unavailable depending on mode
	 */

	/*
	 * nsid setting
	 */
	switch (version) {
	case 0:
		/*
		 * Off
		 */
		(void) xv_set(win->r_nsid_msg, 
				PANEL_INACTIVE, TRUE,
				NULL);
		(void) xv_set(win->r_nsid, 
				PANEL_VALUE, 0,
				PANEL_INACTIVE, TRUE,
				NULL);
		(void) xv_set(win->s_nsid_msg, 
				PANEL_INACTIVE, TRUE,
				NULL);
		(void) xv_set(win->s_nsid, 
				PANEL_VALUE, 0,
				PANEL_INACTIVE, TRUE,
				NULL);
		break;
	case 1:
		/*
		 * On
		 */
		(void) xv_set(win->r_nsid_msg, 
				PANEL_INACTIVE, FALSE,
				NULL);
		(void) xv_set(win->r_nsid, 
				PANEL_INACTIVE, FALSE,
				NULL);
		(void) xv_set(win->s_nsid_msg, 
				PANEL_INACTIVE, FALSE,
				NULL);
		(void) xv_set(win->s_nsid, 
				PANEL_INACTIVE, FALSE,
				NULL);
		break;
	case 2:
		/*
		 * SunScreen mode
		 */
		(void) xv_set(win->r_nsid_msg, 
				PANEL_INACTIVE, TRUE,
				NULL);
		(void) xv_set(win->r_nsid, 
				PANEL_VALUE, 1,
				PANEL_INACTIVE, TRUE,
				NULL);
		(void) xv_set(win->s_nsid, 
				PANEL_VALUE, 0,
				PANEL_INACTIVE, TRUE,
				NULL);
		(void) xv_set(win->s_nsid_msg, 
				PANEL_INACTIVE, TRUE,
				NULL);
		break;
	}

	/*
	 * set key id
	 */
	r_nsid = xv_get(win->r_nsid, PANEL_VALUE);
	if (r_nsid) {
		(void) xv_set(win->key_id, 
					PANEL_INACTIVE, !secure,
					NULL);
	} else {
		(void) xv_set(win->key_id, 
					PANEL_INACTIVE, TRUE,
					NULL);
	}


	(void) xv_set(win->kij_alg_msg, 
				PANEL_INACTIVE, !secure,
				NULL);
	(void) xv_set(win->kij_alg, 
				PANEL_INACTIVE, !secure,
				NULL);
	(void) xv_set(win->kp_alg_msg, 
				PANEL_INACTIVE, !secure,
				NULL);
	(void) xv_set(win->kp_alg, 
				PANEL_INACTIVE, !secure,
				NULL);
}

/*
 * fill out an edit window
 */
static void
populate_edit_win(es_acl_t *access)
{
	skiptool_hostwin_objects	*win = access->win;
	skip_param_t			*params = &access->params;
	char				s[STRSZ], *algname;

	switch (params->version) {
	case SKIP_NONE:
		(void) xv_set(win->secure,
			PANEL_VALUE, 0,
			NULL);
		break;
	case SKIP_V1:
		(void) xv_set(win->secure,
			PANEL_VALUE, 2,
			NULL);
		break;
	case SKIP_V2:
		(void) xv_set(win->secure,
			PANEL_VALUE, 1,
			NULL);
		break;
	}

	(void) xv_set(win->hostaddr,
			PANEL_VALUE, hostname_of(params),
			NULL);

	(void) xv_set(win->r_nsid,
			PANEL_VALUE, params->r_nsid,
			NULL);

	(void) xv_set(win->s_nsid,
			PANEL_VALUE, params->s_nsid,
			NULL);

	if (skip_keyid_to_s(&params->r_mkeyid, params->r_nsid, s)) {
		alert(win->skip_icon, "Invalid key ID", B_TRUE);
		return;
	} 
	(void) xv_set(win->key_id,
			PANEL_VALUE, s,
			NULL);
	
	build_alg_menus(win);
	selectors_available(win);
	algname = skip_kij_alg_to_name(params->kij_alg, params->version);
	(void) set_alg_selector(win->kij_alg, algname);
	algname = skip_kp_alg_to_name(params->kp_alg, params->version);
	(void) set_alg_selector(win->kp_alg, algname);
}

/* new_win()
 *
 * Create a new host window
 */
skiptool_hostwin_objects *
new_win()
{
	skiptool_hostwin_objects	*win;

	win = skiptool_hostwin_objects_initialize(NULL,
					Skiptool_base_window->base_window);
	if (win == NULL) {
		alert(NULL, "Cannot create new window", B_TRUE);
		return (NULL);
	}
	if (skip) {
		(void) xv_set(win->skip_icon,
					PANEL_LABEL_IMAGE,
					skip,
					NULL);
	}
	(void) xv_set(win->hostwin, 
				XV_LABEL, "Edit system...",
				NULL);

	(void) xv_set(win->hostaddr, 
				PANEL_READ_ONLY, TRUE,
				PANEL_VALUE_UNDERLINED, FALSE,
				NULL);
	build_nsid_menu(win);
	return (win);
}

/*ARGSUSED*/
static void
null(es_acl_t *access)
{
}

/* systems_count()
 *
 * count the number of systems authorised to connect
 */
static int
systems_count()
{
	return (skip_list_hosts(ifname, null));
}

static void
show_secure_icon()
{
	char				msg[STRSZ];
	Icon				icon;

	icon = xv_get(Skiptool_base_window->base_window, FRAME_ICON);
	if (icon) {
		sprintf(msg, "%s : on", ifname);
		xv_set(icon, ICON_TRANSPARENT_LABEL, msg, NULL);
	}
}

static void
show_unsecure_icon()
{
	char				msg[STRSZ];
	Icon				icon;

	icon = xv_get(Skiptool_base_window->base_window, FRAME_ICON);
	if (icon) {
		sprintf(msg, "%s : off", ifname);
		xv_set(icon, ICON_TRANSPARENT_LABEL, msg, NULL);
	}
}

/*
 * show current operating mode
 */
void
show_mode()
{
	int				n, mode = skip_get_mode(ifname);
	char				msg[STRSZ];

	n = systems_count();

	switch (mode) {

	case SkipAccessControlOff:
		show_unsecure_icon();
		(void) xv_set(Skiptool_base_window->system_status,
			PANEL_LABEL_STRING, "Any system may connect.",
			NULL);
		set_footer("Access control disabled", B_FALSE);
		xv_set(Skiptool_base_window->access_ctrl,	
			PANEL_VALUE, 0,
			NULL);
		break;

	case SkipInteractive:
	case SkipAccessControlOn:
		show_secure_icon();
		if (n == 1) {
			sprintf(msg, "1 system may connect.");
		} else {
			sprintf(msg, "%d systems may connect.", n);
		}
		(void) xv_set(Skiptool_base_window->system_status,
			PANEL_LABEL_STRING, msg,
			NULL);
		set_footer("Access control enabled", B_FALSE);
		xv_set(Skiptool_base_window->access_ctrl,	
			PANEL_VALUE, 1,
			NULL);
		break;

	default:
		show_unsecure_icon();
		set_footer("Access control failed", B_FALSE);
	}
}

static void
pin_window(skiptool_hostwin_objects *win)
{
	xv_set(win->hostwin, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
}

void
hostwin_apply(skiptool_hostwin_objects *win)
{
	extern 	skiptool_hostwin_objects	*Skiptool_hostwin;
	es_acl_t				a = { 0 }, *access = &a,
						*newaccess;
	boolean_t				secure, is_add_win;
	int					i, rc, rows, version,
						r_nsid, s_nsid;
	char					*kij_algname, *kp_algname,
						*host, *r_mkeyidstr, msg[STRSZ];
	char					r_mkeyidbuf[STRSZ];
	int					old_version = -1;

	is_add_win = (win == Skiptool_hostwin);
	
	if (is_add_win) {
		/*
		 * if this is the add window, create new acl info
		 */
		memset((char *) access, 0, sizeof(es_acl_t));
	} else {
		/*
		 * retrieve ACL info
		 */
		access = (es_acl_t *) xv_get(win->hostwin, WIN_CLIENT_DATA);
		old_version = access->params.version;
	}

	/*
	 * Gather an validate info from the window
	 */
	host = (char *) xv_get(win->hostaddr, PANEL_VALUE);

	secure = xv_get(win->secure, PANEL_VALUE);

	switch (secure) {
	case 0:	/* Off */
		version = SKIP_NONE;
		kij_algname = kp_algname = SKIP_NO_ALG;
		r_nsid = s_nsid = 0;  r_mkeyidstr= "";
		break;

	case 1:	/* On */
	case 2:	/* On (SunScreen compatibility) */
		version = (secure == 1 ? SKIP_V2 : SKIP_V1);
		r_nsid = xv_get(win->r_nsid, PANEL_VALUE);
		s_nsid = xv_get(win->s_nsid, PANEL_VALUE);
		if (r_nsid) {
			r_mkeyidstr = (char *) xv_get(win->key_id, PANEL_VALUE);
			if (strlen(r_mkeyidstr) == 0) {
				/*
				 * don't let the window go without a
				 * key id - unless its IPv4 where we can
				 * insert a default
				 */
				if (r_nsid == SKIP_NSID_IPV4) {
					r_mkeyidstr = host;
				} else {
					alert(win->key_id,
					"You must specify a key ID", B_TRUE);
					pin_window(win);
					return;
				}
			}
		} else {
			 r_mkeyidstr= "";
		}

		kij_algname = (char *) xv_get(win->kij_alg,
					PANEL_CHOICE_STRING,
					xv_get(win->kij_alg,
						PANEL_VALUE));

		kp_algname = (char *) xv_get(win->kp_alg,
					PANEL_CHOICE_STRING,
					xv_get(win->kp_alg,
						PANEL_VALUE));
		break;
	}

	if (skip_host_to_addr(host, &access->params.ip_addr)) {
		if (strlen(host) == 0) {
			sprintf(msg, "No hostname specified");
		} else {
			sprintf(msg, "Bad hostname \"%s\"", host);
		}
		alert(win->hostwin, msg, B_TRUE);
		pin_window(win);
		return;
	}
			
	access->params.version	= version;
        access->params.s_nsid	= s_nsid;
        access->params.r_nsid	= r_nsid;
        access->params.kij_alg	= skip_name_to_kij_alg(kij_algname, version);
        access->params.kp_alg	= skip_name_to_kp_alg(kp_algname, version);
        access->params.mac_alg	= 0;
        access->params.comp_alg	= 0;

        if (skip_s_to_keyid(r_mkeyidstr, r_nsid, &access->params.r_mkeyid)) {
		sprintf(msg, "Bad %s \"%s\"", skip_nsids[r_nsid].name,
								r_mkeyidstr);
		alert(win->hostwin, msg, B_TRUE);
		pin_window(win);
		return;
	}
 
	if (!is_add_win) {
		(void) skip_disallow_host(ifname, &access->params);
	}
	rc = skip_allow_host(ifname, &access->params);

	if (rc < 0) {
		alert(win->hostwin, skip_errmsg, B_TRUE);
		pin_window(win);
		return;
	}

	rows = xv_get(Skiptool_base_window->hostlist, PANEL_LIST_NROWS);

	if (is_add_win) {
		/*
		 * create a permanent entry for the ACL
		 */
		newaccess = (es_acl_t *) calloc(1, sizeof (*access));
		if (newaccess == NULL) {
			alert(win->hostwin,
				"Error adding new system (out of memory)",
				B_TRUE);
			pin_window(win);
			return;
		}
		*newaccess = *access;
		access = newaccess;
		/*
		 * insert new host
		 */
		(void) xv_set(Skiptool_base_window->hostlist,
			PANEL_LIST_INSERT_DUPLICATE, FALSE,
			PANEL_LIST_INSERT, rows,
			PANEL_LIST_STRING, rows, hostname_of(&access->params),
			PANEL_LIST_GLYPH, rows, glyph_of(&access->params),
			PANEL_LIST_CLIENT_DATA, rows, access,
			NULL);
		show_mode();
		return;
	}
	if (access->params.version != old_version) {
		/*
		 * need to update list glyph
		 */
		for (i = 0; i < rows; i++) {
			if (access == (es_acl_t *)
				xv_get(Skiptool_base_window->hostlist,
					PANEL_LIST_CLIENT_DATA, i, NULL)) {
				(void) xv_set(Skiptool_base_window->hostlist,
				PANEL_LIST_GLYPH, i, glyph_of(&access->params),
				NULL);
			}
		}
	}
        (void) skip_keyid_to_s(&access->params.r_mkeyid, r_nsid, r_mkeyidbuf);
 
	(void) xv_set(win->key_id, PANEL_VALUE, r_mkeyidbuf, NULL);
	show_mode();
}

void
save_or_discard(skiptool_hostwin_objects *win)
{
	Panel					panel;
	boolean_t				changed;
	int					notice_stat;
	extern 	skiptool_hostwin_objects	*Skiptool_hostwin;
	es_acl_t				*oldaccess;
	skip_key_var_t				r_mkeyid;
	int					secure, version, r_nsid, s_nsid;
	char					*kij_algname, *kp_algname,
						*host, *r_mkeyidstr, msg[STRSZ];

	if (win == Skiptool_hostwin) {
		xv_set(win->hostwin,
			XV_SHOW, FALSE,
			NULL);
		return;
	}

	host = (char *) xv_get(win->hostaddr, PANEL_VALUE);
	secure = xv_get(win->secure, PANEL_VALUE);

	switch (secure) {
	case 0:	/* Off */
		version = SKIP_NONE;
		kij_algname = kp_algname = SKIP_NO_ALG;
		r_nsid = s_nsid = 0;  r_mkeyidstr= "";
		break;

	case 1:	/* On */
	case 2:	/* On (SunScreen compatibility) */
		version = (secure == 1 ? SKIP_V2 : SKIP_V1);
		r_nsid = xv_get(win->r_nsid, PANEL_VALUE);
		s_nsid = xv_get(win->s_nsid, PANEL_VALUE);
		if (r_nsid) {
			r_mkeyidstr = (char *) xv_get(win->key_id, PANEL_VALUE);
		} else {
			r_mkeyidstr= "";
		}

		kij_algname = (char *) xv_get(win->kij_alg,
					PANEL_CHOICE_STRING,
					xv_get(win->kij_alg,
						PANEL_VALUE));

		kp_algname = (char *) xv_get(win->kp_alg,
					PANEL_CHOICE_STRING,
					xv_get(win->kp_alg,
						PANEL_VALUE));
		break;
	}

	oldaccess = (es_acl_t *) xv_get(win->hostwin, WIN_CLIENT_DATA);

	changed =
		(oldaccess->params.version != version) ||
        	(oldaccess->params.s_nsid != s_nsid) ||
        	(oldaccess->params.r_nsid != r_nsid) ||
        	(oldaccess->params.kij_alg !=
				skip_name_to_kij_alg(kij_algname, version)) ||
        	(oldaccess->params.kp_alg !=
				skip_name_to_kp_alg(kp_algname, version));

        (void) skip_s_to_keyid(r_mkeyidstr, r_nsid, &r_mkeyid);
 
	if (!changed && (oldaccess->params.r_mkeyid.len == r_mkeyid.len) &&
		!memcmp((caddr_t) r_mkeyid.buf,
			(caddr_t) oldaccess->params.r_mkeyid.buf,
			r_mkeyid.len)) {
		xv_set(win->hostwin,
			XV_SHOW, FALSE,
			NULL);
		return;
	}

	panel = xv_get(Skiptool_base_window->hostlist, PANEL_PARENT_PANEL);
	sprintf(msg, "You have changed the configuration for %s", host);
	(void) xv_create(panel,
			NOTICE,
			XV_SHOW, TRUE,
			NOTICE_LOCK_SCREEN, TRUE,
			NOTICE_MESSAGE_STRINGS, msg, NULL,
			NOTICE_BUTTON_YES, "Apply changes",
			NOTICE_BUTTON_NO, "Discard changes",
			NOTICE_STATUS, &notice_stat,
			NULL);
	if (notice_stat == NOTICE_YES) {
		hostwin_apply(win);
	} else {
		populate_edit_win(oldaccess);
	}
}

/* edit_win_done()
 *
 * Handle the closure of an edit window
 */
/*ARGSUSED*/
void
edit_win_done(Frame subframe)
{
	es_acl_t		*access;

	access = (es_acl_t *) xv_get(subframe, WIN_CLIENT_DATA);

	save_or_discard(access->win);
}


void
host_delete()
{
	extern int				editor();
	es_acl_t				*access;
	int					selected, rows;

	selected = xv_get(Skiptool_base_window->hostlist,
				PANEL_LIST_FIRST_SELECTED);

	if (selected == -1) {
		return;
	}

	access = (es_acl_t *) xv_get(Skiptool_base_window->hostlist,
				PANEL_LIST_CLIENT_DATA, selected);

	if (skip_disallow_host(ifname, &access->params) < 0) {
		alert(NULL, skip_errmsg, B_TRUE);
		return;
	}

	inform_list_delete(&access->params.ip_addr);

	if (access->win) {
		xv_destroy_safe(access->win->hostwin);
		free((char *) access->win);
	}
	free((char *) access);

	(void) xv_set(Skiptool_base_window->hostlist,
				PANEL_LIST_DELETE, selected,
				NULL);

 
	rows = xv_get(Skiptool_base_window->hostlist, PANEL_LIST_NROWS);
	if (rows) {
		if (selected == rows) {
			selected = rows - 1;
		}
		(void) xv_set(Skiptool_base_window->hostlist,
				PANEL_LIST_SELECT, selected, TRUE,
				NULL);
		(void) editor(Select, selected);
	}
	show_mode();
}

void
stats(int which)
{
	pid_t					pid;
	int					fd;
	char					*opt, *t, s[STRSZ], *argv[10];
	char					ifstats[STRSZ];
	struct rlimit				rl;

	switch (which) {
	case 0:
		/*
		 * interface stats
		 */
		opt = "-I";
		sprintf(ifstats, "SKIP Interface Statistics (%s)", ifname);
		t = ifstats;
		break;

	case 1:
		/*
		 * header stats
		 */
		opt = "-h";
		t = "SKIP Header Statistics";
		break;

	case 2:
		/*
		 * cryptor stats
		 */
		opt = "-c";
		t = "SKIP Algorithm Statistics";
		break;

	case 3:
		/*
		 * key stats
		 */
		opt = "-k";
		t = "SKIP Key Statistics";
		break;
	}

	if (es_path(SKIPSTAT) == NULL) {
		sprintf(s, "Cannot locate skipstat command");
		alert(Skiptool_base_window->key_status, s, B_TRUE);
		return;
	}

	if (es_path(SKIPSTAT_UI) == NULL) {
		sprintf(s, "Cannot locate skipstat_ui command");
		alert(Skiptool_base_window->key_status, s, B_TRUE);
		return;
	}

	strcpy(s, es_path(SKIPSTAT));

	argv[0] = es_path(SKIPSTAT_UI);
	argv[1] = "-display";
	argv[2] = XDisplayName(NULL);
	argv[3] = "-t";
	argv[4] = t;
	argv[5] = s;
	argv[6] = "-i";
	argv[7] = ifname;
	argv[8] = opt;
	argv[9] = NULL;

	if (getrlimit(RLIMIT_NOFILE, &rl) < 0) {
		return;
	}
	switch (pid = fork()) {

	case -1:
		perror("fork");
		return;

	case 0:	/* child */
		for (fd = 0; fd < rl.rlim_cur; fd ++) {
			(void) close(fd);
		}
		execvp(*argv, argv);
		if (errno == ENOENT) {
			fprintf(stderr, "%s: command not found.\n", *argv);
		} else {
			perror("execvp");
		}
		_exit(0);
	default:
		set_footer("Getting statistics...", B_FALSE);
		(void) notify_set_wait3_func(Skiptool_base_window->base_window,
				notify_default_wait3, pid);
	}
}

void
key_status()
{
	es_acl_t				*access;
	skip_param_t				*params;
	pid_t					pid;
	int					fd;
	char					v[4], r_nsid[4], s[STRSZ],
						*host, *argv[8];
	int					selected;
	struct rlimit				rl;

	selected = xv_get(Skiptool_base_window->hostlist,
				PANEL_LIST_FIRST_SELECTED);

	if (selected == -1) {
		return;
	}

	if (es_path(KEYSTAT) == NULL) {
		sprintf(s, "Cannot locate key status command (%s)", KEYSTAT);
		alert(Skiptool_base_window->key_status, s, B_TRUE);
		return;
	}

	access = (es_acl_t *) xv_get(Skiptool_base_window->hostlist,
				PANEL_LIST_CLIENT_DATA, selected);

	params = &access->params;

	host = hostname_of(params);
	sprintf(v, "%d", params->version);
	sprintf(r_nsid, "%d", params->r_nsid);
	(void) skip_keyid_to_s(&params->r_mkeyid, params->r_nsid, s);

	argv[0] = es_path(KEYSTAT);
	argv[1] = "-display";
	argv[2] = XDisplayName(NULL);
	argv[3] = host;
	argv[4] = v;
	argv[5] = r_nsid;
	argv[6] = s;
	argv[7] = NULL;

	if (getrlimit(RLIMIT_NOFILE, &rl) < 0) {
		return;
	}
	switch (pid = fork()) {

	case -1:
		perror("fork");
		return;

	case 0:	/* child */
		for (fd = 0; fd < rl.rlim_cur; fd ++) {
			(void) close(fd);
		}
		execvp(*argv, argv);
		if (errno == ENOENT) {
			fprintf(stderr, "%s: command not found.\n", *argv);
		} else {
			perror("execvp");
		}
		_exit(0);
	default:
		set_footer("Getting key status...", B_FALSE);
		(void) notify_set_wait3_func(Skiptool_base_window->base_window,
				notify_default_wait3, pid);
	}
}

boolean_t
is_double_click(int row)
{
	static struct timeval		then = { 0, 0 };
	static int			last_row = -1, last_nrows = 0;
	struct timeval			now;
	unsigned long			delta;
	int				nrows;

	nrows = xv_get(Skiptool_base_window->hostlist, PANEL_LIST_NROWS);
	if (nrows != last_nrows) {
		last_nrows = nrows;
		return (B_FALSE);
	}
	(void) gettimeofday(&now, NULL);
	delta = (now.tv_sec - then.tv_sec) * 1000000 +
					(now.tv_usec - then.tv_usec);
	then = now;

	if (row != last_row) {
		last_row = row;
		return (B_FALSE);
	}
	last_row = row;
	return (delta < 300*1000 ? B_TRUE : B_FALSE);
}

int
editor(int op, int row)
{
	skiptool_hostwin_objects		*win;
	es_acl_t				*access;

	access = (es_acl_t *) xv_get(Skiptool_base_window->hostlist,
                                	PANEL_LIST_CLIENT_DATA, row);
	if (access == NULL) {
		fprintf(stderr, "access == NULL in editor\n");
		exit(1);
	}

	switch (op) {

	case Select:
		if (is_double_click(row)) {
			if (access->win == NULL) {
				win = new_win();
				if (win == NULL) {
					return;
				}
				(void) xv_set(win->hostwin,
					WIN_CLIENT_DATA, access,
					NULL);
				access->win = win;
				populate_edit_win(access);
			}
			(void) xv_set(access->win->hostwin,
				FRAME_DONE_PROC, edit_win_done,
				XV_SHOW, TRUE,
				NULL);
		}
		(void) xv_set(Skiptool_base_window->key_status,
			PANEL_INACTIVE, !(access->params.version != SKIP_NONE),
			NULL);
		break;
	}
	return (XV_OK);
}

/* acl_empty()
 *
 * empty the current kernel acl list
 */
static void
acl_empty()
{
	es_destroy_systems(&skip_acl);
}

/* acl_add()
 *
 * Add an entry to the kernel acl list
 */
static void
acl_add(skip_param_t *params)
{
	es_add_system(&skip_acl, params, NULL);
}

/* acl_add_new()
 *
 * Merge new entries into the authorised systems list
 */
static void
acl_add_new()
{
	es_system_t	*entry;
	es_acl_t	*access;

	for (entry = skip_acl; entry; entry = entry->next) {

		access = (es_acl_t *) calloc(1, sizeof (*access));

		if (access == NULL) {
			alert(NULL, "Error adding system (out of memory)",
									B_TRUE);
			return;
		}
		access->params = entry->params;
		(void) xv_set(Skiptool_base_window->hostlist,
			PANEL_LIST_INSERT_DUPLICATE, FALSE,
			PANEL_LIST_INSERT, 0,
			PANEL_LIST_STRING, 0, hostname_of(&entry->params),
			PANEL_LIST_GLYPH, 0, glyph_of(&entry->params),
			PANEL_LIST_CLIENT_DATA, 0, access,
			NULL);
	}
}

/* acl_update_old()
 *
 * Update existing entries in the authorised systems list, removing
 * no longer existing entries.
 */
static void
acl_update_old()
{
	register int		n;
	register es_acl_t	*access;
	skip_param_t		params;

	n = xv_get(Skiptool_base_window->hostlist, PANEL_LIST_NROWS);
	for (; n ; n--) {
		access = (es_acl_t *) xv_get(Skiptool_base_window->hostlist,
			PANEL_LIST_CLIENT_DATA, n - 1);

		if (es_find_system(&skip_acl, &access->params.ip_addr,
								&params)) {
			if (memcmp((char *) &access->params, 
					(char *) &params,
					sizeof(params)) != 0) {
				/*
				 * update parameters from the kernel
			 	 */
				access->params = params;
				if (access->win) {
					populate_edit_win(access);
				}
			}
		} else {
			/*
			 * system has been deleted from kernel
			 */
			if (access->win) {
				xv_destroy_safe(access->win->hostwin);
				free((char *) access->win);
			}
			free((char *) access);
			(void) xv_set(Skiptool_base_window->hostlist,
				PANEL_LIST_DELETE, n - 1,
				NULL);
		}
	}
}

/*
 * create list of the current remote systems from the kernel
 */
void
acl_build()
{
	(void) xv_set(Skiptool_base_window->hostlist, XV_SHOW, FALSE, NULL);
	acl_empty();
	(void) skip_list_hosts(ifname, acl_add);
	acl_update_old();
	acl_add_new();
	(void) xv_set(Skiptool_base_window->hostlist, XV_SHOW, TRUE, NULL);
}

/* required_systems_done()
 *
 * Handle the closure of the required systems window
 */
/*ARGSUSED*/
void
required_systems_done(Frame subframe)
{
	access_update();

	(void) xv_set(Skiptool_required_win->required_win,
			XV_SHOW, FALSE,
			NULL);	
	xv_set(Skiptool_base_window->access_ctrl,
		PANEL_INACTIVE, FALSE,
		PANEL_VALUE, 1,
		NULL);
}

/* required_systems_add()
 *
 * Add contents of required systems list
 */
void
required_systems_add()
{
	int		nrows, is_selected;
	skip_param_t	params = { 0 };
	char		*s;

	nrows = xv_get(Skiptool_required_win->required_list, PANEL_LIST_NROWS);

	for (; nrows ; nrows--) {
		is_selected = (boolean_t)
			xv_get(Skiptool_required_win->required_list,
				PANEL_LIST_SELECTED, nrows - 1);
		if (is_selected) {
			s = (char *)xv_get(Skiptool_required_win->required_list,
				PANEL_LIST_STRING, nrows - 1);
			s = strtok(s, " ");
			if (s == NULL) {
				continue;
			}
			if (skip_host_to_addr(s, &params.ip_addr) < 0) {
				continue;
			}
			(void) skip_allow_host(ifname, &params);
		}
		(void) xv_set(Skiptool_base_window->hostlist,
				PANEL_LIST_DELETE, nrows - 1,
				NULL);
	}
	required_systems_done(Skiptool_required_win->required_win);
}

/* required_systems_build()
 *
 * Get required systems list
 */
int
required_systems_build()
{
	int		nrows;
	es_system_t	*syslist = NULL, *next_syslist;
	skip_param_t	params;
	char		host[STRSZ], label[STRSZ];
	Rect		base, this_win;

	if (xv_get(Skiptool_base_window->access_ctrl, PANEL_INACTIVE)) {
		return(0);
	}

	xv_set(Skiptool_base_window->access_ctrl,
			PANEL_INACTIVE, TRUE,
			NULL);

	frame_get_rect(Skiptool_base_window->base_window, &base);
	frame_get_rect(Skiptool_required_win->required_win, &this_win);
	this_win.r_left = base.r_left + 150;
	this_win.r_top = base.r_top + 200;
	frame_set_rect(Skiptool_required_win->required_win, &this_win);
	
	set_footer("Checking for required systems...", B_FALSE);

	notify_dispatch();

	es_get_broadcast_addr(&syslist);
	es_get_nis_servers(&syslist);
	es_get_dns_servers(&syslist);
	es_get_nfs_servers(&syslist);
	es_get_x11_display(&syslist);
	es_get_local_addr(&syslist);

	if (syslist == NULL) {
		xv_set(Skiptool_base_window->access_ctrl,
					PANEL_INACTIVE, FALSE,
					NULL);
		return(0);
	}

	/*
	 * syslist holds the list of required systems.
	 */
	(void) xv_set(Skiptool_required_win->required_list,
		XV_SHOW, FALSE, NULL);	
	
	nrows = xv_get(Skiptool_required_win->required_list, PANEL_LIST_NROWS);

	for (; nrows ; nrows--) {
		(void) xv_set(Skiptool_required_win->required_list,
				PANEL_LIST_DELETE, nrows - 1,
				NULL);
	}

	for (; syslist; syslist = next_syslist) {
		next_syslist = syslist->next;
		
		params.ip_addr = syslist->params.ip_addr;
		if (skip_get_host(ifname, &params) != 0) {
			/*
			 * host not present, need to add it
			 */
			(void) skip_addr_to_host(&params.ip_addr, host);
			sprintf(label, "%s (%s)", host, syslist->desc);
			(void) xv_set(Skiptool_required_win->required_list,
				PANEL_LIST_INSERT, 0,
				PANEL_LIST_SELECT, 0, TRUE,
				PANEL_LIST_STRING, 0, label,
				NULL);
		} 
		free((char *) syslist);
	}

	(void) xv_set(Skiptool_required_win->required_list,
		XV_SHOW, TRUE,
		NULL);	

	nrows = xv_get(Skiptool_required_win->required_list, PANEL_LIST_NROWS);
	if (nrows > 0) {
		if (nrows == 1) {
			sprintf(label, "1 address currently in use.");
		} else {
			sprintf(label, "%d addresses currently in use.", nrows);
		}
		(void) xv_set(Skiptool_required_win->required_message,
			PANEL_LABEL_STRING, label,
			NULL);
		window_bell(Skiptool_required_win->required_win);
		(void) xv_set(Skiptool_required_win->required_win,
			XV_SHOW, TRUE,
			FRAME_DONE_PROC, required_systems_done,
			NULL);	
		return (EBUSY);
	}
	xv_set(Skiptool_base_window->access_ctrl,
					PANEL_INACTIVE, FALSE,
					NULL);
	return (0);
}

/*
 * retrieve current SKIP key management parameters and update the GUI with them
 */
void
get_key_params()
{
	int		var;
	int		rc;

	rc = skip_var_get(ifname, "skip_key_max_bytes", &var);

	if (rc < 0) {
		return;
	}

	(void) xv_set(Skiptool_key_params_win->skip_key_max_bytes,
			PANEL_VALUE,
			var/1024,
			NULL
	);

	rc = skip_var_get(ifname, "skip_key_max_idle", &var);

	if (rc < 0) {
		return;
	}

	(void) xv_set(Skiptool_key_params_win->skip_key_max_idle,
			PANEL_VALUE,
			var,
			NULL
	);
}

/*
 * set current SKIP key managment parameters from the GUI values
 */
void
set_key_params()
{
	int		var;
	int		rc;

	var = xv_get(Skiptool_key_params_win->skip_key_max_bytes, PANEL_VALUE);
	rc = skip_var_set(ifname, "skip_key_max_bytes", var * 1024);

	if (rc < 0) {
		return;
	}

	var = xv_get(Skiptool_key_params_win->skip_key_max_idle, PANEL_VALUE);
	rc = skip_var_set(ifname, "skip_key_max_idle", var);

	if (rc < 0) {
		return;
	}

}

/*
 * set current SKIP key managment parameters to default
 */
void
set_key_defaults()
{
	(void) xv_set(Skiptool_key_params_win->skip_key_max_bytes,
			PANEL_VALUE, 512,
			NULL);
	(void) xv_set(Skiptool_key_params_win->skip_key_max_idle,
			PANEL_VALUE, 30,
			NULL);
	set_key_params();
}

/*
 * insert skip_es kernel module between IP and driver if not already present
 */
static void
install_skip_es_mod()
{
	int		mode = 1; /*Secure*/
	char		msg[STRSZ];

	sprintf(msg, "%s: saving interface state", ifname);
	set_footer(msg, B_FALSE);
	skip_if_state(ifname, Save);

	sprintf(msg, "%s: bringing interface down", ifname);
	set_footer(msg, B_FALSE);
	skip_if_unplumb(ifname);

	sprintf(msg, "%s: starting secure mode", ifname);
	set_footer(msg, B_FALSE);

	if (skip_if_plumb(ifname, 1) < 0) {
		sprintf(msg, "%s: secure mode FAILED - "
			"trying unsecure mode", ifname);
		set_footer(msg, B_FALSE);
		if (skip_if_plumb(ifname, 0) < 0) {
			sprintf(msg,  "%s: unsecure mode reset FAILED",
				ifname);
			set_footer(msg, B_FALSE);
			return;
		}
		mode = 0;
	}
	sprintf(msg, "%s: restoring interface state", ifname);
	set_footer(msg, B_FALSE);
	skip_if_state(ifname, Restore);
	skip_if_set_flags(ifname, mode);
	show_mode();
}

/*
 * load rasters
 */
static void
load_images()
{
	extern int				systems_list_notify();
	Xv_Screen				screen;
	Display					*display;
	Panel					panel;
	Icon					icon;
	int					screen_no, x, y, w, h, d;

#include "unknown.xbm"
#include "none.xbm"
#include "icon_v1.xbm"
#include "icon_v2.xbm"
#include "skiptool_small.xbm"

	display = (Display *) xv_get(Skiptool_base_window->base_window,
								XV_DISPLAY);
	screen = xv_get(Skiptool_base_window->base_window, XV_SCREEN);
	screen_no = xv_get(screen, SCREEN_NUMBER);

	es_display_depth = DefaultDepth(display, screen_no);

	/*
	 * XXX re-create the hostlist with the right size row height...
	 */
 	panel = xv_get(Skiptool_base_window->hostlist, PANEL_PARENT_PANEL);
	w = xv_get(Skiptool_base_window->hostlist, PANEL_LIST_WIDTH);
	h = xv_get(Skiptool_base_window->hostlist, PANEL_LIST_DISPLAY_ROWS);
	x = xv_get(Skiptool_base_window->hostlist, XV_X);
	y = xv_get(Skiptool_base_window->hostlist, XV_Y);
	xv_destroy_safe(Skiptool_base_window->hostlist);
	Skiptool_base_window->hostlist = xv_create(panel, PANEL_LIST,
		XV_KEY_DATA, INSTANCE, Skiptool_base_window,
		XV_X, x,
		XV_Y, y,
		PANEL_LIST_WIDTH, w,
		PANEL_LIST_DISPLAY_ROWS, h,
		PANEL_LABEL_STRING, "Authorized Systems",
		PANEL_LAYOUT, PANEL_VERTICAL,
		PANEL_READ_ONLY, TRUE,
		PANEL_CHOOSE_ONE, TRUE,
		PANEL_CHOOSE_NONE, FALSE,
		PANEL_NOTIFY_PROC, systems_list_notify,
		PANEL_LIST_ROW_HEIGHT, 20,
		NULL);

	if ((unknown = load_raster(SKIP_UNKNOWN_ICON, &w, &h, &d)) == NULL) {
		unknown = xv_create(XV_NULL, SERVER_IMAGE,
		SERVER_IMAGE_DEPTH, 1,
		SERVER_IMAGE_X_BITS, unknown_bits,
		XV_WIDTH, unknown_width,
		XV_HEIGHT, unknown_height,
		NULL);
	}
	if ((none = load_raster(SKIP_NONE_ICON, &w, &h, &d)) == NULL) {
		none = xv_create(XV_NULL, SERVER_IMAGE,
		SERVER_IMAGE_DEPTH, 1,
		SERVER_IMAGE_X_BITS, none_bits,
		XV_WIDTH, none_width,
		XV_HEIGHT, none_height,
		NULL);
	}
	if ((icon_v1 = load_raster(SKIP_ICON_V1, &w, &h, &d)) == NULL) {
		icon_v1 = xv_create(XV_NULL, SERVER_IMAGE,
		SERVER_IMAGE_DEPTH, 1,
		SERVER_IMAGE_X_BITS, icon_v1_bits,
		XV_WIDTH, icon_v1_width,
		XV_HEIGHT, icon_v2_height,
		NULL);
	}
	if ((icon_v2 = load_raster(SKIP_ICON_V2, &w, &h, &d)) == NULL) {
		icon_v2 = xv_create(XV_NULL, SERVER_IMAGE,
		SERVER_IMAGE_DEPTH, 1,
		SERVER_IMAGE_X_BITS, icon_v2_bits,
		XV_WIDTH, icon_v2_width,
		XV_HEIGHT, icon_v2_height,
		NULL);
	}
	skip = load_raster(SKIP_SMALL_ICON, &w, &h, &d);
	if (skip) {
		(void) xv_set(Skiptool_base_window->skip_icon,
			PANEL_LABEL_IMAGE,
			skip,
			NULL);
		(void) xv_set(Skiptool_hostwin->skip_icon,
			PANEL_LABEL_IMAGE,
			skip,
			NULL);
		(void) xv_set(Skiptool_required_win->skip_icon,
			PANEL_LABEL_IMAGE,
			skip,
			NULL);
		(void) xv_set(Skiptool_about_win->skip_icon,
			PANEL_LABEL_IMAGE,
			skip,
			NULL);
		(void) xv_set(Skiptool_about_win->message8,
			PANEL_LABEL_STRING,
			"skiptool v" SKIP_RELEASE,
			NULL);
		icon = (Icon) xv_create(XV_NULL, ICON,
			ICON_IMAGE, skip,
			ICON_WIDTH, w,
			ICON_HEIGHT, h + 20,
			ICON_TRANSPARENT_LABEL, "Solaris SKIP",
			NULL);
		(void) xv_set(Skiptool_base_window->base_window,
			FRAME_ICON,
			icon,
			NULL);
	}
}

/*
 * Initialise skiptool
 */
void
init(int argc, char *argv[])
{
	char	hostname[STRSZ], msg[STRSZ];

	if (argc > 2) {
		usage(argv[0]);
	}

	progname = argv[0];

	load_images();

	if (argc == 1) {
		strcpy(ifname, skip_default_if());
	} else {
		strcpy(ifname, argv[1]);
	}

	if (skip_pre_checks(ifname)) {
		if (errno == EACCES) {
			fprintf(stderr,
				"you must be root to run this program\n");
		} else {
			fprintf(stderr, "%s\n", skip_errmsg);
			fprintf(stderr,
				"check that SKIP is correctly installed\n");
		}
		exit(1);
	}

	if (skip_var_init(argv[0])) {
		exit(1);
	}

	if (!skip_if_module_present(ifname)) {
		install_skip_es_mod();
		skip_access_ctrl_off(ifname);
	}

#ifdef SYSV
	sysinfo(SI_HOSTNAME, hostname, STRSZ);
#else
	gethostname(hostname, STRSZ);
#endif
	sprintf(msg, "SKIP for Solaris (%s)", hostname);
	(void) xv_set(Skiptool_base_window->base_window,
			XV_LABEL, msg,
			NULL);

	freopen("/tmp/skiptool.log", "w", stdout);
	freopen("/tmp/skiptool.log", "a", stderr);
	setbuf(stdout, NULL);
	setbuf(stderr, NULL);

	get_key_params();
	build_alg_menus(Skiptool_hostwin);
	set_alg_selector(Skiptool_hostwin->kij_alg, SKIP_DEFAULT_KIJ_ALG);
	set_alg_selector(Skiptool_hostwin->kp_alg, SKIP_DEFAULT_KP_ALG);
	build_nsid_menu(Skiptool_hostwin);
	selectors_available(Skiptool_hostwin);
	acl_build();
	show_mode();
	sprintf(msg, "%s", ifname);
	(void) xv_set(Skiptool_base_window->base_window,
		FRAME_RIGHT_FOOTER, msg,
		XV_SHOW, TRUE,
		NULL);
	notify_dispatch();
	tick();
}
