/*
 * dsatest.c - DSA key generator and test driver.
 *
 * Copyright (c) 1995  Colin Plumb.  All rights reserved.
 * For licensing and other legal details, see the file legal.c.
 *
 * This generates DSA primes using a (hopefully) clearly
 * defined algorithm, based on David Kravitz's "kosherizer".
 * It is not, however, identical.
 */
#if HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#if !NO_STRING_H
#include <string.h>
#elif HAVE_STRINGS_H
#include <strings.h>
#endif

#include "bn.h"
#include "prime.h"

#include "cputime.h"
#include "sha.h"

#define BNDEBUG 1

#if BNDEBUG
#include "bnprint.h"
#define bndPut(prompt, bn) bnPrint(stdout, prompt, bn, "\n")
#define bndPrintf printf
#else
#define bndPut(prompt, bn) ((void)(prompt),(void)(bn))
#define bndPrintf (void)
#endif

#include "kludge.h"

/*
 * Generate a bignum of a specified length, with the given
 * high and low 8 bits. "High" is merged into the high 8 bits of the
 * number.  For example, set it to 0x80 to ensure that the number is
 * exactly "bits" bits long (i.e. 2^(bits-1) <= bn < 2^bits).
 * "Low" is merged into the low 8 bits.  For example, set it to
 * 1 to ensure that you generate an odd number.
 *
 * Then XOR the result into the input bignum.  This is to
 * accomodate the kosherizer in all its generality.
 *
 * The bignum is generated using the given seed string.  The
 * technique is from David Kravitz (of the NSA)'s "kosherizer".
 * The string is hashed, and that (with the low bit forced to 1)
 * is used for the low 160 bits of the number.  Then the string,
 * considered as a big-endian array of bytes, is incremented
 * and the incremented value is hashed to produce the next most
 * significant 160 bits, and so on.  The increment is performed
 * modulo the size of the seed string.
 *
 * The seed is returned incremented so that it may be used to generate
 * subsequent numbers.
 *
 * The most and least significant 8 bits of the returned number are forced
 * to the values passed in "high" and "low", respectively.  Typically,
 * high would be set to 0x80 to force the most significant bit to 1.
 */
static int
genRandBn(struct BigNum *bn, unsigned bits, unsigned char high,
unsigned char low, unsigned char *seed, unsigned len)
{
	unsigned char buf1[SHA_DIGESTSIZE];
	unsigned char buf2[SHA_DIGESTSIZE];
	unsigned bytes = (bits+7)/8;
	unsigned l = 0;	/* Current position */
	unsigned i;
	struct SHAContext sha;

	if (!bits)
		return 0;

	/* Generate the first bunch of hashed data */
	shaInit(&sha);
	shaUpdate(&sha, seed, len);
	shaFinal(&sha, buf1);
	/* Increment the seed, ignoring carry out. */
	i = len;
	while (i-- && (++seed[i] & 255) == 0)
		;
	/* XOR in the existing bytes */
	bnExtractBigBytes(bn, buf2, l, SHA_DIGESTSIZE);
	for (i = 0; i < SHA_DIGESTSIZE; i++)
		buf1[i] ^= buf2[i];
	buf1[SHA_DIGESTSIZE-1] |= low;

	while (bytes > SHA_DIGESTSIZE) {
		bytes -= SHA_DIGESTSIZE;
		/* Merge in low half of high bits, if necessary */
		if (bytes == 1 && (bits & 7))
			buf1[0] |= high << (bits & 7);
		if (bnInsertBigBytes(bn, buf1, l, SHA_DIGESTSIZE) < 0)
			return -1;
		l += SHA_DIGESTSIZE;

		/* Compute the next hash we need */
		shaInit(&sha);
		shaUpdate(&sha, seed, len);
		shaFinal(&sha, buf1);
		/* Increment the seed, ignoring carry out. */
		i = len;
		while (i-- && (++seed[i] & 255) == 0)
			;
		/* XOR in the existing bytes */
		bnExtractBigBytes(bn, buf2, l, SHA_DIGESTSIZE);
		for (i = 0; i < SHA_DIGESTSIZE; i++)
			buf1[i] ^= buf2[i];
	}

	/* Do the final "bytes"-long section, using the tail bytes in buf1 */
	/* Mask off excess high bits */
	buf1[SHA_DIGESTSIZE-bytes] &= 255 >> (-bits & 7);
	/* Merge in specified high bits */
	buf1[SHA_DIGESTSIZE-bytes] |= high >> (-bits & 7);
	if (bytes > 1 && (bits & 7))
		buf1[SHA_DIGESTSIZE-bytes+1] |= high << (bits & 7);
	/* Merge in the appropriate bytes of the buffer */
	if (bnInsertBigBytes(bn, buf1+SHA_DIGESTSIZE-bytes, l, bytes) < 0)
		return -1;

	return 0;
}

struct Progress {
	FILE *f;
	unsigned column;
	unsigned wrap;
};

static int
genProgress(void *arg, int c)
{
	struct Progress *p = arg;
	if (++p->column > p->wrap) {
		putc('\n', p->f);
		p->column = 1;
	}
	putc(c, p->f);
	fflush(p->f);
	return 0;
}

static int
dsaGen(struct BigNum *p, unsigned pbits, struct BigNum *q, unsigned qbits,
	struct BigNum *g, struct BigNum *x, struct BigNum *y,
	unsigned char *seed, unsigned len, FILE *f)
{
	struct BigNum h, e;
	int i;
#if CLOCK_AVAIL
	timetype start, stop;
	unsigned long s;
#endif
	struct Progress progress;

	if (f)
		fprintf(f,
		   "Generating a DSA key pair with %u-bit p and %u-bit q,\n"
	           "seed = \"%.*s\"\n", pbits, qbits, (int)len, (char *)seed);
	progress.f = f;
	progress.column = 0;
	progress.wrap = 78;

#if CLOCK_AVAIL
	gettime(&start);
#endif

	/*
	 * Choose a random starting place for q
	 * Starting place is SHA(seed) XOR SHA(seed+1),
	 * With the high *8* bits set to 0.
	 */
	(void)bnSetQ(q, 0);
	if (genRandBn(q, qbits, 0xFF, 0, seed, len) < 0)
		return -1;
	if (genRandBn(q, qbits, (unsigned char)0xFF, (unsigned char)1, seed, len) < 0)
		return -1;
	/* And search for a prime */
	i = primeGen(q, (unsigned (*)(unsigned))0, f ? genProgress : 0,
	             (void *)&progress, 0);
	if (i < 0)
		return -1;
	
	/* ...and for p */
	(void)bnSetQ(p, 0);
	if (genRandBn(p, pbits, 0xC0, 1, seed, len) < 0)
		return -1;

	/* Temporarily double q */
	if (bnLShift(q, 1) < 0)
		return -1;

	bnBegin(&h);
	bnBegin(&e);

	/* Set p = p - (p mod q) + 1, i.e. congruent to 1 mod 2*q */
	if (bnMod(&e, p, q) < 0)
		goto failed;
	if (bnSub(p, &e) < 0 || bnAddQ(p, 1) < 0)
		goto failed;

	if (f)
		genProgress(&progress, ' ');

	/* And search for a prime */
	i = primeGenStrong(p, q, f ? genProgress : 0, (void *)&progress);
	if (i < 0)
		return -1;

	/* Reduce q again */
	bnRShift(q, 1);

	/* Now hunt for a suitable g - first, find (p-1)/q */
	if (bnDivMod(&e, &h, p, q) < 0)
		goto failed;
	/* e is now the exponent (p-1)/q, and h is the remainder (one!) */
	if (bnBits(&h) != 1) {
		bndPut("Huh? p % q = ", &h);
		goto failed;
	}

	if (f)
		genProgress(&progress, ' ');

	/* Search for a suitable h */
	if (bnSetQ(&h, 2) < 0 || bnTwoExpMod(g, &e, p) < 0)
		goto failed;
	i++;
	while (bnBits(g) < 2) {
		if (f)
			genProgress(&progress, '.');
		if (bnAddQ(&h, 1) < 0 || bnExpMod(g, &h, &e, p) < 0)
			goto failed;
		i++;
	}
	if (f)
		genProgress(&progress, '*');
#if CLOCK_AVAIL
	gettime(&stop);
#endif

	/*
	 * Now pick the secret, x.  Choose it a bit larger than q and do
	 * modular reduction to make it uniformly distributed.
	 */
	bnSetQ(x, 0);
	/* XXX SECURITY ALERT Replace with a real RNG! SECURITY ALERT XXX */
	if (genRandBn(x, qbits+8, 0, 0, seed, len) < 0)
		goto failed;
	if (bnMod(x, x, q) < 0 || bnExpMod(y, g, x, p) < 0)
		goto failed;
	i++;
	if (f)
		putc('\n', f);

	printf("%d modular exponentiations performed.\n", i);

#if CLOCK_AVAIL
	subtime(stop, start);
	s = sec(stop);
	bndPrintf("%u/%u-bit time = %lu.%03u sec.", pbits, qbits,
	          s, msec(stop));
	if (s > 60) {
		putchar(' ');
		putchar('(');
		if (s > 3600)
			printf("%u:%02u", (unsigned)(s/3600),
			       (unsigned)(s/60%60));
		else
			printf("%u", (unsigned)(s/60));
		printf(":%02u)", (unsigned)(s%60));
	}
	putchar('\n');
#endif

	bndPut("q = ", q);
	bndPut("p = ", p);
	bndPut("h = ", &h);
	bndPut("g = ", g);
	bndPut("x = ", x);
	bndPut("y = ", y);

	bnEnd(&h);
	bnEnd(&e);

	return 0;

failed:
	bnEnd(&h);
	bnEnd(&e);
	return -1;
}

static int
dsaSign(struct BigNum const *p, struct BigNum const *q, struct BigNum const *g,
	struct BigNum const *x, struct BigNum const *y,
	struct BigNum const *hash, struct BigNum const *k,
	struct BigNum *r, struct BigNum *s)
{
	int retval = -1;
	struct BigNum t;

	(void)y;

	bnBegin(&t);
	/* Make the signature...  first the precomputation */

	/* Compute r = (g^k mod p) mod q */
	if (bnExpMod(r, g, k, p) < 0 || bnMod(r, r, q) < 0)
		goto failed;

	/* Compute s = k^-1 * (hash + x*r) mod q */
	if (bnInv(&t, k, q) < 0)
		goto failed;
 	if (bnMul(s, x, r) < 0 || bnMod(s, s, q) < 0)
		goto failed;

	/* End of precomputation.  Steps after this require the hash. */

	if (bnAdd(s, hash) < 0)
		goto failed;
	if (bnCmp(s, q) > 0 && bnSub(s, q) < 0)
		goto failed;
	if (bnMul(s, s, &t) < 0 || bnMod(s, s, q) < 0)
		goto failed;
	/* Okay, r and s are the signature! */

	retval = 0;

failed:
	bnEnd(&t);
	return retval;
}

/*
 * Returns 1 for a good signature, 0 for bad, and -1 on error.
 */
static int
dsaVerify(struct BigNum const *p, struct BigNum const *q,
          struct BigNum const *g, struct BigNum const *y,
          struct BigNum const *r, struct BigNum const *s,
          struct BigNum const *hash)
{
	struct BigNum w, u1, u2;
	int retval = -1;

	bnBegin(&w);
	bnBegin(&u1);
	bnBegin(&u2);

	if (bnInv(&w, s, q) < 0)
		goto failed;

	if (bnMul(&u1, hash, &w) < 0 || bnMod(&u1, &u1, q) < 0)
		goto failed;
	if (bnMul(&u2, r, &w) < 0 || bnMod(&u2, &u2, q) < 0)
		goto failed;

	/* Now for the expensive part... */

	if (bnDoubleExpMod(&w, g, &u1, y, &u2, p) < 0)
		goto failed;
	if (bnMod(&w, &w, q) < 0)
		goto failed;
	retval = (bnCmp(r, &w) == 0);
failed:
	bnEnd(&u2);
	bnEnd(&u1);
	bnEnd(&w);
	return retval;
}

static int
dsaTest(struct BigNum const *p, struct BigNum const *q, struct BigNum const *g,
	struct BigNum const *x, struct BigNum const *y)
{
	struct BigNum hash, r, s, k;
	unsigned bits;
	int i;
	int verified;
	int retval = -1;
	unsigned char foo[4], bar[4];
#if CLOCK_AVAIL
	timetype start, stop;
	unsigned long cursec, sigsec = 0, versec = 0;
	unsigned curms, sigms = 0, verms = 0;
#endif

	bnBegin(&hash);
	bnBegin(&r);
	bnBegin(&s);
	bnBegin(&k);

	bits = bnBits(q);
	strcpy((char *)foo, "foo");
	strcpy((char *)bar, "bar");

	bndPrintf("\tSigning \tVerifying\tStatus\n");
	for (i = 0; i < 10; i++) {
		/* Pick a random hash, the right length. */
		(void)bnSetQ(&k, 0);
		if (genRandBn(&hash, bits, 0, 0, foo, 4) < 0)
			goto failed;

		/* Make the signature... */

		/*
		 * XXX      SECURITY ALERT      XXX
		 * XXX Replace with a real RNG! XXX
		 * XXX      SECURITY ALERT      XXX
		 */
		(void)bnSetQ(&k, 0);
		if (genRandBn(&k, bnBits(q)+8, 0, 0, bar, 4) < 0)
			goto failed;
		/* Reduce k to the correct range */
		if (bnMod(&k, &k, q) < 0)
			goto failed;
#if CLOCK_AVAIL
		gettime(&start);
#endif
		if (dsaSign(p, q, g, x, y, &hash, &k, &r, &s) < 0)
			goto failed;
#if CLOCK_AVAIL
		gettime(&stop);
		subtime(stop, start);
		sigsec += cursec = sec(stop);
		sigms += curms = msec(stop);
		bndPrintf("\t%lu.%03u\t\t", cursec, curms);
#else
		bndPrintf("\t*\t\t");
#endif
		fflush(stdout);
		/* Okay, r and s are the signature!  Now, verify it.  */

#if CLOCK_AVAIL
		gettime(&start);
#endif
		verified = dsaVerify(p, q, g, y, &r, &s, &hash);
#if CLOCK_AVAIL
		gettime(&stop);
		subtime(stop, start);
		versec += cursec = sec(stop);
		verms += curms = msec(stop);
		bndPrintf("%lu.%03u\t\t", cursec, curms);
#else
		bndPrintf("*\t\t");
#endif
		if (verified > 0) {
			printf("Test successful.\n");
		} else if (verified == 0) {
			printf("\aSignature did NOT check!.\n");
			bndPut("hash = ", &hash);
			bndPut("k = ", &k);
			bndPut("r = ", &r);
			bndPut("s = ", &s);
			getchar();
		} else {
			printf("\a** Error while verifying");
			bndPut("hash = ", &hash);
			bndPut("k = ", &k);
			bndPut("r = ", &r);
			bndPut("s = ", &s);
			getchar();
			goto failed;
		}
	}
#if CLOCK_AVAIL
	sigms += 1000 * (sigsec % i);
	sigsec /= i;
	sigms /= i;
	sigsec += sigms / 1000;
	sigms %= 1000;
	verms += 1000 * (sigsec % i);
	versec /= i;
	verms /= i;
	versec += sigms / 1000;
	verms %= 1000;
	bndPrintf("\t%lu.%03u\t\t%lu.%03u\t\tAVERAGE\n",
	          sigsec, sigms, versec, verms);
#endif
	/* Success */
	retval = 0;

failed:
	bnEnd(&k);
	bnEnd(&s);
	bnEnd(&r);
	bnEnd(&hash);

	return retval;
}

/* Copy the command line to the buffer. */
static unsigned
copy(unsigned char *buf, int argc, char **argv)
{
	unsigned pos, len;
	
	pos = 0;
	while (--argc) {
		len = strlen(*++argv);
		memcpy(buf, *argv, len);
		buf += len;
		pos += len;
		if (argc > 1) {
			*buf++ = ' ';
			pos++;
		}
	}
	return pos;
}

int
main(int argc, char **argv)
{
	unsigned len;
	struct BigNum p, q, g, x, y;
	unsigned char buf[1024];

	if (argc < 2) {
		fprintf(stderr, "Usage: %s <seed>\n", argv[0]);
		fputs("\
<seed> should be a a string of bytes to be hashed to seed the prime\n\
generator.  Note that unquoted whitespace between words will be counted\n\
as a single space.  To include multiple spaces, quote them.\n", stderr);
		return 1;
	}

	bnInit();
	bnBegin(&p);
	bnBegin(&q);
	bnBegin(&g);
	bnBegin(&x);
	bnBegin(&y);

	len = copy(buf, argc, argv);
	dsaGen(&p, 512, &q, 160, &g, &x, &y, buf, len, stdout);
	dsaTest(&p, &q, &g, &x, &y);

	len = copy(buf, argc, argv);
	dsaGen(&p, 768, &q, 160, &g, &x, &y, buf, len, stdout);
	dsaTest(&p, &q, &g, &x, &y);

	len = copy(buf, argc, argv);
	dsaGen(&p, 1024, &q, 160, &g, &x, &y, buf, len, stdout);
	dsaTest(&p, &q, &g, &x, &y);

	bnEnd(&y);
	bnEnd(&x);
	bnEnd(&g);
	bnEnd(&q);
	bnEnd(&p);

	return 0;
}
