/*
 *
 * $Source: /filesv/usr/local/proj/sphinx/spx2/src/lib/cfile/RCS/ver_ctx_util.c,v $
 *
 *
 *  MODULE NAME:    ver_ctx_util.c
 *
 *
 *  AUTHORS:
 *
 *	K. Alagappan
 *
 */


/*
 * COPYRIGHT (C) 1992 DIGITAL EQUIPMENT CORPORATION
 * ALL RIGHTS RESERVED
 *
 * "Digital Equipment Corporation authorizes the reproduction,
 * distribution and modification of this software subject to the following
 * restrictions:
 * 
 * 1.  Any partial or whole copy of this software, or any modification
 * thereof, must include this copyright notice in its entirety.
 *
 * 2.  This software is supplied "as is" with no warranty of any kind,
 * expressed or implied, for any purpose, including any warranty of fitness 
 * or merchantibility.  DIGITAL assumes no responsibility for the use or
 * reliability of this software, nor promises to provide any form of 
 * support for it on any basis.
 *
 * 3.  Distribution of this software is authorized only if no profit or
 * remuneration of any kind is received in exchange for such distribution. 
 * 
 * 4.  This software and all application programs are to be used only for
 * non-commercial purposes. However, media costs associated with the
 * distribution of the software or application programs may be recovered.
 *
 */


#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <syslog.h>
#include <sys/time.h>
#include "BigNum.h"
#include "BigRSA.h"
#include "cdc.h"
#include "cdc_db.h"
#include "random.h"
#include "spxapi_defs.h"

#ifdef sun
#define strncasecmp    strncmp_support
#endif

#define INT_STR  20
#define TOO_BIG -1
#define VF_LCK_RETRY ((unsigned)2)	/* seconds to sleep before
					 * retry if ticket file is
					 * locked */
extern  errno;
extern int cdc_debug;

/*
 * fd must be initialized to something that won't ever occur as a real
 * file descriptor. Since open(2) returns only non-negative numbers as
 * valid file descriptors, and ver_ctx_init always stuffs the return value
 * from open in here even if it is an error flag, we must
 * 	a. Initialize fd to a negative number, to indicate that it is
 * 	   not initially valid.
 *	b. When checking for a valid fd, assume that negative values
 *	   are invalid (ie. when deciding whether ver_ctx_init has been
 *	   called.)
 *	c. In ver_ctx_close, be sure it gets reinitialized to a negative
 *	   number. 
 */
static  fd = -1;
static  keypos;                         /* key position within file */
static	curpos = 0;			/* Position in tfbfr */
static	lastpos = 0;			/* End of tfbfr */
static	char tfbfr[BUFSIZ];		/* Buffer for ticket data */

static ver_ctx_gets(), ver_ctx_read();

/*
 * This file contains routines for manipulating the ticket cache file.
 *
 * The principal's ticket file is in the following format:
 *
 *      localname; uid;
 *      CONTEXT_1
 *      CONTEXT_2
 *      ...
 *      CONTEXT_n
 *      EOF
 *
 *      Where "CONTEXT_X" consists of the following fixed-length
 *      fields from the authtoken structure (see "cdc.h"):
 *
 *      X_name; X_uid; X_deskey; X_pubRSAKey; X_start; X_end;
 *
 */

/*
 * ver_ctx_init() should be called before the other ticket file routines.
 * It takes the name of the ticket file to use, "ver_ctx_name", and a
 * read/write flag "rw" as arguments. 
 *
 * It tries to open the ticket file, checks the mode, and if everything
 * is okay, locks the file.  If it's opened for reading, the lock is
 * shared.  If it's opened for writing, the lock is exclusive. 
 *
 * Returns ASUCCESS if all went well, otherwise one of the following: 
 *
 * NO_TKT_FIL   - file wasn't there
 * TKT_FIL_ACC  - file was in wrong mode, etc.
 * TKT_FIL_LCK  - couldn't lock the file, even after a retry
 */

ver_ctx_init(vf_name, rw)
    char   *vf_name;
{
    int     wflag;
    short   me, getuid();
    struct stat stat_buf;

    switch (rw) {
    case R_TKT_FIL:
	wflag = 0;
	break;
    case W_TKT_FIL:
	wflag = 1;
	break;
    default:
	if (cdc_debug) fprintf(stderr, "ver_ctx_init: illegal parameter\n");
	return TKT_FIL_ACC;
    }
    if (lstat(vf_name, &stat_buf) < 0)
	switch (errno) {
	case ENOENT:
	    return NO_TKT_FIL;
	default:
	    return TKT_FIL_ACC;
	}
    me = getuid();
    if ((stat_buf.st_uid != me && me != 0) ||
	((stat_buf.st_mode & S_IFMT) != S_IFREG))
	return TKT_FIL_ACC;

    /*
     * If "wflag" is set, open the ticket file in append-writeonly mode
     * and lock the ticket file in exclusive mode.  If unable to lock
     * the file, sleep and try again.  If we fail again, return with the
     * proper error message. 
     */

    curpos = sizeof(tfbfr);
    keypos = 0;

    if (wflag) {
	fd = open(vf_name, O_RDWR, 0600);
	if (fd < 0) {
	    return TKT_FIL_ACC;
	}
	if (flock(fd, LOCK_EX | LOCK_NB) < 0) {
	    sleep(VF_LCK_RETRY);
	    if (flock(fd, LOCK_EX | LOCK_NB) < 0) {
		(void) close(fd);
		fd = -1;
		return TKT_FIL_LCK;
	    }
	}
	return ASUCCESS;
    }
    /*
     * Otherwise "wflag" is not set and the ticket file should be opened
     * for read-only operations and locked for shared access. 
     */

    fd = open(vf_name, O_RDONLY, 0600);
    if (fd < 0) {
	return TKT_FIL_ACC;
    }
    if (flock(fd, LOCK_SH | LOCK_NB) < 0) {
	sleep(VF_LCK_RETRY);
	if (flock(fd, LOCK_SH | LOCK_NB) < 0) {
	    (void) close(fd);
	    fd = -1;
	    return TKT_FIL_LCK;
	}
    }
    return ASUCCESS;
}

/*
 * ver_ctx_get_pname() reads the principal's name from the ticket file. It
 * should only be called after ver_ctx_init() has been called.  The
 * principal's name is filled into the "p" parameter.  If all goes well,
 * ASUCCESS is returned.  If ver_ctx_init() wasn't called, TKT_FIL_INI is
 * returned.  If the name was null, or EOF was encountered, or the name
 * was longer than ANAME_SZ, TKT_FIL_FMT is returned. 
 */

ver_ctx_get_pname(p)
    char   *p;
{
    if (fd < 0) {
	if (cdc_debug)
	    fprintf(stderr, "ver_ctx_get_pname called before ver_ctx_init.\n");
	return TKT_FIL_INI;
    }
    if (ver_ctx_gets(p, ANAME_SZ) < 2)	/* can't be just a null */
	return TKT_FIL_FMT;
    return ASUCCESS;
}

/*
 * ver_ctx_close() closes the ticket file and sets "fd" to -1. If "fd" is
 * not a valid file descriptor, it just returns.  It also clears the
 * buffer used to read tickets.
 *
 * The return value is not defined.
 */

ver_ctx_close()
{
    if (!(fd < 0)) {
	(void) flock(fd, LOCK_UN);
	(void) close(fd);
	fd = -1;		/* see declaration of fd above */
    }
    bzero(tfbfr, sizeof(tfbfr));
}

/*
 * ver_ctx_gets() is an internal routine.  It takes a string "s" and a count
 * "n", and reads from the file until either it has read "n" characters,
 * or until it reads a null byte. When finished, what has been read exists
 * in "s". If it encounters EOF or an error, it closes the ticket file. 
 *
 * Possible return values are:
 *
 * n            the number of bytes read (including null terminator)
 *              when all goes well
 *
 * 0            end of file or read error
 *
 * TOO_BIG      if "count" characters are read and no null is
 *		encountered. This is an indication that the ticket
 *		file is seriously ill.
 */

static 
ver_ctx_gets(s, n)
    register char *s;
{
    register count;

    if (fd < 0) {
	if (cdc_debug)
	    fprintf(stderr, "ver_ctx_gets called before ver_ctx_init.\n");
	return TKT_FIL_INI;
    }
    while (tfbfr[curpos] == ' ')  { curpos++; keypos++;  }
    for (count = n - 1; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      ver_ctx_close();
	      return 0;
	    }
	    curpos = 0;
	}
	*s = tfbfr[curpos++];
	keypos++;
	if (*s == ';') {
	    *s = '\0';
	    return (n - count);
	}
	if (*s != '\n')  *s++;
    }
    ver_ctx_close();
    return TOO_BIG;
}

static 
ver_ctx_gethex(s, n)
    register char *s;
{
    register count;
    int      hex, high, low;

    if (fd < 0) {
	if (cdc_debug)
	    fprintf(stderr, "ver_ctx_gets called before ver_ctx_init.\n");
	return TKT_FIL_INI;
    }
    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	ver_ctx_close();
	return(0);
      }
      curpos = 0;
    }
    if (tfbfr[curpos] == '\n')  curpos++;
    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	ver_ctx_close();
	return(0);
      }
      curpos = 0;
    }
    if (tfbfr[curpos] == ' ')  curpos++;
    for (count = n - 1; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      ver_ctx_close();
	      return(0);
	    }
	    curpos = 0;
	}
	if (tfbfr[curpos] == ';') {
	    curpos++;
	    return (n - count);
	}
	if ((tfbfr[curpos] != '\n') && (tfbfr[curpos] != ' '))  {
	  if (curpos == lastpos-1) {
	    sscanf(&tfbfr[curpos],"%01x",&high);
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      ver_ctx_close();
	      return(0);
	    }
	    curpos = 0;
	    sscanf(&tfbfr[curpos],"%01x",&low);
	    hex = high * 16 + low;
	    curpos++;
	  } else {
	    sscanf(&tfbfr[curpos],"%02x", &hex);
	    curpos += 2;
	  }
	  *s = hex;
	  *s++;
	} else curpos++;
    }
    ver_ctx_close();
    return TOO_BIG;
}

static
ver_ctx_getbin(s, n, skip)
    register char *s;
    int n, skip;
{
    register count;

    if (fd < 0) return TKT_FIL_INI;

    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	ver_ctx_close();
	return(0);
      }
      curpos = 0;
    }
    if (!skip)
      if ((tfbfr[curpos] == '\n') || (tfbfr[curpos] == ' '))  curpos++;

    for (count = n ; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      ver_ctx_close();
	      return(0);
	    }
	    curpos = 0;
	}
	*s++ = tfbfr[curpos];
	curpos++;
    }
    if (curpos >= lastpos) {
      if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	ver_ctx_close();
	return(0);
      }
      curpos = 0;
    }
    if (tfbfr[curpos] != ';') {
      syslog(LOG_INFO, "ver_ctx_getbin: curpos is %d, got '%02x'", curpos, tfbfr[curpos]);
      return(-1);
    }
    curpos++;
    return(0);
}

/*
 * ver_ctx_read() is an internal routine.  It takes a string "s" and a count
 * "n", and reads from the file until "n" bytes have been read.  When
 * finished, what has been read exists in "s".  If it encounters EOF or
 * an error, it closes the ticket file.
 *
 * Possible return values are:
 *
 * n		the number of bytes read when all goes well
 *
 * 0		on end of file or read error
 */

static
ver_ctx_read(s, n)
    register char *s;
    register n;
{
    register count;
    
    for (count = n; count > 0; --count) {
	if (curpos >= lastpos) {
	    if ((lastpos = read(fd, tfbfr, sizeof(tfbfr))) == 0) {
	      ver_ctx_close();
	      return 0;
	    }
	    curpos = 0;
	}
	*s++ = tfbfr[curpos++];
    }
    return n;
}
     
/*
 * ver_ctx_add_context() appends an credential info to the end of the
 * credential file.  You must call ver_ctx_init() before calling
 * ver_ctx_add_context().
 *
 * Returns ASUCCESS if all goes well, TKT_FIL_INI if ver_ctx_init() wasn't
 * called previously, and AFAILURE for anything else that went wrong.
 */

#define INTBUF_SZ 40

ver_ctx_add_context(ctx_cache, name)
Verifier_ContextCache *ctx_cache;
char         *name;
{
    off_t  lseek();
    int    buflen, count;
    char intbuf[20];

    if (fd < 0) {
      if (cdc_debug)
	fprintf(stderr, "ver_ctx_add_context called before ver_ctx_init\n");
      return TKT_FIL_INI;
    }

    if (write(fd, ctx_cache->deskey.bytes, DESKEY_SZ) != DESKEY_SZ)
      goto bad;
    write (fd, "; ", 2);

    bzero(intbuf, 20);
    sprintf(intbuf, "%x", ctx_cache->before);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
        goto bad;
    write(fd, "; ", 2);
    bzero(intbuf, 20);
    sprintf(intbuf, "%x", ctx_cache->after);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
        goto bad;
    write(fd, "; ", 2);

    bzero(intbuf, 20);
    sprintf(intbuf, "%x", ctx_cache->deleg_flag);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
        goto bad;
    write(fd, "; ", 2);

    bzero(intbuf, 20);
    sprintf(intbuf, "%x", ctx_cache->claimant_fn_len);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
        goto bad;
    write(fd, "; ", 2);

    bzero(intbuf, 20);
    sprintf(intbuf, "%x", ctx_cache->encryptedkey_len);
    count = strlen(intbuf);
    if (write(fd, intbuf, count) != count)
        goto bad;
    write(fd, "; ", 2);

    buflen = ctx_cache->claimant_fn_len + ctx_cache->encryptedkey_len;
    if (write(fd, ctx_cache->charbuf, buflen) != buflen)
      goto bad;
    write (fd, "; ", 2);

    return (ASUCCESS);
  bad:
    syslog(LOG_INFO, "ver_ctx_add_context: wrote bad argument in file\n");
    return(AFAILURE);
}

ver_ctx_get_context(name, ctx_cache)
char *name;
Verifier_ContextCache *ctx_cache;
{
    char str[INT_STR], claimant_name[FULLNAME_SZ];

    if (fd < 0) {
      if (cdc_debug)
	fprintf(stderr, "tf_get_pname called before tf_init.\n");
      return TKT_FIL_INI;
    }

    if (ver_ctx_getbin(ctx_cache->deskey.bytes, DESKEY_SZ, 1) < 0) {
      syslog(LOG_INFO, "deskey - TKT_FIL_FMT\n");
      return(-1);
    }

    bzero(str, INT_STR);
    if (ver_ctx_gets(str, INT_STR) < 0) {
      syslog(LOG_INFO, "before - TKT_FIL_FMT\n");
      return(-1);
    }
    sscanf(str, "%x", &ctx_cache->before);

    bzero(str, INT_STR);
    if (ver_ctx_gets(str, INT_STR) < 0) {
      syslog(LOG_INFO, "after - TKT_FIL_FMT\n");
      return(-1);
    }
    sscanf(str, "%x", &ctx_cache->after);

    bzero(str, INT_STR);
    if (ver_ctx_gets(str, INT_STR) < 0) {
      syslog(LOG_INFO, "deleg_flag - TKT_FIL_FMT\n");
      return(-1);
    }
    sscanf(str, "%x", &ctx_cache->deleg_flag);

    bzero(str, INT_STR);
    if (ver_ctx_gets(str, INT_STR) < 0) {
      syslog(LOG_INFO, "claimant_fn_len - TKT_FIL_FMT\n");
      return(-1);
    }
    sscanf(str, "%x", &ctx_cache->claimant_fn_len);

    bzero(str, INT_STR);
    if (ver_ctx_gets(str, INT_STR) < 0) {
      syslog(LOG_INFO, "encryptedkey_len - TKT_FIL_FMT\n");
      return(-1);
    }
    sscanf(str, "%x", &ctx_cache->encryptedkey_len);

    if (ver_ctx_getbin(ctx_cache->charbuf, ctx_cache->claimant_fn_len + ctx_cache->encryptedkey_len, 0) < 0) {
      syslog(LOG_INFO, "charbuf - TKT_FIL_FMT\n");
      return(-1);
    }

    strncpy(claimant_name, ctx_cache->charbuf, ctx_cache->claimant_fn_len);
    if (strncasecmp(claimant_name, name, strlen(name)) != 0) {
      syslog(LOG_INFO, "name doesn't match - fail\n");
      return(AFAILURE);
    }
    return (ASUCCESS);
}
