/*
 *
 * $Source: /filesv/usr/local/proj/sphinx/spx2/src/lib/api/RCS/read_pub_key.c,v $
 *
 *
 *  MODULE NAME:    read_pub_key.c
 *
 *
 *  AUTHORS:
 *
 *	K. Alagappan
 *
 */


/*
 * COPYRIGHT (C) 1992 DIGITAL EQUIPMENT CORPORATION
 * ALL RIGHTS RESERVED
 *
 * "Digital Equipment Corporation authorizes the reproduction,
 * distribution and modification of this software subject to the following
 * restrictions:
 * 
 * 1.  Any partial or whole copy of this software, or any modification
 * thereof, must include this copyright notice in its entirety.
 *
 * 2.  This software is supplied "as is" with no warranty of any kind,
 * expressed or implied, for any purpose, including any warranty of fitness 
 * or merchantibility.  DIGITAL assumes no responsibility for the use or
 * reliability of this software, nor promises to provide any form of 
 * support for it on any basis.
 *
 * 3.  Distribution of this software is authorized only if no profit or
 * remuneration of any kind is received in exchange for such distribution. 
 * 
 * 4.  This software and all application programs are to be used only for
 * non-commercial purposes. However, media costs associated with the
 * distribution of the software or application programs may be recovered.
 *
 */


#include <stdio.h>
#include <syslog.h>
#include <sys/time.h>
#include "cdc.h"
#include "cdc_db.h"
#include "BigNum.h"
#include "BigRSA.h"
#include "random.h"
#include "spxapi_defs.h"
#include "SPHINX-types.h"

#ifdef sun
#define strcasecmp	strcmp_support
#endif

static KTEXT_ST rdpubkey_pkt_st;
KTEXT rdpubkey_pkt = &rdpubkey_pkt_st;       /* request packet  to CDC */

static CrossNameList   crossnames;
static char            walk_down_name[FULLNAME_SZ];

extern char *rindex();
char *rdn_to_str();

/*
 *  status returned :
 *  	1   -  public key found
 *     -1   -  unable to read certifs from CDC  (request packet saved)
 *     -2   -  tree walk down fails
 *     -3   -  tree walk up fails
 *     -4   -  X.500 target name in unknown domain
 *     -5   -  certif path not found in CDC
 *     -6   -  unusual error (syntax, etc.)  --- shouldn't occur
 */

int read_pub_key(fullname, target_pubkey_p, ta_cred, cdc_server, debugflag)
char          *fullname;
RSAKeyStorage *target_pubkey_p;
TrustedCred   *ta_cred;
char          *cdc_server;
int           debugflag;
{
  int             ta_found, j;
  int             status, so_far_len;
  char            *resolvefrom = NULL;
  char            cdc_domain[FULLNAME_SZ], *cp;
  TrustedAuthority *ta;
  struct type_SPHINX_CertPath        *certpath;

  /*
   *  from our list of TAs, decide which one is the best to use
   *    if TA name matches target, then we already have the public key
   *        so return
   *    if TA name is immediate superior of target, only request certif
   *        for target (resolvefrom name set to NULL)
   *    if TA name is superior of target (not immediate though), request
   *        certif path for target (resolvefrom name set to TA name)
   *    if unable to find a TA that is superior for target, look for
   *        cross certificates
   */

  ta_found = 0;
  ta = ta_cred->head;
  crossnames.num = 0;
  walk_down_name[0] = '\0';

  while ((!ta_found) && (ta)) {

    if ((j = str_cmp(ta->fullname, fullname)) >= 0) {

      if (j == 0) {
        /**  One of our TAs matches the target principal ; so return it  **/
        memcpy(target_pubkey_p, ta->pubRSAKey, sizeof(RSAKeyStorage));
        return(1);
      }
      ta_found = 1;
      strcpy(walk_down_name, ta->fullname);
      crossnames.num = 0;
      if (j == 1) {
	resolvefrom = NULL;
      } else {

        /*  map X.500 fullname to appropriate CDC domain -- use CDC_CONF */
        status = x500_to_cdc(fullname, cdc_domain);

        if (status != ASUCCESS) return(-4);   /*  CDC domain not found  */

	if (strcasecmp(cdc_domain, ta->fullname) != 0) {
	  /*
	   *  since TA name is superior to domain name, setup request to
	   *  look for incoming certificate to the domain.
	   */
	  crossnames.num = 1;
	  strcpy(crossnames.names[0], ta->fullname);
	  resolvefrom = NULL;
	} else resolvefrom = ta->fullname;
      }
    }
    ta = (TrustedAuthority *) ta->next;
  }
  
  if (debugflag) {
    printf("ta_found is %d\n", ta_found);
    if (ta_found) {
      printf("resolvefrom is '%s'\n", resolvefrom);
      printf("num of crossname is %d\n", crossnames.num);
      if (crossnames.num == 1)
	printf("crossnames.names[0] is '%s'\n", crossnames.names[0]);
    }
  }


  /*
   *  if we haven't found a valid TA yet, check our TA list for cross
   *  certifiers.  If so, request certif path for target (resolvefrom
   *  name set to NULL and with list of cross certifiers).
   */

  if (!ta_found) {
    ta = ta_cred->head;
    strcpy(walk_down_name, "");
    while ((crossnames.num < 4) && (ta)) {
      if (ta->cross_certifier) {
	ta_found = 1;
	strcpy(crossnames.names[crossnames.num], ta->fullname);
	crossnames.num++;
      }
      ta = (TrustedAuthority *) ta->next;
    }
    if (!ta_found) {
      syslog(LOG_INFO, "tree walk up fails -- no local cross certifiers found");
      if (debugflag)
	printf("tree walk up fails -- no local cross certifiers found\n");
      return(-3);
    }

    /*  map X.500 fullname to appropriate CDC domain -- using CDC_SERVERS */
    status = x500_to_cdc(fullname, cdc_domain);

    if (status != ASUCCESS) return(-4);   /*  CDC domain not found  */

    if (debugflag) {
      printf("found %d cross certifiers\n", crossnames.num);
      for (j=0;j<crossnames.num;j++)
	printf("%dth is '%s'\n", j, crossnames.names[j]);
    }
  }

  /*
   *  issue request for certificates from CDC
   *      fullname    -  target's X.500 fullname
   *      resolvefrom -  superior's name for whom we have the public key
   *      crossnames  -  list of cross certifiers
   *      certpath    -  list of certificates (returned by CDC)
   *      cdc_server  -  specific CDC server to try contacting
   *      debugflag   -  verbose mode
   *
   *  NOTE: It is not possible to have a resolvefrom name and crossnames.
   *        Either 1) crossnames contains a list of names and resolvefrom is
   *        NULL or 2) crossname contains an empty list and resolvefrom name
   *        contains the name of the target's superior for whom we have
   *        a public key.  As an optimization, if the superior's name is
   *        the immediate superior for target, we can set the resolvefrom
   *        name to NULL (since this is the default).
   *
   */

  rdpubkey_pkt->length = 0;  /*  reset length  */
  if ((status =
       read_certif_cstub(rdpubkey_pkt,
			 fullname,
			 resolvefrom,
			 &crossnames,
			 &certpath,
			 cdc_server,0)) != 0) {
    switch(status) {
      case CDC_BUILD_REQ_ERROR :
	return(-6);
        break;
      case CDC_UNAVAILABLE :
	/*
	 *  even though this host isn't able to communicate with CDC
	 *  server, maybe the other side can.  therefore save request
	 *  certif packet and return to caller
	 */
        if (rdpubkey_pkt->length <= 0)
	  return(-6);
	/*
	 *  the static variable - rdpubkey_pkt contains the cdc request, this
	 *  can be retrieved using forward_cdc_request()
	 */
	return(-1);
	break;
      case CDC_CHECK_RES_ERROR :
	return(-6);
	break;
      case CDC_CHECK_RES_UNKNOWN :
	return(-6);
	break;
      default :
	return(-5);
	break;
    }
  }

   /*
    *  We now need to validate the certification path returned using our
    *  TA list.
    */

  status =
    tree_walk_down_using_certifs(fullname, target_pubkey_p, ta_cred, certpath, debugflag);
  return(status);
}


/*
 *  status returned :
 *  	1   -  public key found
 *     -2   -  tree walk down fails
 *     -4   -  unable to map target fullname to CDC domain
 *     -5   -  certif path not found in CDC
 *     -6   -  unusual error (syntax, etc.)  --- shouldn't occur
 */

int partial_read_pub_key(fullname, target_pubkey_p, ta_cred, resp_pkt, resp_pktlen, debugflag)
char          *fullname;
RSAKeyStorage *target_pubkey_p;
TrustedCred   *ta_cred;
char          *resp_pkt;
int           resp_pktlen;
int           debugflag;
{
  int             j, x_found;
  int             status, so_far_len;
  char            *cp;
  TrustedAuthority *ta;
  struct type_SPHINX_CertPath        *certpath;

  /*
   *  we don't need to issue request from CDC, since we have input
   *  already
   */

  if (status = check_certif_res_apdu(resp_pkt, resp_pktlen, &certpath, debugflag))
    switch(status) {
      case CDC_CHECK_RES_ERROR :
	return(-6);
	break;
      case CDC_CHECK_RES_UNKNOWN :
	return(-6);
	break;
      default :
        syslog(LOG_INFO, "check_certif_res_apdu returns %d", status);
	return(-5);
	break;
  }

  /*
   *  We now need to validate the certification path returned using our
   *  TA list.
   */

  status =
    tree_walk_down_using_certifs(fullname, target_pubkey_p, ta_cred, certpath, debugflag);
  return(status);
}



int tree_walk_down_using_certifs(fullname, target_pubkey_p, ta_cred, certpath, debugflag)
char          *fullname;
RSAKeyStorage *target_pubkey_p;
TrustedCred   *ta_cred;
struct type_SPHINX_CertPath        *certpath;
int    debugflag;
{
  int             j, x_found, verified_first_certif = 0;
  int             status, so_far_len;
  char            *certif_issu, *certif_subj;
  char            cdc_domain[FULLNAME_SZ], *cp;
  int             subject_uid_len, issuer_uid_len;
  unsigned char   subject_uid[UID_SZ], issuer_uid[UID_SZ], *encoded_pubkey;
  TrustedAuthority *ta;
  struct type_SPHINX_Certificate     *cert;
  int  dbg_found_local_x_certif = 0;                /*  only for debugging  */
  int  dbg_found_local_x_certif_match_name = 0;     /*  only for debugging  */
  int  dbg_found_local_x_certif_that_verifies = 0;  /*  only for debugging  */

   /*
    *  We now need to validate the certification path returned using our
    *  TA list.
    *
    *      verified_first_certif variable indicates if we've already
    *      used one of our TAs to get started.  If so, don't check TA
    *      list any more.
    */

  verified_first_certif = 0;

  if (crossnames.num != 0) {

    x_found = 0;
    if (debugflag)
      printf("since we used a cross certifier, find out which one\n");
    cert = certpath->Certificate;
    certif_issu = rdn_to_str(cert->cinfo->issuer);
    certif_subj = rdn_to_str(cert->cinfo->subject);
    if (debugflag) {
      printf("issuer's name for cross certif is '%s'\n", certif_issu);
      printf("subject's name for cross certif   '%s'\n", certif_subj);
    }

    dbg_found_local_x_certif = 0;                /*  only for debugging  */
    dbg_found_local_x_certif_match_name = 0;     /*  only for debugging  */
    dbg_found_local_x_certif_that_verifies = 0;  /*  only for debugging  */

    ta = ta_cred->head;
    while ((!x_found) && (ta)) {
      if (ta->cross_certifier) {
	dbg_found_local_x_certif++;
        if (strcasecmp(certif_issu, ta->fullname) == 0) {
	  dbg_found_local_x_certif_match_name++;
  	  if (verify_certif_aux(cert, ta->pubRSAKey)==1) {
	    dbg_found_local_x_certif_that_verifies++;
	    x_found = 1;   /* found cross certif that verifies first certif */
	    break;
	  }
        }
      }
      ta = (TrustedAuthority *) ta->next;
    }

    if (!x_found) {
      syslog(LOG_INFO, "unable to verify initial certif with TA using cross certif");
      syslog(LOG_INFO, "found_local %d, name_match %d, verifies %d", dbg_found_local_x_certif, dbg_found_local_x_certif_match_name, dbg_found_local_x_certif_that_verifies);
      if (debugflag)
	printf("unable to verify initial certif with TA using cross certif\n");
      return(-2);
    }

    if (debugflag)
      printf("... success\nfound cross certifier, check name relationship\n");
    if (strcasecmp(cdc_domain, certif_subj) != 0) {
      syslog(LOG_INFO,"read_pub_key : domain prefix doesn't match subject name in certif");
      syslog(LOG_INFO, "fullname - '%s'", cdc_domain);
      syslog(LOG_INFO, "subject name - '%s'", certif_subj);
    }

    if (!key_from_certif(cert->cinfo,target_pubkey_p)) {
      syslog(LOG_INFO,"error decoding target public key");
      if (debugflag)
	printf("error decoding target public key\n");
      return(-2);
    }
    uid_from_certif(cert->cinfo,issuer_uid,&issuer_uid_len,
		    subject_uid,&subject_uid_len);
    strcpy(walk_down_name, certif_subj);
    verified_first_certif = 1;   /*  used one of TAs to get started  */
    free(certif_issu);
    free(certif_subj);
    certpath = certpath->next;
  }

  while ((strcasecmp(walk_down_name, fullname) != 0) && (certpath != NULL)) {
    if (debugflag) {
      printf("\n****\ncurrent walk_down_name is '%s'\n", walk_down_name);
    }
    cert = certpath->Certificate;
    certif_issu = rdn_to_str(cert->cinfo->issuer);
    certif_subj = rdn_to_str(cert->cinfo->subject);
    if (debugflag) {
      printf("issuer's name in next certif is '%s'\n", certif_issu);
      printf("subject's name in next certif is '%s'\n", certif_subj);
    }

    /*
     *  if we haven't verified our first certif yet, then check if one of
     *  TAs is able to verify it.
     */

    if (!verified_first_certif) {
      ta = ta_cred->head;
      while ((!verified_first_certif) && (ta)) {
        if (strcasecmp(certif_issu, ta->fullname) == 0) {
          if (verify_certif_aux(cert, ta->pubRSAKey)==1) {
            verified_first_certif = 1;
            break;
          }
        }
        ta = (TrustedAuthority *) ta->next;
      }
      if (!verified_first_certif) {
	/*
	 *  if still haven't found a TA to start us off, then error
	 */
	syslog(LOG_INFO, "unable to verify initial certif with one our TAs");
	if (debugflag)
	  printf("unable to verify initial certif with one our TAs\n");
	return(-2);
      }
    } else {
      if (debugflag)
	printf("verify this certif using previous target_pubkey\n");

      if (verify_certif_aux(cert, target_pubkey_p)==1) {
	if (debugflag) printf("... success\n");
      } else {
        if (debugflag) {
	  printf("... fail\n");
	  printf("issuer's public key is :\n");
	  encoded_pubkey = (unsigned char *) EncodePublic(target_pubkey_p);
	  dumphex(encoded_pubkey, DecodeTotalLength(encoded_pubkey));
        }
	syslog(LOG_INFO, "verifying certif during tree walk down fails");
	return(-2);
      }
    }

    if (strncasecmp(certif_issu, certif_subj, strlen(certif_issu)) != 0) {
      syslog(LOG_INFO,"read_pub_key : ca name doesn't match subject name in certif");
      syslog(LOG_INFO, "issu_fullname - '%s'", certif_issu);
      syslog(LOG_INFO, "subject name - '%s'", certif_subj);
    }
    if (!key_from_certif(cert->cinfo,target_pubkey_p)) {
      syslog(LOG_INFO,"error decoding target public key");
      if (debugflag) printf("error decoding target public key\n");
      return(-2);
    }
    uid_from_certif(cert->cinfo,issuer_uid,&issuer_uid_len,
		    subject_uid,&subject_uid_len);
    free(certif_issu);
    free(certif_subj);
    so_far_len = strlen(walk_down_name);
    strcpy(walk_down_name, fullname);
    if ((cp = index(&walk_down_name[so_far_len+1], '/')) != 0) *cp = '\0';
    certpath = certpath->next;
  }
  if (strcasecmp(walk_down_name, fullname) == 0) return(1);
  if (debugflag) {
    printf("walk_down_name is '%s'\n", walk_down_name);
    printf("fullname is '%s'\n", fullname);
    printf("looks like all the certificates weren't found in CDC\n");
  } 
  return(-2);
}

forward_cdc_request_len(len)
int  *len;
{
  if (rdpubkey_pkt->length > 0)
    *len = rdpubkey_pkt->length;
  else *len = 0;
}

forward_cdc_request_data(buf)
char *buf;
{
  if (rdpubkey_pkt->length > 0)
    bcopy(rdpubkey_pkt->dat, buf, rdpubkey_pkt->length);
}
