/* Written by Germano Caronni and Werner Almesberger */
/* (c) by G. Caronni in '94 */
/* This program is under the GNU Public License Version 2 */

#include <stdio.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

#include <process.h>
#include <time.h>
#include <sys/timeb.h>
#include <winsock.h>

#ifdef _AIX
#include <sys/select.h>
#endif

#include "getopt.h"

#include "dic.h"
#include "keyclnt.h"
#include "pipe.h"

#ifdef LINT

typedef int WORD;
typedef int WSADATA;
typedef int fd_set;

#endif

static char buf[MAX_MSG];
static int debug = 0;
static int byebye = 0;

extern char *optarg;
extern int optind;

double measure(double);
static void integrity(void);
static void send_rep(char *);
static double iterations(time_t);
static void usage(const char *);
static void send_g(double, time_t);

static double bloody_floor(double);

static int limit_packets = 0;
static int n_packets = 1;

double measure(double iter)
{
	struct timeb tstruct;
	double len;

	printf("Performance test: %.0f iterations in ", iter);
	fflush(stdout);
	ftime(&tstruct);
	len = tstruct.time * 1000.0 + (double) tstruct.millitm;
	init("112233445566", iter, "1234567812345678", "1234567812345678");
	doit();
	ftime(&tstruct);
	len = (((double) tstruct.time * 1000.0 + (double) tstruct.millitm) - len) / 1000.0;

	printf("%3.2f seconds.\n", len);
	return len;
}


static void integrity(void)
{
	printf("Integrity test ");
	fflush(stdout);
	/*init("282604780651",iter,"77E459E9C593F4F0","BF8886D2A950841C"); */
	init("272604780000", 1000000.0, "77E459E9C593F4F0", "BF8886D2A950841C");
	if (doit() == 1 && strcmp("282604780651", getresult()) == 0) {
		printf("okay. \n");
	} else {
		printf(" FAILED !\n");
		exit(-1);
	}
}


static void send_rep(char *m)
{
	fd_set fdset;
	struct timeval to;

	struct sockaddr_in from;
	int len, fromlen, fds, msglen;

	time_t retry_delay = RETRY_DELAY;
	time_t timeout;

	while (1) {
		FD_ZERO(&fdset);
		FD_SET(RECV, &fdset);
		to.tv_sec = 0;
		to.tv_usec = 0;
		if ((fds = select(RECV + 1, &fdset, NULL, NULL, &to)) < 0) {
			PDIE("select")
		}
		if (!fds)
			break;
		fromlen = sizeof(from);
		(void) RECVFROM(RECV, buf, MAX_MSG, 0, (struct sockaddr *) &from, &fromlen);
	}
	len = strlen(m);

	timeout = time((time_t *) 0) + MAX_DELAY * 60;

	while (time((time_t *) 0) < timeout) {
		if (debug)
			printf("Send: %s\n", m);
		if (SENDTO(SEND, m, len, 0, (struct sockaddr *) &addr, sizeof(addr)) < 0)
			perror("sendto");
		FD_ZERO(&fdset);
		FD_SET(RECV, &fdset);
		to.tv_sec = retry_delay > MAX_RETRY_DELAY ?
		    MAX_RETRY_DELAY :
		    (retry_delay = retry_delay + RETRY_DELAY);
		to.tv_usec = 0;
		if ((fds = select(RECV + 1, &fdset, NULL, NULL, &to)) < 0) {
			PDIE("select")
		}
		if (!fds)
			continue;

		fromlen = sizeof(from);
		if ((msglen = RECVFROM(RECV, buf, MAX_MSG, 0, (struct sockaddr *) &from,
				       &fromlen)) < 0) {
			PDIE("recvfrom");
		}
		if (from.sin_addr.s_addr != addr.sin_addr.s_addr) {
			fprintf(stderr, "Received packet from host %lX\n",
				(unsigned long) from.sin_addr.s_addr);
			continue;
		}
		buf[msglen] = 0;
		if (debug)
			printf("Rcvd: %s\n", buf);
		return;
	}
	fprintf(stderr, "\nARGLLLLLLLLLLLLGNAAAAAAAAAA!!!!!\n");
	exit(2);
}

static double iterations(time_t time_mode)
{
	double len;
	double iter;

	printf("Job length scheduled to %d minutes.\n", time_mode / 60);
	printf("Timeout occurs after %d minutes.\n", time_mode / 30);

	len = measure(2.0e5);

	iter = 2.0e5 * (double) time_mode / len;
	iter = bloody_floor(iter);

	printf("Requesting %.0f keys.\n", iter);

	return iter;
}

static void usage(const char *prog)
{
	fprintf(stderr, "usage: %s -m\n", prog);
	fprintf(stderr, "       %s [-d] [-t job-length] [-K killfile] [-p port]\n"
		        "              [-c n] [-T task-length] server\n",
		prog);
}

static int exists(const char *fname)
{
	struct stat sbuf;
	return stat(fname, &sbuf) == 0;
}

static void send_g(double iter, time_t time_mode)
{
	char msg[MAX_MSG];
	char lbuff[1024];

	if (gethostname(lbuff, sizeof(lbuff))==-1)
		strcpy(lbuff, "unknown");

	sprintf(msg, "G%.0f %ld <%s,%s-%s-%s,%s,%s,%u>", iter, 
		(unsigned long)time_mode * 2,
		VERSION, "Win32", "X", "Intel", lbuff,
		"win_user", (unsigned int) getpid());

	send_rep(msg);

}



static double bloody_floor(double in)
{
	unsigned long int hi;
	unsigned long int lo;

	hi = in / 4294967296.0;
	lo = in - 4294967296.0 * ((double)hi);

	return ((double) hi) * 4294967296.0 + ((double) lo);
}



int main(int argc, char *argv[])
{
	unsigned long id;
	double range;
	int number = 0;
	double iter, old_iter;

	struct timeb tstart, tstop;
	time_t time_mode = SECONDS;

	time_t expected;
	time_t dying_time = 0;

	struct hostent *temp;
	char msg[MAX_MSG], clear[20], start[20], cipher[20];
	const char *prog = argv[0];
	const char *killfile = CLIENT_KILL;
	int x;

	/* const char *lockfile = LOCK;     */
	/* int lockfd;                      */
	/* int nicety = NICE_VAL;           *//* wird nicht benutzt */
	/* int forkalways=0;                */
	/* int ischild = 0;                 */

	struct in_addr our_ip;
	int our_family;
	short dic_port;

	WORD wVersionRequested;
	WSADATA wsaData;

	int err;


	wVersionRequested = MAKEWORD(1, 1);
	if ((err = WSAStartup(wVersionRequested, &wsaData)) != 0) {
		PDIE("no winsock.dll v1.1 found");
	}
	if (argc == 2 && !strcmp(argv[1], "-m")) {
		printf("DIC Client v%s\n", VERSION);
		measure(1000000.0);
		integrity();
		exit(0);
	}
	our_ip.s_addr = INADDR_ANY;
	our_family = AF_INET;
	dic_port = DIC_PORT;

	while ((x = getopt(argc, argv, "T:c:I:dt:K:p:")) != EOF) {
		switch (x) {
		case 'T':
			dying_time = time((time_t *) NULL) + atoi(optarg) * 60;
			break;
		case 'c':
			limit_packets = 1;
			n_packets = atoi(optarg);
			break;
		case 'I':
			if ((temp = gethostbyname(optarg))) {
				memcpy(&our_ip, temp->h_addr, temp->h_length);
				our_family = temp->h_addrtype;
			} else {
				our_family = AF_INET;
				if ((our_ip.s_addr = inet_addr(optarg)) == -1)
					PDIE(optarg);
			}
			our_ip.s_addr = ntohl(our_ip.s_addr);
			break;
		case 'd':
			debug = 1;
			break;
		case 't':
			time_mode = (time_t) (atoi(optarg) * 60);
			break;
		case 'K':
			killfile = optarg;
			break;
		case 'p':
			dic_port = atoi(optarg);
			break;
		default:
			usage(prog);
			exit(1);
		}
	}


	argv += optind - 1;
	argc -= optind - 1;

	if (debug)
		time_mode = 5;

	if (argc != 2) {
		usage(prog);
		exit(0);
	}
	if (exists(killfile)) {
		printf("DIC Client v%s: Startup prohibited by %s\n",
		       VERSION, killfile);
		exit(1);
	}
	printf("DIC Client v%s\n", VERSION);

	SetPriorityClass(GetCurrentProcess(), IDLE_PRIORITY_CLASS);
	integrity();
	if (dying_time != 0)
		printf("This client will terminate at %s",
		       ctime(&dying_time));

	iter = iterations(time_mode);

	if ((SEND = socket(PF_INET, SOCK_DGRAM, 0)) < 0)
		PDIE("socket");
	memset(&addr, 0, sizeof(addr));
	addr.sin_family = our_family;
	addr.sin_port = htons(0);
	addr.sin_addr.s_addr = htonl(our_ip.s_addr);
	if (bind(SEND, (struct sockaddr *) &addr, sizeof(addr)) < 0)
		PDIE("bind");
	addr.sin_port = htons(dic_port);
	if ((temp = gethostbyname(argv[1]))) {
		addr.sin_family = temp->h_addrtype;
		memcpy(&addr.sin_addr, temp->h_addr, temp->h_length);
	} else {
		addr.sin_family = AF_INET;
		if ((addr.sin_addr.s_addr = inet_addr(argv[1])) == -1)
			PDIE(argv[1]);
	}


	send_g(iter, time_mode);

	while (1) {
	      again:
		if (buf[0] == 'K') {
			printf("Client was killed (%s)\n", buf);
			exit(2);
		} else if (buf[0] == 'R') {
			if (!iter)
				iter = iterations(time_mode);
			send_g(iter, time_mode);
			goto again;
		} else if (buf[0] == 'I') {
			/* server is giving us an idle packet, this happens when our
			 * new range was set to 0. 
			 * This happens a) when we want it because of a pending kill,
			 * where we just wanted to collect a confirmation, and b) when
			 * our iter calculation produced junk.
			 */
			if (byebye) {
				printf("Got return receipt from server -- terminating.\n");
				exit(0);
			}
			/* what now ? send a H packet which is nearly a G packet */
			if (!iter)
				iter = iterations(time_mode);
			sprintf(msg, "H%.0f %d", iter, time_mode * 2);
			send_rep(msg);
			goto again;
		}
		if (sscanf(buf, "J%lu %19s %19s %19s %lf", &id, clear, cipher, start,
			   &range) != 5) {
			printf("Invalid msg: '%s'\n", buf);
			exit(1);
		}
		expected = time(NULL);
		expected += (((double) time_mode) * range / iter);
		printf("[%d:%lf --> %.24s] ", number, range, ctime(&expected));
		fflush(stdout);

		init(start, range, clear, cipher);

		ftime(&tstart);

		x = doit();
		if (x == 1) {
			printf("\nFound %s\n", getresult());
			sprintf(msg, "F%s %s %s", clear, cipher, getresult());
			send_rep(msg);
			goto again;
		}
		fflush(stdout);

		ftime(&tstop);

		if (tstop.time < tstart.time)
			tstop.time += tstart.time;

		old_iter = range;


		{
			double old_kps, cur_kps, new_kps, span;

			old_kps = iter / (double) time_mode;
			span = tstop.time - tstart.time;
			span += (tstop.millitm - tstart.millitm) / 1.0e3;
			cur_kps = range / span;
			if (cur_kps < old_kps) {
				new_kps = (3.0 * cur_kps + 2.0 * old_kps) / 5.0;
			} else if (cur_kps > 3.5 * old_kps) {
				new_kps = 1.5 * old_kps;
			} else {
				new_kps = (cur_kps + 4.0 * old_kps) / 5.0;
			}
			if (dying_time && time(NULL) + time_mode >= dying_time) {
				time_mode = dying_time - time(NULL) + 1;
			}
			iter = bloody_floor(new_kps * (double) time_mode);
		}


		if (!byebye) {
			byebye = exists(killfile);
		}
		if (!byebye) {
		     if (limit_packets && !(--n_packets)) {
			  byebye = 1;
		     }
		}
		if (!byebye)
		     if (dying_time && time((time_t *) NULL) >= dying_time) {
			  byebye = 1;
		     }

		if (byebye) {
			printf("Batch done in %3.2f minutes. Sending last message...\n",
			     ((double) tstop.time - tstart.time) / 60.0);
			sprintf(msg, "D%s %d %d", buf, 0, time_mode * 2);
		} else {
			printf("Batch done in %3.2f minutes. "
			       "Requesting new job with size %.0f\n",
			((double) tstop.time - tstart.time) / 60.0, iter);
			sprintf(msg, "D%s %.0f %d", buf, iter, time_mode * 2);
		}
		send_rep(msg);
	}

	return 0;

}
