package Amazon::API::Botocore::Shape::Serializer;

use strict;
use warnings;

BEGIN {
  use English qw(-no_match_vars);

  eval { require Log::Log4perl; };

  if ($EVAL_ERROR) {
    no strict qw(refs); ## no critic (ProhibitNoStrict)

    *{ __PACKAGE__ . "::$_" } = sub { }
      for qw(DEBUG INFO WARN ERROR FATAL);
  }
  else {
    no warnings;        ## no critic (ProhibitNoWarnings)

    Log::Log4perl->import(qw(:easy));
  }
}

use parent qw(Class::Accessor::Fast);

use Amazon::API::Constants              qw(:chars :booleans);
use Amazon::API::Botocore::Shape::Utils qw(require_shape is_hash is_array);

use Carp;
use Data::Dumper;
use English      qw(-no_match_vars);
use JSON         qw(decode_json encode_json);
use Scalar::Util qw(reftype);
use List::Util   qw(any none);
use POSIX        qw(strftime);

use URL::Encode qw(url_decode_utf8);

our $VERSION = '2.0.12'; ## no critic (RequireInterpolationOfMetachars)

__PACKAGE__->follow_best_practice;

__PACKAGE__->mk_accessors(qw( service delete_empty_members debug logger ));

########################################################################
sub new {
########################################################################
  my ( $class, @args ) = @_;

  my %options = ref $args[0] ? %{ $args[0] } : @args;

  my $self = $class->SUPER::new( \%options );

  return $self;
}

########################################################################
sub _serialize_structure {
########################################################################
  my ( $self, $this, $data, $service, $exists ) = @_;

  $service ||= $self->get_service;

  TRACE(
    Dumper [
      class               => ref $this,
      serialize_structure => $this,
      data                => $data,
      exists              => $exists,
    ]
  );

  my $members = $this->get_members;

  return if !defined $data;

  #  my $result = {};
  my $result = $exists ? {} : undef;

  foreach my $m ( keys %{$members} ) {
    my $member        = $members->{$m};
    my $location_name = $member->{locationName} || $member->{location} || $m;

    my $has_location
      = ( exists $member->{locationName} ) || ( exists $member->{location} );

    my $shape = $member->{shape};
    my $serialized_data;

    DEBUG(
      sub {
        return Dumper(
          [ member_name  => $m,
            member       => $member,
            shape        => $shape,
            location     => $location_name,
            has_location => $has_location,
            exists       => $exists,
          ]
        );
      }
    );

    my $member_data = $data->{$location_name};

    DEBUG(
      sub {
        return Dumper(
          [ member       => $m,
            shape        => $shape,
            data         => $data,
            location     => $location_name,
            member_data  => $member_data,
            has_location => $has_location,
            exists       => $exists,
          ]
        );
      }
    );

    # because we may have serialized an XML object...
    if ( $member_data && ref($member_data) && $shape eq 'String' ) {
      $member_data = $member_data->{content} // $EMPTY;
    }

    DEBUG(
      sub {
        return Dumper(
          [ member_data   => $member_data,
            member        => $member,
            location_name => $location_name,
            has_location  => $has_location,
            caller_exists => $exists,
            exists        => exists $data->{$location_name},
          ]
        );
      }
    );

    # has location means this structure member should be inserted into
    # a hash at that location - when it doesn't exist the data is
    # inserted directly into the member name

    # if member data does not exist, then entire data structure is
    # passed to the serializer
    if ( !$has_location || $member_data || exists $data->{$location_name} ) {
      $serialized_data = $self->serialize(
        shape   => $shape,
        data    => $member_data || $data,
        service => $service,
        exists  => exists $data->{$location_name},
      );

      DEBUG(
        sub {
          return Dumper(
            [ "back from serializing $shape",
              data          => $serialized_data,
              location_name => $location_name,
              member        => $m,
              flattened     => $this->{flattened},
            ]
          );
        }
      );
    }

    # skip empty members
    next if ( !exists $data->{$location_name} ) && !defined $serialized_data;

    if ( 1 || $this->{type} ne 'structure' || !$has_location ) {

      $result //= {};
      if ( $this->{flattened} ) {
        $result->{$m} = $serialized_data;
      }
      else {
        if ( is_hash($serialized_data)
          && exists $serialized_data->{$location_name} ) {
          $result->{$m} = $serialized_data->{$location_name};
        }
        else {
          $result->{$m} = $serialized_data;
        }
      }
    }
    else {
      DEBUG('returning serialized_data');
      $result = $serialized_data;
    }

    DEBUG(
      sub {
        return Dumper(
          [ result          => $result,
            member          => $m,
            serialized_data => $serialized_data,
            location_name   => $location_name,
            exists          => $exists,
          ]
        );
      }
    );
  }

  return $result;
}

########################################################################
sub _serialize_list {
########################################################################
  my ( $self, $this, $data, $service, $exists ) = @_;

  $service ||= $self->get_service;

  DEBUG( Dumper( [ 'serialize_list', $this, $data, $service ] ) );

  my $member        = $this->get_member;
  my $location_name = $member->{locationName};

  if ( !$location_name ) {
    if ( ref $data && reftype($data) eq 'HASH' && exists $data->{member} ) {
      $location_name = 'member';
    }
  }

  my $list = $location_name ? $data->{$location_name} : $data;

  my $member_shape = $member->{shape};

  DEBUG(
    Dumper(
      [ 'serialize_list',
        member        => $member,
        location_name => $location_name,
        list          => $list,
        member_shape  => $member_shape,
      ]
    )
  );

  my $result = [];

  if ( $list && !ref $list ) {
    $list = [$list];
  }
  elsif ( $list && reftype($list) eq 'HASH' ) {

    my @key_values;

    DEBUG( Dumper( [ 'pre-pre-marker', $list ] ) );

    if (0) {
      foreach my $key ( keys %{$list} ) {
        push @key_values, { $key => $list->{$key} };
        next;

        DEBUG( Dumper( [ 'pre-marker', $key ] ) );

        if ( ref $list->{$key} && exists $list->{$key}->{value} ) {
          push @key_values,
            {
            value => $list->{$key}->{value},
            key   => $key
            };
        }
        else {
          push @key_values, { $key => $list->{$key} };
        }
      }
    }
    else {
      $list = [$list];
    }

    DEBUG( Dumper( [ key_values => $member_shape, \@key_values ] ) );

    my %result;

    foreach my $p ( @{$list} ) {

      DEBUG( Dumper( [ 'post-marker', $p, $member_shape ] ) );

      my $serialized_data = $self->serialize(
        shape   => $member_shape,
        data    => $p,
        service => $service,
        exists  => $location_name && exists $data->{$location_name},
      );

      next if !defined $serialized_data;

      push @{$result}, $serialized_data;
    }

    local $LIST_SEPARATOR = q{,};

    DEBUG(
      sub {
        return Dumper(
          [ 'ref(self)' => ref($self),
            'ref(this)' => ref($this),
            result      => $result,
            list        => $list
          ]
        );
      }
    );
  }
  else {
    foreach my $elem ( @{$list} ) {

      DEBUG(
        sub {
          return Dumper(
            [ shape => $member->{shape},
              elem  => $elem
            ]
          );
        }
      );

      my $serialized_data = $self->serialize(
        shape   => $member->{shape},
        data    => $elem,
        service => $service,
        exists  => $exists,
      );

      next if !defined $serialized_data;

      push @{$result}, $serialized_data;

    }
  }

  return if !$exists && !@{$result};

  return $result;
}

########################################################################
sub serialize {
########################################################################
  my ( $self, %args ) = @_;

  my ( $shape, $service, $data, $exists )
    = @args{qw(shape service data exists)};

  $service ||= $self->get_service;

  DEBUG(
    sub {
      return Dumper(
        [ shape   => $shape,
          data    => $data,
          service => $service,
          exists  => $exists,
        ]
      );
    }
  );

  my $class = require_shape( $shape, $service );

  croak "unable to require class for $shape: $EVAL_ERROR\n"
    if !$class;

  my $this = $class->new;

  my $type = $this->get_type;

  DEBUG(
    sub {
      return Dumper(
        [ class   => $class,
          type    => $type,
          exists  => $exists,
          data    => $data,
          service => $service,
          this    => $this,
        ]
      );
    }
  );

  my %serializers = (
    structure => sub {
      return $self->_serialize_structure( $this, $data, $service, $exists );
    },

    list => sub {
      return $self->_serialize_list( $this, $data, $service, $exists );
    },

    string => sub {

      if ($data) {
        $data = ref $data ? $data->{content} : $data;
      }

      return $data
        if $data;

      return $exists ? $EMPTY : $data;
    },

    boolean => sub {
      $data = ref $data ? $data->{content} : $data;

      return
        if !defined $data;

      $data //= 0;

      if ( any { $data eq $_ } qw( 0 1 true false) ) {
        return {
          0     => JSON::false,
          1     => JSON::true,
          true  => JSON::true,
          false => JSON::false,
        }->{$data};
      }

      return $data;
    },

    integer => sub {
      return ref $data ? $data->{content} : $data;
    },

    long => sub {
      return ref $data ? $data->{content} : $data;
    },

    map => sub {
      return $data
        if !$this->{flattened};

      my $serialize           = {};
      my $key_location_name   = $this->{key}->{locationName};
      my $key_shape_name      = $this->{key}->{shape};
      my $value_location_name = $this->{value}->{locationName};
      my $value_shape_name    = $this->{value}->{shape};

      DEBUG(
        sub {
          return Dumper(
            [ type                => $type,
              data                => $data,
              locationName        => $this->{locationName},
              key_location_name   => $key_location_name,
              key_shape_name      => $key_shape_name,
              value_location_name => $value_location_name,
              value_shape_name    => $value_shape_name,
            ]
          );
        }
      );

      $data ||= [];

      if ( ref $data && reftype($data) ne 'ARRAY' ) {
        $data = [$data];
      }

      foreach my $item ( @{$data} ) {

        my $key = $self->serialize(
          service => $service,
          data    => $item->{$key_location_name}->{content},
          shape   => $key_shape_name,

        );

        my $value = $self->serialize(
          service => $service,
          data    => $item->{$value_location_name}->{content},
          shape   => $value_shape_name,
        );

        if ( defined $value ) {
          $serialize->{$key} = $value;
        }
      }

      return { $this->{locationName} => $serialize };
    },
    blob => sub {
      return $data;
    },
    timestamp => sub {
      $data = ref $data ? $data->{content} : $data;

      return
        if !$data;

      $data =~ s/^(.*)[.]000Z$/$1\+00:00/xsm;

      if ( $data =~ /^[\d.]+$/xsm ) {
        my $epoch = $data;

        my $nanoseconds;

        ( $epoch, $nanoseconds ) = split /[.]/xsm, $epoch;

        if ( defined $nanoseconds ) {
          $data = strftime( '%Y-%m-%dT%H:%M:%S.%%d%z', localtime $epoch );
          $data = sprintf $data, ".$nanoseconds" * 1_000_000;
        }
        else {
          $data = strftime( '%Y-%m-%dT%H:%M:%S%z', localtime $epoch );
        }

        if ( $data =~ /(\d{4})$/xsm ) {
          my ( $h, $m ) = unpack 'A2A2', $1;
          $data =~ s/\d{4}$/$h:$m/xsm;
        }

      }

      return $data;
    },
  );

  my $result;

  if ( $serializers{$type} ) {
    DEBUG(
      sub {
        return Dumper(
          [ type   => $type,
            data   => $data,
            exists => $exists
          ]
        );
      }
    );

    $result = $serializers{$type}->($exists);

    #    return
    #      if ref $result && reftype($result) eq 'HASH' && !%{$result};
    #
    #    return
    #      if ref $result && reftype($result) eq 'ARRAY' && !@{$result};

    DEBUG( sub { return Dumper( [ 'result', $result ] ) } );
  }
  else {
    DEBUG(
      sub {
        return Dumper(
          [ unknown_type => $type,
            this         => $this,
            data         => $data,
            exists       => $exists
          ]
        );
      }
    );

    $result = $self->serialize(
      shape   => $this->{shape},
      data    => $data,
      service => $service,
      exists  => $exists,
    );
  }

  return $result;
}

1;
