package Chart::Plotly::Trace::Splom::Dimension;
use Moose;
use MooseX::ExtraArgs;
use Moose::Util::TypeConstraints qw(enum union);
if ( !defined Moose::Util::TypeConstraints::find_type_constraint('PDL') ) {
    Moose::Util::TypeConstraints::type('PDL');
}

use Chart::Plotly::Trace::Splom::Dimension::Axis;

our $VERSION = '0.031';    # VERSION

# ABSTRACT: This attribute is one of the possible options for the trace splom.

sub TO_JSON {
    my $self       = shift;
    my $extra_args = $self->extra_args // {};
    my $meta       = $self->meta;
    my %hash       = %$self;
    for my $name ( sort keys %hash ) {
        my $attr = $meta->get_attribute($name);
        if ( defined $attr ) {
            my $value = $hash{$name};
            my $type  = $attr->type_constraint;
            if ( $type && $type->equals('Bool') ) {
                $hash{$name} = $value ? \1 : \0;
            }
        }
    }
    %hash = ( %hash, %$extra_args );
    delete $hash{'extra_args'};
    if ( $self->can('type') && ( !defined $hash{'type'} ) ) {
        $hash{type} = $self->type();
    }
    return \%hash;
}

has axis => ( is  => "rw",
              isa => "Maybe[HashRef]|Chart::Plotly::Trace::Splom::Dimension::Axis", );

has label => ( is            => "rw",
               isa           => "Str",
               documentation => "Sets the label corresponding to this splom dimension.",
);

has name => (
    is  => "rw",
    isa => "Str",
    documentation =>
      "When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.",
);

has templateitemname => (
    is  => "rw",
    isa => "Str",
    documentation =>
      "Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.",
);

has values => ( is            => "rw",
                isa           => "ArrayRef|PDL",
                documentation => "Sets the dimension values to be plotted.",
);

has valuessrc => ( is            => "rw",
                   isa           => "Str",
                   documentation => "Sets the source reference on plot.ly for  values .",
);

has visible => (
    is  => "rw",
    isa => "Bool",
    documentation =>
      "Determines whether or not this dimension is shown on the graph. Note that even visible false dimension contribute to the default grid generate by this splom trace.",
);

__PACKAGE__->meta->make_immutable();
1;

__END__

=pod

=encoding utf-8

=head1 NAME

Chart::Plotly::Trace::Splom::Dimension - This attribute is one of the possible options for the trace splom.

=head1 VERSION

version 0.031

=head1 SYNOPSIS

 use Chart::Plotly qw(show_plot);
 use Chart::Plotly::Trace::Splom;
 
 use Data::Dataset::Classic::Iris;
 
 my $convert_array_to_arrayref = sub {[@_]};
 my $iris = Data::Dataset::Classic::Iris::get(as => 'Data::Table');
 my $data = $iris->group(['species'],[$iris->header], [$convert_array_to_arrayref, $convert_array_to_arrayref, $convert_array_to_arrayref, $convert_array_to_arrayref, $convert_array_to_arrayref], [map { join "", map {ucfirst} split /_/, $_ } $iris->header], 0 );
 
 my @data_to_plot;
 my $iterator = $data->iterator();
 while (my $row = $iterator->()) {
     my $dimensions = [
         map { { label => $_, values => $row->{$_} } } qw(SepalLength SepalWidth PetalLength PetalWidth)
     ];
     push @data_to_plot, Chart::Plotly::Trace::Splom->new(
         name => $row->{species},
         dimensions => $dimensions
     );
 }
 
 show_plot([@data_to_plot]);

=head1 DESCRIPTION

This attribute is part of the possible options for the trace splom.

This file has been autogenerated from the official plotly.js source.

If you like Plotly, please support them: L<https://plot.ly/> 
Open source announcement: L<https://plot.ly/javascript/open-source-announcement/>

Full reference: L<https://plot.ly/javascript/reference/#splom>

=head1 DISCLAIMER

This is an unofficial Plotly Perl module. Currently I'm not affiliated in any way with Plotly. 
But I think plotly.js is a great library and I want to use it with perl.

=head1 METHODS

=head2 TO_JSON

Serialize the trace to JSON. This method should be called only by L<JSON> serializer.

=head1 ATTRIBUTES

=over

=item * axis

=item * label

Sets the label corresponding to this splom dimension.

=item * name

When used in a template, named items are created in the output figure in addition to any items the figure already has in this array. You can modify these items in the output figure by making your own item with `templateitemname` matching this `name` alongside your modifications (including `visible: false` or `enabled: false` to hide it). Has no effect outside of a template.

=item * templateitemname

Used to refer to a named item in this array in the template. Named items from the template will be created even without a matching item in the input figure, but you can modify one by making an item with `templateitemname` matching its `name`, alongside your modifications (including `visible: false` or `enabled: false` to hide it). If there is no template or no matching item, this item will be hidden unless you explicitly show it with `visible: true`.

=item * values

Sets the dimension values to be plotted.

=item * valuessrc

Sets the source reference on plot.ly for  values .

=item * visible

Determines whether or not this dimension is shown on the graph. Note that even visible false dimension contribute to the default grid generate by this splom trace.

=back

=head1 AUTHOR

Pablo Rodríguez González <pablo.rodriguez.gonzalez@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2019 by Pablo Rodríguez González.

This is free software, licensed under:

  The MIT (X11) License

=cut
