package Chart::Plotly::Trace::Surface::Lighting;
use Moose;
use MooseX::ExtraArgs;
use Moose::Util::TypeConstraints qw(enum union);
if ( !defined Moose::Util::TypeConstraints::find_type_constraint('PDL') ) {
    Moose::Util::TypeConstraints::type('PDL');
}

our $VERSION = '0.021';    # VERSION

# ABSTRACT: This attribute is one of the possible options for the trace surface.

sub TO_JSON {
    my $self       = shift;
    my $extra_args = $self->extra_args // {};
    my $meta       = $self->meta;
    my %hash       = %$self;
    for my $name ( sort keys %hash ) {
        my $attr = $meta->get_attribute($name);
        if ( defined $attr ) {
            my $value = $hash{$name};
            my $type  = $attr->type_constraint;
            if ( $type && $type->equals('Bool') ) {
                $hash{$name} = $value ? \1 : \0;
            }
        }
    }
    %hash = ( %hash, %$extra_args );
    delete $hash{'extra_args'};
    if ( $self->can('type') && ( !defined $hash{'type'} ) ) {
        $hash{type} = $self->type();
    }
    return \%hash;
}

has ambient => ( is            => "rw",
                 isa           => "Num",
                 documentation => "Ambient light increases overall color visibility but can wash out the image.",
);

has diffuse => ( is            => "rw",
                 isa           => "Num",
                 documentation => "Represents the extent that incident rays are reflected in a range of angles.",
);

has fresnel => (
    is  => "rw",
    isa => "Num",
    documentation =>
      "Represents the reflectance as a dependency of the viewing angle; e.g. paper is reflective when viewing it from the edge of the paper (almost 90 degrees), causing shine.",
);

has roughness => (
        is            => "rw",
        isa           => "Num",
        documentation => "Alters specular reflection; the rougher the surface, the wider and less contrasty the shine.",
);

has specular => (
         is            => "rw",
         isa           => "Num",
         documentation => "Represents the level that incident rays are reflected in a single direction, causing shine.",
);

__PACKAGE__->meta->make_immutable();
1;

__END__

=pod

=encoding utf-8

=head1 NAME

Chart::Plotly::Trace::Surface::Lighting - This attribute is one of the possible options for the trace surface.

=head1 VERSION

version 0.021

=head1 SYNOPSIS

 use Chart::Plotly qw(show_plot);
 use Chart::Plotly::Trace::Surface;
 use English qw(-no_match_vars);
 
 my $surface = Chart::Plotly::Trace::Surface->new(
     x => [ 0 .. 10 ],
     y => [ 0 .. 10 ],
     z => [
         map {
             my $y = $ARG;
             [ map { $ARG - $y * $y } ( 0 .. 10 ) ]
         } ( 0 .. 10 )
     ]
 );
 
 show_plot( [$surface] );

=head1 DESCRIPTION

This attribute is part of the possible options for the trace surface.

This file has been autogenerated from the official plotly.js source.

If you like Plotly, please support them: L<https://plot.ly/> 
Open source announcement: L<https://plot.ly/javascript/open-source-announcement/>

Full reference: L<https://plot.ly/javascript/reference/#surface>

=head1 DISCLAIMER

This is an unofficial Plotly Perl module. Currently I'm not affiliated in any way with Plotly. 
But I think plotly.js is a great library and I want to use it with perl.

=head1 METHODS

=head2 TO_JSON

Serialize the trace to JSON. This method should be called only by L<JSON> serializer.

=head1 ATTRIBUTES

=over

=item * ambient

Ambient light increases overall color visibility but can wash out the image.

=item * diffuse

Represents the extent that incident rays are reflected in a range of angles.

=item * fresnel

Represents the reflectance as a dependency of the viewing angle; e.g. paper is reflective when viewing it from the edge of the paper (almost 90 degrees), causing shine.

=item * roughness

Alters specular reflection; the rougher the surface, the wider and less contrasty the shine.

=item * specular

Represents the level that incident rays are reflected in a single direction, causing shine.

=back

=head1 AUTHOR

Pablo Rodríguez González <pablo.rodriguez.gonzalez@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Pablo Rodríguez González.

This is free software, licensed under:

  The MIT (X11) License

=cut
