/* Win.c */

#include "Sys.h"

#include <errno.h>
#include <signal.h>
#include <stdlib.h>

#include "Util.h"
#include "Main.h"
#include "Version.h"
#include "Recent.h"
#include "RCmd.h"
#include "LGets.h"
#include "GetPass.h"
#include "Serverio.h"

int gMultiLineMode = 0;
int gPendingNL = 0;

string gPrompt = kLineModePrompt;
string gFullPrompt = kLineModePrompt;
char *gSprintfBuf = NULL;
int gScreenWidth = 80;

extern longstring gLockFileName;
extern int gConnected, gVerbosity;
extern RemoteSiteInfo gRmtInfo;
extern string gHost;
extern int gStdout, gRealStdout, gOtherSessionRunning;
extern FILE *gTraceLogFile;
extern int gDebug, gTrace;
extern LineList gCmdHistory;
extern int gIsToTTY, gIsFromTTY, gStartup;
extern str16 gTelegrams;
extern char *rl_prompt;
extern char *rl_display_prompt;
extern int rl_visible_prompt_length;

void Exit(int exitStatus)
{
	if (gOtherSessionRunning == 0) {
		(void) UNLINK(gLockFileName);
	}
	exit(exitStatus);
}	/* Exit */

static void TTYWaitForReturn(void)
{
	int tty;
	int junk;

	tty = open("/dev/tty", O_RDWR);
	if (tty != -1) {
		write(tty, "[Hit return]", 12);
		read(tty, &junk, 1);
		close(tty);
	}
}	/* TTYWaitForReturn */


void Beep(int on)
{
	static time_t lastBeepTime = 0;
	time_t now;

	time(&now);

	/* Don't flood the user with beeps. Once per two seconds is reasonable. */
	if ((on > 0) && ((int) (now - lastBeepTime) > 1)) {
		fprintf(stderr, "\007");	/* ^G */
		fflush(stderr);
	}
	lastBeepTime = now;
}	/* Beep */


void FlushListWindow(void)
{
	fflush(stdout);
	fflush(stderr);		/* Overkill, since stderr _should_ be unbuffered. */
}	/* FlushListWindow */

void
BuildFullPrompt(int redisplay)
{
	char *save_prompt = rl_prompt;

	if (*gTelegrams) {
		STRNCPY(gFullPrompt, gTelegrams);
		STRNCAT(gFullPrompt, gPrompt);
	} else {
		STRNCPY(gFullPrompt, gPrompt);
	}
	if (gAtPrompt != kAtSubPrompt)
	    STRNCAT(gFullPrompt, kPromptTail);
	if (redisplay) {
		rl_display_prompt = rl_prompt = gFullPrompt;
		rl_visible_prompt_length = rl_expand_prompt(rl_prompt);
/*		rl_display_prompt = gFullPrompt;
		rl_prompt = "";
*/		rl_redisplay();
/*		rl_display_prompt = rl_prompt = save_prompt;
*/	}
}

/* Read a line of input, and axe the end-of-line. */
char *Gets(char *str, size_t size)
{
	Echo(stdin, 1);		/* Turn echo on, if it wasn't already. */
	BuildFullPrompt(0);
	return LineModeGets(gFullPrompt, str, size);
}	/* Gets */


void GetAnswer(char *prompt, char *answer, size_t siz, int noEcho)
{
	PTRZERO(answer, siz);

	if (noEcho)
		GetPass(prompt, answer, siz);
	else {
		StdioGets(prompt, answer, siz);
	}
}	/* GetAnswer */

void SetScreenInfo(void)
{
	string pr;
	char *cp;
	int len;
	size_t maxPCwdLen;

	if (gConnected) {
		STRNCPY(pr, gRmtInfo.nickName);
		len = (int) strlen(pr);
		if (len > kPromptLimit) {
			STRNCAT(pr, "...");
			cp = cp - (kPromptLimit - len - 3);
			STRNCAT(pr, cp);
		}
		SetPrompt(pr);
	} else {
		SetPrompt(kUseDefaultPrompt);
	}
}	/* SetScreenInfo */

/* Prints a message, if you have debbuging mode turned on. */

/*VARARGS*/
#ifndef HAVE_STDARG_H
void DebugMsg(va_alist)
	va_dcl
#else
void DebugMsg(char *fmt0, ...)
#endif
{
	va_list ap;
	char *fmt;

#ifndef HAVE_STDARG_H
	va_start(ap);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	fmt = fmt0;
#endif

	if (gDebug == kDebuggingOn) {
		(void) fprintf(kDebugStream, "#DB# ");
		(void) vfprintf(kDebugStream, fmt, ap);
		(void) fflush(kDebugStream);
	}
	if ((gTrace == kTracingOn) && (gTraceLogFile != NULL)) {
		(void) fprintf(gTraceLogFile, "#DB# ");
		(void) vfprintf(gTraceLogFile, fmt, ap);
		(void) fflush(gTraceLogFile);
	}
	va_end(ap);
}	/* DebugMsg */




/* This is similar to DebugMsg, but only writes to the debug log
 * file.  This is useful for putting messages in the log that
 * shouldn't show up on screen (i.e. would make a mess in visual
 * mode.
 */

/*VARARGS*/
#ifndef HAVE_STDARG_H
void TraceMsg(va_alist)
	va_dcl
#else
void TraceMsg(char *fmt0, ...)
#endif
{
	va_list ap;
	char *fmt;

#ifndef HAVE_STDARG_H
	va_start(ap);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	fmt = fmt0;
#endif

	if ((gTrace == kTracingOn) && (gTraceLogFile != NULL)) {
		(void) fprintf(gTraceLogFile, "#TR# ");
		(void) vfprintf(gTraceLogFile, fmt, ap);
		(void) fflush(gTraceLogFile);
	}
	va_end(ap);
}	/* TraceMsg */




/* Prints to our own standard output stream. */

/*VARARGS*/
#ifndef HAVE_STDARG_H
void PrintF(va_alist)
	va_dcl
#else
void PrintF(char *fmt0, ...)
#endif
{
	va_list ap;
	char *fmt;

#ifndef HAVE_STDARG_H
	va_start(ap);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	fmt = fmt0;
#endif

	/* If it's an important message, don't use this function, use
	 * EPrintF() instead.
	 */
	if (gVerbosity > kErrorsOnly) {
		(void) vsprintf(gSprintfBuf, fmt, ap);
		(void) write(gStdout, gSprintfBuf, strlen(gSprintfBuf));
	}
	if ((gTrace == kTracingOn) && (gTraceLogFile != NULL)) {
		(void) vfprintf(gTraceLogFile, fmt, ap);
		(void) fflush(gTraceLogFile);
	}
	va_end(ap);
}	/* PrintF */



/*VARARGS*/
#ifndef HAVE_STDARG_H
void BoldPrintF(va_alist)
	va_dcl
#else
void BoldPrintF(char *fmt0, ...)
#endif
{
	va_list ap;
	char *fmt;

#ifndef HAVE_STDARG_H
	va_start(ap);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	fmt = fmt0;
#endif

	if (gVerbosity > kErrorsOnly) {
		(void) vsprintf(gSprintfBuf, fmt, ap);
		(void) write(gStdout, gSprintfBuf, strlen(gSprintfBuf));
	}
	if ((gTrace == kTracingOn) && (gTraceLogFile != NULL)) {
		(void) vfprintf(gTraceLogFile, fmt, ap);
		(void) fflush(gTraceLogFile);
	}
	va_end(ap);
}	/* BoldPrintF */

/* Prints to stderr. */

/*VARARGS*/
#ifndef HAVE_STDARG_H
void EPrintF(va_alist)
	va_dcl
#else
void EPrintF(char *fmt0, ...)
#endif
{
	va_list ap;
	char *fmt;
	char *cp;

#ifndef HAVE_STDARG_H
	va_start(ap);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	fmt = fmt0;
#endif

	if (gVerbosity > kQuiet) {
		(void) vfprintf(stderr, fmt, ap);
	}
	if ((gTrace == kTracingOn) && (gTraceLogFile != NULL)) {
		/* Special hack so when progress meters use \r's we don't
		 * print them in the trace file.
		 */
		(void) vsprintf(gSprintfBuf, fmt, ap);
		for (cp = gSprintfBuf; ; ) {
			/* Replace all carriage returns with newlines. */
			cp = strchr(cp, '\r');
			if (cp == NULL)
				break;
			*cp++ = '\n';
		}
		fputs(gSprintfBuf, gTraceLogFile);
		(void) fflush(gTraceLogFile);
	}
	va_end(ap);
}	/* EPrintF */





/*VARARGS*/
#ifndef HAVE_STDARG_H
void Error(va_alist)
	va_dcl
#else
void Error(int pError0, char *fmt0, ...)
#endif
{
	va_list ap;
	char *fmt;
	int pError;
	longstring buf2;
	
#ifndef HAVE_STDARG_H
	va_start(ap);
	pError = va_arg(ap, int);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	fmt = fmt0;
	pError = pError0;
#endif

	if (gVerbosity > kQuiet) {
		(void) fprintf(stderr, "Error");
		if (gDebug == kDebuggingOn)
			(void) fprintf(stderr, "(%d)", errno);
		(void) fprintf(stderr, ": ");
		(void) vfprintf(stderr, fmt, ap);
		(void) fflush(stderr);
		if ((pError == kDoPerror) && (errno > 0))
			perror("Reason");
	}
	if ((gTrace == kTracingOn) && (gTraceLogFile != NULL)) {
		(void) fprintf(gTraceLogFile, "Error(%d): ", errno);
		(void) vfprintf(gTraceLogFile, fmt, ap);
#ifdef HAVE_STRERROR
		if ((pError == kDoPerror) && (errno > 0))
			(void) fprintf(gTraceLogFile, "Reason: %s\n", strerror(errno));
#endif
		(void) fflush(gTraceLogFile);
	}
	va_end(ap);
}	/* Error */



/*VARARGS*/
#ifndef HAVE_STDARG_H
void MultiLinePrintF(va_alist)
	va_dcl
#else
void MultiLinePrintF(char *fmt0, ...)
#endif
{
	va_list ap;
	char *fmt;

#ifndef HAVE_STDARG_H
	va_start(ap);
	fmt = va_arg(ap, char *);
#else
	va_start(ap, fmt0);
	fmt = fmt0;
#endif

	if (gVerbosity > kErrorsOnly) {
		(void) vsprintf(gSprintfBuf, fmt, ap);
		(void) write(gStdout, gSprintfBuf, strlen(gSprintfBuf));
	}
	if ((gTrace == kTracingOn) && (gTraceLogFile != NULL)) {
		(void) vfprintf(gTraceLogFile, fmt, ap);
		(void) fflush(gTraceLogFile);
	}

	va_end(ap);
}	/* MultiLinePrintF */


void SetPrompt(char *pr)
{
	string p;

	if ((pr == kUseDefaultPrompt)
		|| STREQ(pr, kLineModePrompt)) {
		STRNCPY(p, kLineModePrompt);
	} else {
		STRNCPY(p, pr);
	}
	STRNCPY(gPrompt, p);
}	/* SetPrompt */


void InitWindows(void)
{
	char *cp;

	cp = (char *) getenv("COLUMNS");
	if (cp != NULL)
		gScreenWidth = atoi(cp);

	/* Prompt will already be set if connected. */
	if (gConnected == 0)
		SetPrompt(kUseDefaultPrompt);
}	/* InitWindows */

/* eof */

