/* RCmd.c */

#include "Sys.h"

#include <ctype.h>
#include <setjmp.h>
#include <arpa/telnet.h>
#include "Util.h"
#include "RCmd.h"

#include <signal.h>
#include <setjmp.h>

#include "proto.h"
#include "Open.h"
#include "Main.h"

/* NOTE: There is more commentary at the bottom of this source file. */

/* User-settable verbosity level.
 * This isn't really used at the moment.
 */
int gVerbosity = kTerse;

int gNetworkTimeout = kDefaultNetworkTimeout;

jmp_buf	gTStrTimeOut;

extern FILE *gControlIn, *gControlOut;
extern int gDataSocket, gDoneApplication;
extern int gDebug, gConnected, gTrace;
extern jmp_buf gCmdLoopJmp;
extern int gPreferredDataPortMode, gReadingStartup;
extern int gAttemptingConnection;

/* The user can control the level of output printed by the program
 * with this.
 */
int SetVerbose(int newVerbose)
{
	int old;
	
	old = gVerbosity;
	gVerbosity = newVerbose;
	return (old);
}	/* SetVerbose */





static
void TimeoutGetTelnetString(int unused)
{
	longjmp(gTStrTimeOut, 1);
}	/* TimeoutGetTelnetString */



/* Since the control stream is defined by the Telnet protocol (RFC 854),
 * we follow Telnet rules when reading the control stream.  We use this
 * routine when we want to read a response from the host.
 */
int GetTelnetString(char *str, size_t siz, FILE *cin, FILE *cout)
{
	int c;
	size_t n;
	int eofError;
	char *cp;

	if ((cin == NULL) || (cout == NULL)) {
		eofError = -1;
		goto done;
	}

	if (setjmp(gTStrTimeOut) != 0) {
		Error(kDontPerror, "Host is not responding to commands, hanging up.\n");
		if (gAttemptingConnection == 0) {
			HangupOnServer();
			if (!gDoneApplication)
				longjmp(gCmdLoopJmp, 1);
		} else {
			/* Give up and return to Open(). */
			DoClose(0);
		}
		eofError = -1;
		goto done;
	}
	SIGNAL(SIGALRM, TimeoutGetTelnetString);
	alarm(gNetworkTimeout);

	cp = str;
	--siz;		/* We'll need room for the \0. */
	for (n = (size_t)0, eofError = 0; ; ) {
		c = fgetc(cin);
checkChar:
		if (c == EOF) {
eof:
			eofError = -1;
			break;
		} else if (c == '\r') {
			/* A telnet string can have a CR by itself.  But to denote that,
			 * the protocol uses \r\0;  an end of line is denoted \r\n.
			 */
			c = fgetc(cin);
			if (c == '\n') {
				/* Had \r\n, so done. */
				goto done;
			} else if (c == EOF) {
				goto eof;
			} else if (c == '\0') {
				c = '\r';
				goto addChar;
			} else {
				/* Telnet protocol violation! */
				goto checkChar;
			}
		} else if (c == '\n') {
			/* Really shouldn't get here.  If we do, the other side
			 * violated the TELNET protocol, since eoln's are CR/LF,
			 * and not just LF.
			 */
			goto done;
		} else if (c == IAC) {
			/* Since the control connection uses the TELNET protocol,
			 * we have to handle some of its commands ourselves.
			 * IAC is the protocol's escape character, meaning that
			 * the next character after the IAC (Interpret as Command)
			 * character is a telnet command.  But, if there just
			 * happened to be a character in the text stream with the
			 * same numerical value of IAC, 255, the sender denotes
			 * that by having an IAC followed by another IAC.
			 */
			
			/* Get the telnet command. */
			c = fgetc(cin);
			
			switch (c) {
				case WILL:
				case WONT:
					/* Get the option code. */
					c = fgetc(cin);
					
					/* Tell the other side that we don't want
					 * to do what they're offering to do.
					 */
					(void) fprintf(cout, "%c%c%c",IAC,DONT,c);
					(void) fflush(cout);
					break;
				case DO:
				case DONT:
					/* Get the option code. */
					c = fgetc(cin);
					
					/* The other side said they are DOing (or not)
					 * something, which would happen if our side
					 * asked them to.  Since we didn't do that,
					 * ask them to not do this option.
					 */
					(void) fprintf(cout, "%c%c%c",IAC,WONT,c);
					(void) fflush(cout);
					break;

				case EOF:
					goto eof;

				default:
					/* Just add this character, since it was most likely
					 * just an escaped IAC character.
					 */
					goto addChar;
			}
		} else {
addChar:
			/* If the buffer supplied has room, add this character to it. */
			if (n < siz) {
				*cp++ = c;				
				++n;
			}
		}
	}

done:
	*cp = '\0';
	alarm(0);
	return (eofError);
}	/* GetTelnetString */




/* This creates the complete command text to send, and writes it
 * on the stream.
 */
static
void SendCommand(char *cmdspec, va_list ap)
{
	longstring command;
	int result;

	(void) vsprintf(command, cmdspec, ap);
	if (strncmp(command, "PASS", SZ(4)))
		DebugMsg("RCmd:  \"%s\"\n", command);
	else
		DebugMsg("RCmd:  \"%s\"\n", "PASS xxxxxxxx");
	STRNCAT(command, "\n");	/* Use TELNET end-of-line. */
	if (gControlOut != NULL) {
		result = fputs(command, gControlOut);
		if (result < 0)
			Error(kDoPerror, "Could not write to control stream.\n");
		(void) fflush(gControlOut);
	}
}	/* SendCommand */

/*VARARGS*/
#ifndef HAVE_STDARG_H
int SendServer(va_alist)
        va_dcl
#else
int SendServer(char *cmdspec0, ...)
#endif
{
	va_list ap;
	char *cmdspec;
	int result;

#ifndef HAVE_STDARG_H
	va_start(ap);
	cmdspec = va_arg(ap, char *);
#else
	va_start(ap, cmdspec0);
	cmdspec = cmdspec0;
#endif
	if (gControlOut == NULL) {
		va_end(ap);
		return 0;
	}

	SendCommand(cmdspec, ap);
	va_end(ap);

	return 1;
}	/* SendServer */


/* eof */
