/* Open.c */

#include "Sys.h"

#ifdef SYSLOG
#	include <syslog.h>
#endif

#include "proto.h"
#include "Open.h"
#include "Util.h"
#include "GetPass.h"
#include "Cmds.h"
#include "RCmd.h"
#include "Recent.h"
#include "Getopt.h"
#include "Macro.h"
#include "Socket.h"
#include "Serverio.h"
#include "Main.h"

/* Port number we're connecting to. */
unsigned int gEmpirePort = kPortUnset;

/* Flag indicating whether we are connected to a remote host. */
int gConnected = 0;

/* A structure containing some extra information we've learned about
 * the remote host, so we don't need to find out the hard way each
 * time we do something.
 */
extern RemoteSiteInfo gRmtInfo;

/* Name of the host we're connected to. */
string gHost;

/* We keep track if we've logged in to the host yet.  It's possible to
 * be connected but not authorized (authenticated, whatever) to do anything.
 */
int gLoggedIn = 0;

/* Need to know if we are trying to connect and login, so if we get
 * hangup during that, we will come back to the open without longjmp'ing
 * out.
 */
int gAttemptingConnection = 0;

/* If we connect successfully, we'll want to save this port number
 * when we add a host file entry.
 */
int gPortNumberUsed;

/* Open.c externs */
extern char *gOptArg;
extern int gOptInd, gIsToTTY;
extern longstring gLocalCWD;
extern long gEventNumber;
extern int gTransferType, gCurType;
extern int gConnected, gHasPASV, gVerbosity;
extern int gDoneApplication;
extern string gEmailAddress, gAnonPassword;
extern UserInfo gUserInfo;
extern FILE *gLogFile;
extern char gIPStr[32];
extern string gVersion;
extern int gRmtInfoIsNew;

/* This is used primarily for non-anonymous logins.  We'll have to ask
 * the user some questions, like their username, password, etc.
 */
int LoginQuestion(
	char *prompt,
	char *answer,
	size_t siz,
	char *defanswer,
	int noEcho)
{
	string prompt2;
	
	/* Only do this if we have an empty string as the answer. */
	if (*answer == '\0') {
		if (defanswer != NULL)
			sprintf(prompt2, "%s [%s]: ", prompt, defanswer);
		else
			sprintf(prompt2, "%s: ", prompt);
		GetAnswer(prompt2, answer, siz, noEcho);
		if ((*answer == '\0') && (defanswer != NULL))
			Strncpy(answer, defanswer, siz);
	}
	return (*answer == '\0' ? (-1) : 0);
}	/* LoginQuestion */

int GotFromServer(int code, string expect)
{
	int result;

	InitServerLine();
	result = ReadServerLine(0);
	TraceServerLine();
	if (result == C_EXIT)
		return 0;

	if (gServerCode == code && !strcmp(gServerLine, expect))
		return 1;

	if (gServerCode == C_BADCMD && !strcmp(gServerLine, "Command client not found"))
		return 1;

	if (gServerCode == C_CMDERR || gServerCode == C_EXIT)
		PrintServerLine();
	else
		Error(kDontPerror, "Expected: '%d %s'\nGot: '%d %s'\n",
		      code, expect, gServerCode, gServerLine);

	return 0;
}

int Login()
{
	int result = -1;
	string expect;

	SendServer("client esh %s", gVersion);
	sprintf(expect, "talking to esh %s", gVersion);
	if (!GotFromServer(C_CMDOK, expect)) {
/* #*#		goto done; */
	}
	SendServer("user %s", gUserInfo.userName);
	sprintf(expect, "hello %s", gUserInfo.userName);
	if (!GotFromServer(C_CMDOK, expect))
		goto done;

	SendServer("coun %s", gRmtInfo.coun);
	sprintf(expect, "country name %s", gRmtInfo.coun);
	if (!GotFromServer(C_CMDOK, expect))
		goto done;

	SendServer("pass %s", gRmtInfo.pass);
	if (!GotFromServer(C_CMDOK, "password ok"))
		goto done;

	SendServer("play");
	if (!GotFromServer(C_INIT, "2"))
		goto done;
	PrintF("connected!\n");

	Slurp(0);

	SendServer("nation");
	SlurpNation(0);

	result = 0;

done:
	return result;
}	/* Login */




/* After closing a site, set or restore some things to their original
 * states.
 */
void PostCloseStuff(void)
{
#ifdef SYSLOG
	syslog (LOG_INFO, "%s disconnected from %s.", gUserInfo.userName, gHost);
#endif
	if (gLoggedIn) {
		gRmtInfo.port = gPortNumberUsed;
		gRmtInfo.nCalls++;
		time((time_t *) &gRmtInfo.lastCall);
		STRNCPY(gRmtInfo.lastIP, gIPStr);
		if (gEventNumber > 0L) {
			/* Only do these if we were not in batch mode (colon-mode). */
			(void) RunPrefixedMacro("close.", gRmtInfo.nickName);
			(void) RunPrefixedMacro("close.", "any");
		} else {
			/* Only do these if we are running colon mode. */
			(void) RunPrefixedMacro("colon.close.", gRmtInfo.nickName);
			(void) RunPrefixedMacro("colon.close.", "any");
		}
		SaveRemoteInfo(NULL);
	}
	gLoggedIn = 0;
	gAtPrompt = kAtEshPrompt;
	SetPrompt(kLineModePrompt);
}	/* PostCloseStuff */



/* Close the connection to the remote host and cleanup. */
void DoClose(int tryQUIT)
{
	if (tryQUIT != 0) {
		SendServer("quit");
		Slurp(1);	/* We are expecting EOF after this cmd. */
	}
	
	CloseControlConnection();
	PostCloseStuff();
}	/* DoClose */




int CloseCmd(int argc, char **argv)
{
	DoClose(gConnected);
	SetScreenInfo();	/* Need for line mode. */
	return kNoErr;
}	/* CloseCmd */




/* Given a pointer to an OpenOptions (structure containing all variables
 * that can be set from the command line), this routine makes sure all
 * the variables have valid values by setting them to their defaults.
 */
 
void InitOpenOptions(OpenOptions *openopt)
{
	PTRZERO(openopt, sizeof(OpenOptions));
	openopt->port = kPortUnset;
}	/* InitOpenOptions */





/* This is responsible for parsing the command line and setting variables
 * in the OpenOptions structure according to the user's flags.
 */

int GetOpenOptions(int argc, char **argv, int fromMain)
{
	OpenOptions openopt;
	int opt;

	/* First setup the openopt variables. */
	InitOpenOptions(&openopt);

	/* Tell Getopt() that we want to start over with a new command. */
	GetoptReset();
	while ((opt = Getopt(argc, argv, "n")) >= 0) {
		switch (opt) {		
		case 'n':
			/* don't do something (yet undetermined) */
			break;

		default:
			if (fromMain)
				break;
					
		usage:
			return kUsageErr;
		}
	}
	argv += gOptInd;
	argc -= gOptInd;

	if (argv[0] == NULL) {
		if (!fromMain)
			goto usage;
		else
			return kNoErr;
	} else {
		(void) STRNCPY(openopt.nickName, argv[0]);
		(void) STRNCPY(openopt.hostname, argv[0]);
	}

	if (argc > 1)
		openopt.port = atoi(argv[1]);
	if (argc > 2)
		STRNCPY(openopt.coun, argv[2]);
	if (argc > 3)
		STRNCPY(openopt.pass, argv[3]);

	if (GetRemoteInfo(&openopt))
		goto usage;
	return kNoErr;
}	/* GetOpenOptions */




/* Now that we're logged in, do some other stuff prior to letting the
 * user type away at the shell.
 */
void PostLoginStuff(RemoteSiteInfoPtr rsip)
{
	time_t now;
	string cbuf;
	char *p;
	
	gLoggedIn = 1;

	/* Since we connected okay, save this port number for later. */
	gPortNumberUsed = rsip->port;

	STRNCPY(gHost, rsip->name);
#ifdef SYSLOG
	syslog (LOG_INFO, "%s logged into %s.", gUserInfo.userName, gHost);
#endif
	if (gLogFile != NULL) {
		(void) time(&now);
		strcpy(cbuf, ctime(&now));
		if ((p = strchr(cbuf, '\n')) != 0)
			*p = 0;
		fprintf(gLogFile, "%s %s %d %s\n", cbuf, gHost, gPortNumberUsed,
			gRmtInfo.coun);
	}
	
	(void) RunPrefixedMacro("open.", "any");
	(void) RunPrefixedMacro("open.", gRmtInfo.nickName);
	SetScreenInfo();	/* Need for line mode. */
}	/* PostLoginStuff */



/* Given a properly set up RemoteSiteInfo, we try connecting to the site,
 * redialing if necessary, and do some initialization steps so the user
 * can send commands.
 */
int Open(RemoteSiteInfoPtr rsip)
{
	int					hErr;
	int					loginResult;

	/* If we haven't already setup our interactive shell, which
	 * would happen if you gave a host on the command line, then
	 * we need to do that now because we want the remote host's
	 * startup information to be displayed.
	 */
	Startup();
	
	PrintF("Trying to connect to %s...", rsip->name);
	FlushListWindow();

	gAttemptingConnection = 1;
	hErr = OpenControlConnection(rsip->name, rsip->port);
	if (hErr == kConnectErrFatal) {
		/* Irrecoverable error, so don't bother redialing.
		 * The error message should have already been printed
		 * from OpenControlConnection().
		 */
		DebugMsg("Cannot recover from open error %d.\n", hErr);
	} else if (hErr == kConnectNoErr) {
		/* We were hooked up successfully. */
		
		SetPostHangupOnServerProc(PostCloseStuff);
		loginResult = Login();
		
		if (loginResult == 0) {
			PostLoginStuff(rsip);
			gAttemptingConnection = 0;
			return(kNoErr);	/* Login okay, so done. */
		}
		/* Otherwise, an error during login occurred, so try again. */
	} else /* (hErr == kConnectErrReTryable), so redial. */ {
		/* Display error now. */
		FlushListWindow();
	}
	
	DoClose(gConnected);
	gAttemptingConnection = 0;

	/* Display error now. */
	FlushListWindow();

	return (kCmdErr);
}	/* Open */




int OpenCmd(int argc, char **argv)
{
	int result;

	/* If there is already a site open, close that one so we can
	 * open a new one.
	 */
	DoClose(gConnected);

	if ((result = GetOpenOptions(argc, argv, 0)) == kNoErr)
		result = Open(&gRmtInfo);

	return result;
}	/* OpenCmd */

/* Open.c */
