/* Mail.c
 *
 * Purpose:  Give the player a nice interface to reading teles & annos
 * Ken Stevens, 1995
 */

#include <ctype.h>
#include "proto.h"
#include "Sys.h"
#include "Util.h"
#include "Mail.h"
#include "Recent.h"
#include "Cmds.h"
#include "Serverio.h"

string gNationStatus;

extern int gConnected;
extern RemoteSiteInfo gRmtInfo;
extern int screenheight;
extern string gPrompt;

static int height;
static char *StatusChar = "N URD";

static unsigned char
ParseHeader(string header, MailPtr mp)
{
	char *p = header;
	char *d = strstr(p, "dated");
	char *end;
	unsigned char status;
	unsigned int day;
	unsigned int year;
	unsigned int cnum = 0;

	STRNCPY(mp->header, header);
	/* The following mail status convention comes from pei */
	p = end = mp->header + strlen(header) -1;
	while (p >= mp->header && isspace(*p))
		*p-- = '\0';
	if (end == p + 1)
		mp->status = kMessageRead;
	else if (end == p + 2)
		mp->status = kMessageUnread;
	else if (end == p + 3)
		mp->status = kMessageReplied;
	else
		mp->status = kMessageNew;
		
	if (d == NULL || sscanf(d + 6, "%3s %3s %u %8s %u",
				mp->weekday, mp->month, &day, mp->time,
				&year) != 5) {		
		strcpy(mp->weekday, "Bad");
		strcpy(mp->month, "Bad");
		day = 0;
		strcpy(mp->time, "--:--:--");
		year = 1965; /* The year I was born :-) */
	}
	mp->day = day;
	mp->year = year;

	if (sscanf(header, "> Telegram from %s (#%u)",
		   mp->country, &cnum) == 2) {
		mp->type = kMailTele;
	} else if (sscanf(header, "> BULLETIN from %s ",
			  mp->country) == 1) {
		mp->type = kMailBull;
	} else if (sscanf(header, ">%cAnnouncement from %s (#%u)", 
			  &status, mp->country, &cnum) == 2) {
		mp->type = kMailAnno;
		if (status == '*')
			mp->status = kMessageUnread;
	} else if (strstr(header, "> Production Report")) {
		mp->type = kMailProd;
		*mp->country = '\0';
	} else {
		mp->type = kMailUndef;
		*mp->country = '\0';
	}
	if (*mp->country)
		mp->country[strlen(mp->country) - 1] = '\0';
	mp->cnum = cnum;
} /* ParseHeader */

static void
NewMail(MailListPtr mlp)
{
	MailPtr mp;

	if (mlp->mail == NULL) {
		mlp->last = 0;
		mlp->size = kMailBlockSize;
		if ((mlp->mail = (MailPtr)calloc(mlp->size, sizeof(Mail))) == NULL)
		    OutOfMemory();
	} else {
		++mlp->last;
		if (mlp->last == mlp->size) {
			mlp->size += kMailBlockSize;
			if ((mlp->mail = (MailPtr)realloc(mlp->mail, (mlp->size * sizeof(Mail)))) == NULL)
				OutOfMemory();
		}
	}
	mp = mlp->mail + mlp->last;
	InitLineList(&mp->mailData);
	mp->subj[0] = '\0';
	mp->subjlen = 0;
} /* NewMail */

static void
InitMailList(MailListPtr mlp)
{
	*mlp->fName = '\0';
	mlp->fp = NULL;
	mlp->mail = NULL;
	mlp->last = -1;
	mlp->size = -1;
	height = screenheight - 2;
} /* InitMailList */

static void
DisposeMailListContents(MailListPtr mlp)
{
	int i;
	MailPtr mp;

	if (mlp->mail == NULL)
		return;

	for (mp = mlp->mail,i = 0; i <= mlp->last; ++mp, ++i)
		DisposeLineListContents(&mp->mailData);
	free(mlp->mail);
	InitMailList(mlp);
} /* DisposeMailListContents */

static int
ParseMailLine(longstring str, MailListPtr mlp, int *newp)
{
	MailPtr mp;

	if (*str == kMailHeaderChar) {
		NewMail(mlp);
		mp = mlp->mail + mlp->last;
		ParseHeader(str, mp);
		if (!*newp) {
			switch (mp->status) {
			case kMessageNew:
				*newp = 1;
			case kMessageUnread:
				mlp->current = mp - mlp->mail;
			}
		}
	} else if (mlp->last >= 0) {
		mp = mlp->mail + mlp->last;
		AddLine(&mp->mailData, str);
		if (mp->subjlen < kMailSubjLen - 1) {
			if (mp->subjlen) {
				STRNCAT(mp->subj, " ");
				STRNCAT(mp->subj, str);
			} else {
				STRNCPY(mp->subj, str);
			}
			mp->subjlen = strlen(mp->subj);
		}
	} else if (*str) {
		Error(kDontPerror,
		      "'%s' does not start with '%c'\n",
		      mlp->fName, kMailHeaderChar);
		return 1;
	} 
	return 0;
}



static void
ReadMailFromServer(char *command, MailListPtr mlp)
{

	mlp->fp = fopen(mlp->fName, "a");
	if (mlp->fp == NULL) {
		Error(kDontPerror, "Unable to open '%s' for append\n",
		      mlp->fName);
		*mlp->fName = '\0';
		return;
	}
	SendServer("%s y", command);
	do {
		InitServerLine();
		if (ReadServerLine(0) == C_EXIT)
			break;
		if (gServerCode == C_DATA) {
			if (strcmp(gServerLine, "You have a new telegram waiting ...") &&
			    strcmp(gServerLine, "You have a new announcement waiting ...") &&
			    strcmp(gServerLine, "No telegrams for you at the moment...") &&
			    strcmp(gServerLine, "No announcements for you at the moment...") &&
			    strcmp(gServerLine, "You lost your capital... better designate one"))
				fprintf(mlp->fp, "%s\n", gServerLine);
		} else if (gServerCode != C_PROMPT)
			ParseServerLine();
	} while (gServerCode != C_PROMPT);

	(void) fclose(mlp->fp);
	mlp->fp = NULL;
} /* ReadMailFromServer */


static void
ReadMailFile(MailListPtr mlp)
{
	longstring str;
	int new = 0;

	mlp->fp = fopen(mlp->fName, "r");
	if (mlp->fp == NULL)
		return;

	mlp->current = 0;
	while (1) {
		if ((fgets(str, ((int)sizeof(str)) - 1, mlp->fp)) == NULL)
			break;
		if (*str)
			str[strlen(str) - 1] = '\0';
		if (ParseMailLine(str, mlp, &new))
			break;
	}
	(void) fclose(mlp->fp);
	mlp->fp = NULL;
} /* ReadMailFile */

static void
WriteMailFile(MailListPtr mlp)
{
	int i;
	MailPtr mp;
	LinePtr lp;

	mlp->fp = fopen(mlp->fName, "w");
	if (mlp->fp == NULL) {
		Error(kDoPerror, "Unable to open %s for output", mlp->fName);
		*mlp->fName = '\0';
		return;
	}
	for (i = 0; i <= mlp->last; ++i) {
		mp = mlp->mail + i;
		switch (mp->status) {
		case kMessageDeleted:
			continue;
		case kMessageRead:
			STRNCAT(mp->header, " ");
			break;
		case kMessageUnread:
		case kMessageNew:
			STRNCAT(mp->header, "  ");
			break;
		case kMessageReplied:
			STRNCAT(mp->header, "   ");
			break;
		}
		fprintf(mlp->fp, "%s\n", mp->header);
		for (lp = mp->mailData.first; lp != NULL; lp = lp->next)
			fprintf(mlp->fp, "%s\n", lp->line);
	}
	(void) fclose(mlp->fp);
	mlp->fp = NULL;
}

PrintMailHeaders(MailListPtr mlp)
{
	int headstart;
	MailPtr mp;
	int i;
	string mark;

	if (mlp->last < height) {
		headstart = 0;
	} else {
		headstart = mlp->current - height/2;
		if (headstart < 0)
			headstart = 0;
	}
	for (i = headstart; i <= mlp->last && i < headstart + height; ++i) {
		mp = &mlp->mail[i];
		switch (mp->type) {
		case kMailProd:
			STRNCPY(mark, "----");
			break;
		case kMailBull:
			STRNCPY(mark, "   *");
			break;
		default:
			sprintf(mark, "%d", mp->cnum);
			break;
		}
		PrintF("%3d %c%c %4s %-12s %3s %3s %2d %8s %-30.30s\n",
		       i + 1,
		       i == mlp->current ? '>' : ' ',
		       StatusChar[mp->status],
		       mark,
		       mp->country,
		       mp->weekday,
		       mp->month,
		       mp->day,
		       mp->time,
		       mp->subj);
	}
}

static void
SetMailPrompt(MailListPtr mlp)
{
	static string prompt;

	sprintf(prompt, "(%d/%d)", mlp->current + 1, mlp->last + 1);
	SetPrompt(prompt);
}

static char *
StripSpace(char *line)
{
	char *cp;

	cp = line + strlen(line) - 1;
	while (cp >= line && isspace(*cp))
		*cp-- = '\0';
	for (cp = line; *cp && isspace(*cp); ++cp);
	return cp;
}

static void
ProcessDelete(char *s, MailListPtr mlp, int undelete)
{
	char *s2;
	int i, last;
	MailPtr mp;

	if (strchr(s, ' ')) {
		strtok(s, " ");
		while ((s2 = strtok(NULL, " ")) != NULL)
			ProcessDelete(s2, mlp, undelete);
		return;
	}
	if (sscanf(s, "%d-%d", &i, &last) == 2) {
		--i;
		--last;
		if (i > last ||
		    i < 0 ||
		    last > mlp->last) {
			PrintF("Bad range: %d-%d\n", i + 1, last + 1);
		} else {
			for (; i <= last; ++i) {
				mp = mlp->mail + i;
				kNewStatus(mp);
			}
		}
	} else if (sscanf(s, "%d", &i)) {
		++i;
		if (i < 0 || i > mlp->last)
			PrintF("Bad message number: %d\n", i + 1);
		else {
			mp = mlp->mail + i;
			kNewStatus(mp);
		}			
	} else if (*s == 'b') {
		for (i = 0; i <= mlp->last; ++i) {
			mp = mlp->mail + i;
			if (mp->type == kMailBull)
				kNewStatus(mp);
		}
	} else if (*s == 't') {
		for (i = 0; i <= mlp->last; ++i) {
			mp = mlp->mail + i;
			if (mp->type == kMailTele)
				kNewStatus(mp);
		}
	} else if (*s == 'p') {
		for (i = 0; i <= mlp->last; ++i) {
			mp = mlp->mail + i;
			if (mp->type == kMailProd)
				kNewStatus(mp);
		}
	} else if (*s == '*') {
		for (i = 0; i <= mlp->last; ++i) {
			mp = mlp->mail + i;
			kNewStatus(mp);
		}
	} else {
		PrintF("Bad argument to %s: '%s'\n",
		       undelete?"undelete":"delete",
		       s);
	}
}

static void
PrintMessage(MailListPtr mlp)
{
	MailPtr mp = mlp->mail + mlp->current;
	LinePtr lp;

	PrintF("Message #%d (%d lines)\n",
	       mlp->current + 1,
	       mp->mailData.nLines);
	if (mp->mailData.nLines <= height) {
		for (lp = mp->mailData.first; lp != NULL; lp = lp->next)
			PrintF("%s\n", lp->line);
	} else {
		for (lp = mp->mailData.first; lp != NULL; lp = lp->next)
			MultiLinePrintF("%s\n", lp->line);
	}
	if (mp->status == kMessageNew ||
	    mp->status == kMessageUnread)
		mp->status = kMessageRead;
}

static void
MailHelp()
{
	MultiLinePrintF("%s", "\
Commands are:\n\
  [num]               print message\n\
  h [num]             print the list of headers {starting at <num]\n\
  r [num]             reply to message [num]\n\
  f [num] <country>   forward message [num] to <country>\n\
  f [num]             follow up announcement [num]\n\
  s [num] <file>      save message [num] to a file called <file>\n\
  d [message list]    delete [message list]\n\
  u [message list]    undelete [message list]\n\
  z                     next screen of headers\n\
  Z                     previous screen of headers\n\
  q                     quit and save changes\n\
  x                     quit without saving changes\n\
Arguments shown in [] brackets are optional.  Default [num] and\n\
[message list] is current message.  <message list> may also be any of the following:\n\
    <n>                 message number <n>\n\
    <n>-<m>             messages <n> through <m>\n\
    <n1> <n2> ... <nm>  messages <n1>, <n2>, ..., and <nm>\n\
    p                   all Production Reports\n\
    b                   all BULLETINS\n\
    t                   all telegrams\n\
    *                   all messages\n\
  \n\
  The meanings of the message markers are:\n\
    N                   new\n\
    U                   unread\n\
    R                   replied to\n\
    D                   deleted\n");
}

static int
MailShell(MailListPtr mlp)
{
	MailPtr mp;
	char line[256];
	char *s;
	char c;
	int d;
	string prompt;

	STRNCPY(prompt, gPrompt);
	SetMailPrompt(mlp);
	if (Gets(line, sizeof(line)) == NULL)
		return 1;
	s = StripSpace(line);
	while (*s != 'q' && *s != 'x') {
		mp = mlp->mail + mlp->current;
		switch (c = *s) {
		case 'h':
			s = StripSpace(s + 1);
			if (sscanf(s, "%d", &d) == 1) {
				mlp->current = d - 1 + height / 2;
				if (mlp->current < 0)
					mlp->current = 0;
				if (mlp->current > mlp->last)
					mlp->current = mlp->last;
			}
			PrintMailHeaders(mlp);
			break;
		case 'z':
			if (mlp->last - mlp->current < height / 2) {
				PrintF("On last screen full of messages.\n");
			} else {
				mlp->current += (height - 2);
				if (mlp->current > mlp->last)
					mlp->current = mlp->last;
				PrintMailHeaders(mlp);
			}
			break;
		case 'Z':
			if (mlp->current <= height / 2) {
				PrintF("On first screen full of messages.\n");
			} else {
				mlp->current -= (height - 2);
				if (mlp->current < 0)
					mlp->current = 0;
				PrintMailHeaders(mlp);
			}
			break;
		case 's':
		case 'r':
		case 'f':
			PrintF("Not implemented yet.\n");
			break;
		case 'd':
		case 'u':
			s = StripSpace(s + 1);
			if (*s)
				ProcessDelete(s, mlp, c == 'u');
			else
				mp->status = kMessageDeleted;
			break;
		case '\0':
			if (mlp->current == mlp->last &&
			    mp->status != kMessageUnread &&
			    mp->status != kMessageNew)
				PrintF("No more messages.\n");
			else {
				if (mp->status == kMessageRead ||
				    mp->status == kMessageDeleted) {
					if (++mlp->current > mlp->last)
						mlp->current = mlp->last;
				}					
				PrintMessage(mlp);
			}
			break;
		default:
			if (sscanf(s, "%d", &d)) {
				if (d < 1 || d > mlp->last + 1)
					PrintF("Bad message number: %d\n",
					       d);
				else {
					mlp->current = d;
					PrintMessage(mlp);
				}
			} else {
				MailHelp();
			}
		}
		if (gConnected) {
			SendServer("ter 0,1"); /* Show we're not idle */
			QuietSlurp();
		}
		SetMailPrompt(mlp);
		if (Gets(line, sizeof(line)) == NULL)
			return 1;
		s = StripSpace(line);
	}
	SetPrompt(prompt);
	return (*s == 'x');
} /* MailShell */

MailMain(char *command, char *prefix, RemoteSiteInfoPtr rsip, char *message)
{
	MailList ml;
	MailPtr mp;

	InitMailList(&ml);
	Path(ml.fName, sizeof(ml.fName), rsip->dir, prefix);
	STRNCAT(ml.fName, rsip->nickName);
	
	if (gConnected && rsip == &gRmtInfo)
		ReadMailFromServer(command, &ml);
	ReadMailFile(&ml);
	if (ml.mail != NULL) {
		PrintMailHeaders(&ml);
		if (!MailShell(&ml)) {
			PrintF("Saving %s in %s\n", message, ml.fName);
			WriteMailFile(&ml);
		}
	} else {
		PrintF("No %s at the moment...\n", message);
	}
	DisposeMailListContents(&ml);
} /* MailMain */

int MailCmd(int argc, char **argv)
{
	RemoteSiteInfoPtr rsip;
	OpenOptions openopt;

	if (argc == 2) {
		InitOpenOptions(&openopt);
		STRNCPY(openopt.nickName, argv[1]);
		rsip = FindRemoteInfo(&openopt);
		if (rsip == NULL)
			return kUsageErr;
	} else if (gConnected) {
		rsip = &gRmtInfo;
	} else {
		return kUsageErr;
	}
	if (**argv == 'm') {
		if (*gNationStatus == 'D')
			return MailMain("read 0", "tele.", rsip, "telegrams");
		else
			return MailMain("read", "tele.", rsip, "telegrams");
	} else
		return MailMain("wire", "anno.", rsip, "announcements");
	return 0;
} /* MailCmd */

/* Unfortunately, this is needed by the mail parser */
void SlurpNation()
{
	do {
		InitServerLine();
		if (ReadServerLine(0) == C_EXIT)
			break;
		if (gServerCode == C_DATA)
			sscanf(gServerLine, "Nation status is %s", gNationStatus);
		else
			ParseServerLine();
	} while (gServerCode != C_PROMPT);
} /* SlurpNation */
