/* Cmds.c */

#include "Sys.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/telnet.h>
#include <netdb.h>
#include <errno.h>
#ifdef HAVE_NET_ERRNO_H
#	include <net/errno.h>
#endif
#include <setjmp.h>
#include <ctype.h>
#include <signal.h>

#include "Util.h"
#include "RCmd.h"
#include "Cmds.h"
#include "Cmdline.h"
#include "MakeArgv.h"
#include "Macro.h"
#include "Main.h"
#include "DateSize.h"
#include "Open.h"
#include "Getopt.h"
#include "Recent.h"
#include "Cpp.h"
#include "Prefs.h"
#include "Serverio.h"
#include "Socket.h"
#include "Tips.h"
#include "Version.h"
#include "readline/history.h"

/* Full path of the local current working directory. */
longstring gLocalCWD = "";

/* Full path of the previous local working directory. */
longstring gPrevLocalCWD = "";

/* Upon receipt of a signal during paging a local file, we jump here. */
jmp_buf gShellJmp;

/* Flag for debugging mode. */
#if (kAlpha > 0) || (kBeta > 0)
int gDebug = kDebuggingOff;
int gTrace = kTracingOn;
#else
int gDebug = kDebuggingOff;
int gTrace = kTracingOff;
#endif

extern int gNumCommands;
extern Command gCommands[];
extern int gVerbosity;
extern RemoteSiteInfo gRmtInfo;
extern UserInfo gUserInfo;
extern CppSymbol gCppSymbols[];
extern int gNumCppSymbols;
extern string gVersion;
extern longstring gPager;
extern int gDoneApplication, gConnected;
extern FILE *gTraceLogFile;
extern int gStdout;
extern MacroNodePtr gFirstMacro;
extern int gNumGlobalMacros, gOtherSessionRunning;
extern char *gOptArg;
extern int gOptInd;
extern struct hostent *GetHostEntry(char *host, struct in_addr *ip_address);
extern int gNumRecents;
extern RemoteSiteInfoPtrList gHostNickNames;


/*ARGSUSED*/
static void SigLocalPage(int sigNum)
{
	longjmp(gShellJmp, 1);
}	/* SigLocalPage */




int LocalPageCmd(int argc, char **argv)
{
	volatile FILE *fp, *pager;
	int i, errs, opt;
	longstring pageCmd;
	volatile LineList globFiles;
	volatile int useBuiltIn;
	volatile Sig_t si, sp;
	string str;

	GetoptReset();
	useBuiltIn = 0;
	while ((opt = Getopt(argc, argv, "bp")) >= 0) {
		switch (opt) {
			case 'b':
				useBuiltIn = 1;
				break;
			case 'p':
				useBuiltIn = 0;
				break;
			default:
				return (kUsageErr);
		}
	}
	argv += gOptInd;
	argc -= gOptInd;

	si = SIGNAL(SIGINT, SigLocalPage);
	sp = SIGNAL(SIGPIPE, SigLocalPage);
	errs = 0;
	
	if (useBuiltIn == 0) {
		if (gPager[0] == '\0') {
			EPrintF("You haven't specified a program to use as a pager.\n");
			EPrintF("You can set this from the preferences screen (prefs command).\n");
			errs = -1;
			goto done;
		}
	}
	
	for (i=0; i<argc; i++) {
			fp = fopen(argv[i], "r");
			if (fp == NULL) {
				Error(kDoPerror, "Can't open %s.\n", argv[i]);
				--errs;
			} else if (useBuiltIn == 1) {
				PrintF("*** %s ***\n", argv[i]);
				if (setjmp(gShellJmp) != 0) {
					/* Command was interrupted. */
					(void) SIGNAL(SIGINT, SIG_IGN);
					fclose((FILE *) fp);
					--errs;
					goto done;
				} else {
					while (fgets(str, ((int) sizeof(str)) - 1, (FILE *)fp) != NULL)
						MultiLinePrintF("%s", str);
				}
				(void) SIGNAL(SIGINT, SIG_IGN);
				fclose((FILE *) fp);
			} else {
				STRNCPY(pageCmd, gPager);
				STRNCAT(pageCmd, " ");
				STRNCAT(pageCmd, argv[i]);
				pager = (volatile FILE *) 0;
				if (setjmp(gShellJmp) != 0) {
					/* Command was interrupted. */
					(void) SIGNAL(SIGINT, SIG_IGN);
					(void) SIGNAL(SIGPIPE, SIG_IGN);
					if (pager != ((volatile FILE *) 0))
						PClose((FILE *) pager);
					fclose((FILE *) fp);
					--errs;
					goto done;
				} else {
					pager = POpen(pageCmd, "w", 1);
					while (fgets(str, ((int) sizeof(str)) - 1, (FILE *)fp) != NULL)
						fputs(str, (FILE *) pager);
					PClose((FILE *) pager);
					fclose((FILE *) fp);
				}
			}
		}

 done:
	(void) SIGNAL(SIGINT, si);
	(void) SIGNAL(SIGPIPE, sp);
	Beep(0);	/* User should be aware that it took a while, so no beep. */
	return (errs);
}	/* LocalPageCmd */







int LocalPwdCmd(int argc, char **argv)
{
	(void) GetCWD(gLocalCWD, sizeof(gLocalCWD));
	PrintF("Current local directory is %s.\n", gLocalCWD);
	return 0;
}	/* LocalPwdCmd */










/* cd to 'dir' on the local host.  The dir specified may have glob
 * characters, or ~stuff in it also.
 */
int DoLocalChdir(char *dir, int quiet)
{
	int result;

	if ((result = chdir(dir)) < 0) {
		Error(kDoPerror, "Could not change local directory to %s.\n", dir);
	} else {
		(void) GetCWD(gLocalCWD, sizeof(gLocalCWD));
		if (!quiet)
			PrintF("Current local directory is %s.\n", gLocalCWD);
	}

	return (result);
}	/* DoLocalChdir */



/* Stub command that lcd's to the appropriate directory, or if none
 * was supplied, carry on the tradition and make that the same as
 * lcd'ing to the home directory.
 */
int LocalChdirCmd(int argc, char **argv)
{
	int result;
	char *cp;
	longstring str;

	if (argc < 2)
		cp = gUserInfo.home;
	else if (STREQ(argv[1], "-") && (gPrevLocalCWD[0] != '\0'))
		cp = gPrevLocalCWD;
	else
		cp = argv[1];
	STRNCPY(str, gLocalCWD);
	result = DoLocalChdir(cp, 0);	/* Sets gLocalCWD? #*# */
	if (result == 0)
		STRNCPY(gPrevLocalCWD, str);
	return (result);
}	/* LocalChdirCmd */




/* Changes the debugging status, or prints some extra debugging
 * info depending on the parameter given.
 */
int DebugCmd(int argc, char **argv)
{
	char *cp;
	int i;

	if (argc == 1) {
		PrintF("Debug Mode = %d.  Trace Mode = %d.\n", gDebug, gTrace);
	} else {
#if (LIBMALLOC == FAST_MALLOC)
		if (STREQ(argv[1], "memchk")) {
			struct mallinfo mi;
		
			mi = mallinfo();
			PrintF("\
total space in arena:               %d\n\
number of ordinary blocks:          %d\n\
number of small blocks:             %d\n\
number of holding blocks:           %d\n\
space in holding block headers:     %d\n\
space in small blocks in use:       %d\n\
space in free small blocks:         %d\n\
space in ordinary blocks in use:    %d\n\
space in free ordinary blocks:      %d\n\
cost of enabling keep option:       %d\n",
				mi.arena,
				mi.ordblks,
				mi.smblks,
				mi.hblks,
				mi.hblkhd,
				mi.usmblks,
				mi.fsmblks,
				mi.uordblks,
				mi.fordblks,
				mi.keepcost
			);
			return 0;
		}
#endif
#if (LIBMALLOC == DEBUG_MALLOC)
		if (STREQ(argv[1], "memchk")) {
			PrintF("malloc_chain_check: %d\n\n", malloc_chain_check(0));
			PrintF("malloc_inuse: %lu\n", malloc_inuse(NULL));
			return 0;
		}
		if (STREQ(argv[1], "memdump")) {
			malloc_dump(1);
			return 0;
		}
#endif
		for (cp = argv[1]; (*cp != '\0') && isdigit(*cp); )
			++cp;
		if (*cp == '\0') {
			gDebug = atoi(argv[1]);
			return 0;
		} else if (ISTREQ(argv[1], "macro")) {
			/* Dump specified macro, or if NULL, all of them. */
			DumpMacro(argv[2]);
		} else if (ISTREQ(argv[1], "segv")) {
			/* Intentionally bomb the program... */
			*((int *) 0) = 99;
		} else if (ISTREQ(argv[1], "multi")) {
			for (i=1; i<=60; i++)
				MultiLinePrintF("This is line %d.\n", i);
		} else if (ISTREQ(argv[1], "trace")) {
			if (argc > 2)
				gTrace = atoi(argv[2]);
			else
				gTrace = !gTrace;
			if (gTrace) {
				if (gTraceLogFile == NULL)
					OpenTraceLog();
			} else {
				if (gTraceLogFile != NULL)
					CloseTraceLog();
			}
		} else if (ISTREQ(argv[1], "tips")) {
			/* Dump all the tips. */
			PrintAllTips();
		}
	}
	return 0;
}	/* DebugCmd */


int PrintMacro(int argc, char **argv)
{
	if (argc == 2)
		DumpMacro(argv[1]);
	else
		DumpMacro(NULL);
	return 0;
}



/* Sets the verbosity level of our informational messages. */
int VerboseCmd(int argc, char **argv)
{
	int newVerbose;

	if (argc == 1)
		PrintF("Verbosity = %d.\n", gVerbosity);
	else {
		newVerbose = atoi(argv[1]);
		if (newVerbose < kQuiet)
			newVerbose = kQuiet;
		else if (newVerbose > kVerbose)
			newVerbose = kVerbose;
		(void) SetVerbose(newVerbose);
	}
	return 0;
}	/* VerboseCmd */




void DoQuit(int exitStatus)
{
	/* Only do this once, in case we get caught with infinite recursion. */
	if (++gDoneApplication <= 1) {
		if (gConnected)
			DoClose(1);
		(void) RunPrefixedMacro("quit.", "esh");
		(void) RunPrefixedMacro("end.", "esh");
		if (gOtherSessionRunning == 0) {
			WriteRemoteInfoFile();
			CloseLogs();
			WritePrefs();
		}
	}
	Exit(exitStatus);
}	/* DoQuit */



int QuitCmd(int argc, char **argv)
{
	DoQuit(kExitNoErr);
	/*NOTREACHED*/
	return 0;
}	/* QuitCmd */


int SendCmd(int argc, char **argv)
{
	SendServer("%s", CMDLINEFROMARGS(argc, argv));
	Slurp(0);

	return 0;
}

int IgnoreCmd(int argc, char **argv)
{
	return 0;
}


/* Prints the command list, or gives a little more detail about a
 * specified command.
 */
int HelpCmd(int argc, char **argv)
{
	CommandPtr c;
	MacroNodePtr macp;
	int showall = 0, helpall = 0;
	char *arg;
	int i, j, k, n;
	int nRows, nCols;
	int nCmds2Print;
	int screenColumns;
	int len, widestName;
	char *cp, **cmdnames, spec[16];
	CMNamePtr cm;

	if (argc == 2) {
		showall = (STREQ(argv[1], "showall"));
		helpall = (STREQ(argv[1], "helpall"));
	}
	if (argc == 1 || showall) {
		MultiLinePrintF("\
Commands may be abbreviated.  'help showall' shows aliases, invisible and\n\
unsupported commands.  'help <command>' gives a brief description of <command>.\n\n");

		/* First, see how many commands we will be printing to the screen.
		 * Unless 'showall' was given, we won't be printing the hidden
		 * (i.e. not very useful to the end-user) commands.
		 */
		c = gCommands;
		nCmds2Print = 0;
		for (n = 0; n < gNumCommands; c++, n++)
			if ((!iscntrl(c->name[0])) && (!(c->flags & kCmdHidden) || showall))
				nCmds2Print++;

		if ((cmdnames = (char **) malloc(sizeof(char *) * nCmds2Print)) == NULL)
			OutOfMemory();

		/* Now form the list we'll be printing, and noting what the maximum
		 * length of a command name was, so we can use that when determining
		 * how to print in columns.
		 */
		c = gCommands;
		i = 0;
		widestName = 0;
		for (n = 0; n < gNumCommands; c++, n++) {
			if ((!iscntrl(c->name[0])) && (!(c->flags & kCmdHidden) || showall)) {
				cmdnames[i++] = c->name;
				len = (int) strlen(c->name);
				if (len > widestName)
					widestName = len;
			}
		}

		if ((cp = (char *) getenv("COLUMNS")) == NULL)
			screenColumns = 80;
		else
			screenColumns = atoi(cp);

		/* Leave an extra bit of whitespace for the margins between columns. */
		widestName += 2;
		
		nCols = (screenColumns + 0) / widestName;
		nRows = nCmds2Print / nCols;
		if ((nCmds2Print % nCols) > 0)
			nRows++;

		for (i = 0; i < nRows; i++) {
			for (j = 0; j < nCols; j++) {
				k = nRows * j + i;
				if (k < nCmds2Print) {
					(void) sprintf(spec, "%%-%ds",
						(j < nCols - 1) ? widestName : widestName - 2
					);
					MultiLinePrintF(spec, cmdnames[k]);
				}
			}
			MultiLinePrintF("\n");
		}
		free(cmdnames);
		
		if (gNumGlobalMacros > 0) {
			MultiLinePrintF("\nMacros:\n\n");
			/* Now do the same for the macros. */
			if ((cmdnames = (char **) malloc(sizeof(char *) * gNumGlobalMacros)) == NULL)
				OutOfMemory();
	
			/* Form the list we'll be printing, and noting what the maximum
			 * length of a command name was, so we can use that when determining
			 * how to print in columns.
			 */
			macp = gFirstMacro;
			widestName = 0;
			for (i = 0; macp != NULL; macp = macp->next) {
					cmdnames[i++] = macp->name;
					len = (int) strlen(macp->name);
					if (len > widestName)
						widestName = len;
			}
			nCmds2Print = i;
	
			/* Leave an extra bit of whitespace for the margins between columns. */
			widestName += 2;
			
			nCols = (screenColumns + 0) / widestName;
			nRows = nCmds2Print / nCols;
			if ((nCmds2Print % nCols) > 0)
				nRows++;
	
			for (i = 0; i < nRows; i++) {
				for (j = 0; j < nCols; j++) {
					k = nRows * j + i;
					if (k < nCmds2Print) {
						(void) sprintf(spec, "%%-%ds",
							(j < nCols - 1) ? widestName : widestName - 2
						);
						MultiLinePrintF(spec, cmdnames[k]);
					}
				}
				MultiLinePrintF("\n");
			}
			free(cmdnames);
		}
	} else if (helpall) {
		/* Really intended for me, so I can debug the help strings. */
		for (c = gCommands, n = 0; n < gNumCommands; c++, n++) {
			PrintCmdHelp(c);
			PrintCmdUsage(c);
		}
	} else {
		/* For each command name specified, print its help stuff. */
		while (--argc > 0) {
			arg = *++argv;
			cm = GetCommandOrMacro(arg);
			if (cm == kAmbiguousName)
				MultiLinePrintF("\"%s:\" Ambiguous command or macro name.\n", arg);
			else if (cm == kNoName)
				MultiLinePrintF("\"%s:\" Invalid command or macro name.\n", arg);
			else if (cm->isCmd) {
				c = cm->u.cmd;
				PrintCmdHelp(c);
				PrintCmdUsage(c);
			} else {
				MultiLinePrintF("\"%s\" is a macro, so no help is available.\n", arg);
			}
		}
	}
	return 0;
}									   /* HelpCmd */


int HistCmd(int argc, char **argv)
{
          register HIST_ENTRY **the_list;
          register int i;
	  int num;
	  int last;

	  num = kDefaultHistList;
	  if (argc == 2)
		  num = atoi(argv[1]);

	  remove_history(where_history());
	  last = where_history();
	  if (num < 0 || num > last)
		  num = last;

          the_list = history_list();
          if (the_list)
		  for (i = last - num; the_list[i]; i++)
			  PrintF("%d: %s\n", i + history_base, the_list[i]->line);
	  return 0;
}


int VersionCmd(int argc, char **argv)
{
	int i;
	longstring line;
	longstring sym;
	char num[32];
	int symsOnLine;
	int symLen;
	int lineLen;
	
	MultiLinePrintF("Version:       %s\n", gVersion);
	MultiLinePrintF("Author:        Ken Stevens, Empire.Net (children@empire.net)\n");
	MultiLinePrintF("Archived at:   ftp://ftp.empire.net/pub/empire/player/clients/\n");

#ifdef __DATE__
	MultiLinePrintF("Compile Date:  %s\n", __DATE__);
#endif
#ifdef MK
	MultiLinePrintF("MK: %s\n", MK);
#endif

	MultiLinePrintF("\nCompile options:\n\n");
	line[0] = '\0';
	symsOnLine = 0;
	lineLen = 0;
	for (i=0; i<gNumCppSymbols; i++) {
		STRNCPY(sym, gCppSymbols[i].name);
		if (gCppSymbols[i].symType == 0) {
			if (gCppSymbols[i].l != 1L) {
				sprintf(num, "=%ld", gCppSymbols[i].l);
				STRNCAT(sym, num);
			}
			STRNCAT(sym, "  ");
		} else {
			STRNCAT(sym, "=\"");
			STRNCAT(sym, gCppSymbols[i].s);
			STRNCAT(sym, "\"  ");
		}
		symLen = (int) strlen(sym);
		if (lineLen + symLen > 79) {
			MultiLinePrintF("%s\n", line);
			line[0] = '\0';
			symsOnLine = 0;
			lineLen = 0;
		}
		STRNCAT(line, sym);
		++symsOnLine;
		lineLen += symLen;
	}
	if (symsOnLine) {
		MultiLinePrintF("%s\n", line);
	}
	return 0;
}	/* VersionCmd */




int ShellCmd(int argc, char **argv)
{
	int result;
	char *theShell;
	char *cmdLine;
	VSig_t si;

	si = (VSig_t) 0;
	if ((theShell = (char *) getenv("SHELL")) == NULL)
		theShell = gUserInfo.shell;
	if (theShell == NULL)
		theShell = "/bin/sh";

	if (setjmp(gShellJmp) != 0) {
		/* Command was interrupted. */
		(void) SIGNAL(SIGINT, SIG_IGN);
		result = 1;
	} else {
		si = SIGNAL(SIGINT, SigLocalPage);
		if (argc < 2)
			result = system(theShell);
		else {
			/* We have a hack where we keep a copy of the original
			 * command line before parsing at position argc + 2.
			 */
			cmdLine = CMDLINEFROMARGS(argc, argv);
			
			/* Skip the : and whitespace after it. */
			while ((*cmdLine == ':') || isspace(*cmdLine))
				cmdLine++;
			result = system(cmdLine);
		}
	}
	if (si != (VSig_t) 0)
		(void) SIGNAL(SIGINT, si);
	return result;
}	/* ShellCmd */




int EchoCmd(int argc, char **argv)
{
	longstring str;
	int i;
	int noNewLine = 0;

	for (i=1; i<argc; i++) {
		(void) FlagStrCopy(str, sizeof(str), argv[i]);
		/* The above writes an @ sign after the nul if we were supposed
		 * to not issue a final newline.
		 */
		noNewLine = (str[strlen(str) + 1] == '@');
		PrintF("%s%s", (i > 1 ? " " : ""), str);
	}
	if (!noNewLine)
		PrintF("\n");
	return 0;
}	/* EchoCmd */


void MyInetAddr(char *dst, size_t siz, char **src, int i)
{
	struct in_addr *ia;
	char *cp;
	
	Strncpy(dst, "???", siz);
	if (src != (char **) 0) {
		ia = (struct in_addr *) src[i];
		cp = inet_ntoa(*ia);
		if ((cp != (char *) 0) && (cp != (char *) -1))
			Strncpy(dst, cp, siz);
	}
}	/* MyInetAddr */


/* On entry, you should have 'host' be set to a symbolic name (like
 * cse.unl.edu), or set to a numeric address (like 129.93.3.1).
 * If the function fails, it will return NULL, but if the host was
 * a numeric style address, you'll have the ip_address to fall back on.
 */

struct hostent *GetHostEntry(char *host, struct in_addr *ip_address)
{
	struct in_addr ip;
	struct hostent *hp;
	
	/* See if the host was given in the dotted IP format, like "36.44.0.2."
	 * If it was, inet_addr will convert that to a 32-bit binary value;
	 * it not, inet_addr will return (-1L).
	 */
	ip.s_addr = inet_addr(host);
	if (ip.s_addr != INADDR_NONE) {
		hp = gethostbyaddr((char *) &ip, (int) sizeof(ip), AF_INET);
	} else {
		/* No IP address, so it must be a hostname, like ftp.wustl.edu. */
		hp = gethostbyname(host);
		if (hp != NULL)
			ip = * (struct in_addr *) hp->h_addr_list;
	}
	if (ip_address != NULL)
		*ip_address = ip;
	return (hp);
}	/* GetHostEntry */


int LookupCmd(int argc, char **argv)
{
	int i, j;
	struct hostent *hp;
	char *host, **cpp;
	struct in_addr ip_address;
	int shortMode, opt;
	char ipStr[16];

	shortMode = 1;
	
	GetoptReset();
	while ((opt = Getopt(argc, argv, "v")) >= 0) {
		if (opt == 'v')
			shortMode = 0;
		else
			return kUsageErr;
	}

	for (i=gOptInd; i<argc; i++) {
		hp = GetHostEntry((host = argv[i]), &ip_address);
		if ((i > gOptInd) && (shortMode == 0))
			PrintF("\n");
		if (hp == NULL) {
			PrintF("Unable to get information about site %s.\n", host);
		} else if (shortMode) {
			MyInetAddr(ipStr, sizeof(ipStr), hp->h_addr_list, 0);
			PrintF("%-40s %s\n", hp->h_name, ipStr);
		} else {
			PrintF("%s:\n", host);
			PrintF("    Name:     %s\n", hp->h_name);
			for (cpp = hp->h_aliases; *cpp != NULL; cpp++)
				PrintF("    Alias:    %s\n", *cpp);
			for (j = 0, cpp = hp->h_addr_list; *cpp != NULL; cpp++, ++j) {
				MyInetAddr(ipStr, sizeof(ipStr), hp->h_addr_list, j);
				PrintF("    Address:  %s\n", ipStr);	
			}
		}
	}
	return 0;
}	/* LookupCmd */

int GameCmd(int argc, char **argv)
{
	RemoteSiteInfoPtr rsip;
	OpenOptions openopt;
	int i;

	if (gNumRecents == 0) {
		PrintF("There are no games listed in your hosts file.\n");
		return kCmdErr;
	}
	
	if (argc == 2) {
		InitOpenOptions(&openopt);
		STRNCPY(openopt.nickName, argv[1]);
		if ((rsip = FindRemoteInfo(&openopt)) == NULL) {
			PrintF("No such game: '%s'\n", argv[1]);
			return kCmdErr;
		}
		PrintRemoteSiteInfo(rsip, 0);
		return 0;
	}
	for (i = 0; i < gNumRecents; i++)
		PrintRemoteSiteInfo(gHostNickNames[i], i);
	return 0;	
}
