# status.tcl --
#
#	Implements the player state display across the top of the
#	Main Window.
#

namespace eval NSStatus {

variable Priv

# NSStatus::InitModule --
#
#	One-time-only-ever initialization.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitModule {} {

	variable Priv

	set widget [Global main,widget]

	set font [Global font,fixed,small]
	set fontWid [font measure $font "W"]
	set statusWid [expr $fontWid * 11]
	set widgetWid [winfo reqwidth $widget]
	set Priv(numPerRow) [expr $widgetWid / $statusWid]

	set data {
		statusText fill
		statusBG background
		statusBL bevellight
		statusBD beveldark
		statusText2 fill2
		statusBG2 background2
		statusBL2 bevellight2
		statusBD2 beveldark2
	}
	foreach {name varname} $data {
		set $varname [Value $name]
	}

	set Priv(names) [list state winner cut stun hunger blind confused \
		afraid poisoned speed study]

	foreach status $Priv(names) {

		set itemId [$widget create text -width $statusWid -height 16 \
			-bevel yes -visible no -fill $fill -fill2 $fill2 -anchor nw \
			-background $background -background2 $background2 \
			-bevellight $bevellight -bevellight2 $bevellight2 \
			-beveldark $beveldark -beveldark2 $beveldark2 \
			-justify center]

		set Priv($status,itemId) $itemId
		set Priv($status,showing) 0

		# Hack -- Big status message
		set Priv($status,text) [angband player status $status]
	}

	set Priv(statusWid) $statusWid

	# Catch "Status" events
	qebind $widget <Status> {
		NSStatus::SetText %d [angband player status %d]
	}

	set data {
		statusText -fill
		statusBG -background
		statusBL -bevellight
		statusBD -beveldark
		statusText2 -fill2
		statusBG2 -background2
		statusBL2 -bevellight2
		statusBD2 -beveldark2
	}
	foreach {name option} $data {
		NSValueManager::AddClient $name "NSStatus::ColorChanged $name $option"
	}

	# Hack -- Big status message
	InitStatusMessage
}

# NSStatus::SetText --
#
#	Called as a qebind <Status> script. Sets the text of one of
#	the status items.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc SetText {status text} {

	variable Priv

	set widget [Global main,widget]

	set arrange 0

	if {[string compare $text ""]} {

		$widget itemconfigure $Priv($status,itemId) -text $text

		if !$Priv($status,showing) {
			set Priv($status,showing) 1
			$widget itemconfigure $Priv($status,itemId) -visible yes
			set arrange 1
		}

		# Hack -- Big status message
		if {[string compare $text $Priv($status,text)]} {
			set Priv($status,text) $text
			SetStatusMessage $text
		}

	} elseif $Priv($status,showing) {

		set Priv($status,showing) 0
		$widget itemconfigure $Priv($status,itemId) -visible no
		set arrange 1

		# Hack -- Big status message
		set Priv($status,text) ""
	}

	if $arrange {
		Arrange 0
	}
}

# NSStatus::Arrange --
#
#	Arranges all the status items.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Arrange resize {

	variable Priv

	set widget [Global main,widget]

	set count 0
	set row 0
	foreach status $Priv(names) {
		if $Priv($status,showing) {
			if $count {
				$widget itemconfigure $Priv($status,itemId) \
					-x [expr $count * $Priv(statusWid)] \
					-y [expr $row * 16]
			} else {
				$widget itemconfigure $Priv($status,itemId) \
					-x 0 -y [expr $row * 16]
			}
			if {[incr count] >= $Priv(numPerRow)} {
				set count 0
				incr row
			}
		}
	}

	# If this was called because the widget resized, then position the
	# big status message.
	if $resize {
		set itemId $Priv(statusMessage,itemId)
		set x [expr [winfo reqwidth $widget] / 2]
		set y 32
		$widget itemconfigure $itemId -x $x -y $y
	}
}

# NSStatus::Configure --
#
#	Called when the main widget resizes.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Configure {} {

	variable Priv

	set widget [Global main,widget]

	set statusWid $Priv(statusWid)
	set widgetWid [winfo reqwidth $widget]
	set Priv(numPerRow) [expr $widgetWid / $statusWid]

	foreach status $Priv(names) {
		$widget itemconfigure $Priv($status,itemId) -x 0 -y 0
	}

	Arrange 1
}

# NSStatus::ColorChanged --
#
#	Called when a color (value) changes.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ColorChanged {color option} {

	variable Priv

	set widget [Global main,widget]
	set color [Value $color]

	foreach status $Priv(names) {
		$widget itemconfigure $Priv($status,itemId) $option $color
	}
}

# NSStatus::InitStatusMessage --
#
#	Initialize the big status message. This is a text item that
#	appears briefly whenever a status message is set.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitStatusMessage {} {

	variable Priv

	set widget [Global main,widget]

	set font {Times 14 bold}
	set x [expr [winfo reqwidth $widget] / 2]
	set y 32

	# Hack -- Calculate the maximum dimensions of a status message
	set width [font measure $font "Mortal wound"]
	incr width 16
	set height [font metrics $font -linespace]

	set itemId [$widget create text -width $width -height $height \
		-bevel no -visible no -fill 35 -fill2 200 -anchor center \
		-background 0 -background2 0 \
		-font $font -x $x -y $y -justify center]

	set Priv(statusMessage,itemId) $itemId
	set Priv(statusMessage,afterId) ""
	set Priv(statusMessage,queue) {}
	set Priv(statusMessage,visible) 0
}

# NSStatus::SetStatusMessage --
#
#	Set the text of the big status message. If the status message is
#	already showing, then just queue the text to be displayed later.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc SetStatusMessage text {

	variable Priv

	# XXX Mega-hack -- Ignore "Rest" and "Repeat" messages
	if {[string first Rest $text] != -1} return
	if {[string first Rep $text] != -1} return

	set widget [Global main,widget]
	set itemId $Priv(statusMessage,itemId)

	lappend Priv(statusMessage,queue) $text

	if $Priv(statusMessage,visible) return

	after cancel $Priv(statusMessage,afterId)
	$widget itemconfigure $itemId \
		-visible yes -text $text
	set Priv(statusMessage,afterId) [after 1000 NSStatus::ClearStatusMessage]

	set Priv(statusMessage,visible) 1
}

# NSStatus::ClearStatusMessage --
#
#	Called as an "after" script. If the big-status-message queue is empty,
#	then hide the big status message. Otherwise display the next element
#	in the queue.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ClearStatusMessage {} {

	variable Priv

	set widget [Global main,widget]
	set itemId $Priv(statusMessage,itemId)

	set Priv(statusMessage,queue) [lrange $Priv(statusMessage,queue) 1 end]
	if {[llength $Priv(statusMessage,queue)]} {
		$widget itemconfigure $itemId \
			-text [lindex $Priv(statusMessage,queue) 0]
		set Priv(statusMessage,afterId) [after 1000 NSStatus::ClearStatusMessage]
	} else {
		$widget itemconfigure $itemId -visible no
		set Priv(statusMessage,visible) 0
	}
}

# namespace eval NSStatus
}
