# misc-window.tcl --
#
#	Implements other game windows.
#

namespace eval NSMiscWindow {

variable Priv

# NSMiscWindow::Info --
#
#	Query and modify info.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Info {info args} {

	variable Priv

	# Set info
	if {[llength $args]} {
		switch -- $info {
			default {
				set Priv($info) [lindex $args 0]
			}
		}

	# Get info
	} else {
		switch -- $info {
			default {
				return $Priv($info)
			}
		}
	}
}

# NSMiscWindow::InitAuxWindows --
#
#	Initialize other windows.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitAuxWindows {} {

	global NSMainWindow
	global Windows

	# Get the Main Window
	set winMain $Windows(main,win)

	# Big screens get the Tool Window
	if {0 && [winfo screenwidth .] >= 800} {
	
		#
		# Toolbar
		#
	
		set win .tools
		toplevel $win
		wm title $win "Tool"
		wm transient $win $winMain
		wm resizable $win no no
		Term_KeyPress_Bind $win
	
		# Start out withdrawn (hidden)
		wm withdraw $win
	
		set Windows(tool,win) $win
	
		set child [InitDisplay_Toolbar $win]
		pack $child \
			-side top -expand yes -fill x
	}

	#
	# Message line
	#

	set win .message
	toplevel $win
	wm title $win "Message"
	wm transient $win $winMain
	wm resizable $win yes no
	wm protocol $win WM_DELETE_WINDOW "bell"
	Term_KeyPress_Bind $win

	# This window plays sound
	SoundWindow $win

	# Start out withdrawn (hidden)
	wm withdraw $win

	set Windows(message,win) $win

	set child [InitDisplay_Message $win]
	pack $child \
		-side top -expand yes -fill x

	#
	# Misc info window
	#

	set win .misc
	toplevel $win
	wm title $win "Misc"
	wm transient $win $Windows(main,win)
	wm resizable $win no no
	wm protocol $win WM_DELETE_WINDOW "bell"
	Term_KeyPress_Bind $win

	# This window plays sound
	SoundWindow $win

	# Start out withdrawn (hidden)
	wm withdraw $win

	set Windows(misc,win) $win

	set child [InitDisplay_Misc $win]
	pack $child \
		-expand no

	#
	# Micro-map window
	#

	set win .micromap
	toplevel $win
	wm title $win "Micro Map"
	wm resizable $win yes yes
	wm transient $win $winMain
	wm protocol $win WM_DELETE_WINDOW "bell"
	Term_KeyPress_Bind $win

	# This window plays sound
	SoundWindow $win

	# Start out withdrawn (hidden)
	wm withdraw $win

	set Windows(micromap,win) $win

	set child [InitDisplay_MicroMap $win]
	place $child -relx 0.5 -rely 0.5 -anchor center

	#
	# Progress window
	#

	set win .progress
	toplevel $win
	wm title $win "Progress"
	wm transient $win $winMain
	wm resizable $win no no
	wm protocol $win WM_DELETE_WINDOW "bell"
	Term_KeyPress_Bind $win

	# This window plays sound
	SoundWindow $win

	# Start out withdrawn (hidden)
	wm withdraw $win

	set Windows(progress,win) $win

	set child [InitDisplay_Progress $win]
	pack $child \
		-expand no
}

proc NewButton {canvas x y image command} {

	$canvas create line $x [expr $y + 16] $x $y [expr $x + 17] $y -tags "$image border$image"
	$canvas create line $x [expr $y + 17] [expr $x + 17] [expr $y + 17] [expr $x + 17] [expr $y - 1] -tags "$image border$image"
	$canvas create image [expr $x + 1] [expr $y + 1] -image $image -anchor nw -tags $image

	$canvas bind $image <Enter> \
		"$canvas itemconfigure border$image -fill gray60"
	$canvas bind $image <Leave> \
		"$canvas itemconfigure border$image -fill Black"

	$canvas bind $image <ButtonPress-1> \
		"$canvas move $image 1 1 ; $command"
	$canvas bind $image <ButtonRelease-1> \
		"$canvas move $image -1 -1"
}

proc InitDisplay_Toolbar parent {

	global Angband

	set menu $parent.popup
	menu $menu -tearoff 0

	image create photo Image_ButtonActivate \
		-file [file join $Angband(dirTK) image button-activate.gif]
#	image create photo Image_ButtonInfo \
#		-file [file join $Angband(dirTK) image button-info.gif]
	image create photo Image_ButtonFood \
		-file [file join $Angband(dirTK) image button-food.gif]
	image create photo Image_ButtonPotion \
		-file [file join $Angband(dirTK) image button-potion.gif]
	image create photo Image_ButtonScroll \
		-file [file join $Angband(dirTK) image button-scroll.gif]
	image create photo Image_ButtonRod \
		-file [file join $Angband(dirTK) image button-rod.gif]
	image create photo Image_ButtonWand \
		-file [file join $Angband(dirTK) image button-wand.gif]
	image create photo Image_ButtonStaff \
		-file [file join $Angband(dirTK) image button-staff.gif]
	image create photo Image_ButtonDown \
		-file [file join $Angband(dirTK) image button-down.gif]
	image create photo Image_ButtonUp \
		-file [file join $Angband(dirTK) image button-up.gif]

	set canvas $parent.tools
	canvas $canvas -width [expr (16 + 4) * 9] -height 22 \
		-scrollregion {0 0 0 0} -highlightthickness 0 -background Black

	set x 1
	NewButton $canvas $x 2 Image_ButtonActivate "InventoryPopup $menu $canvas equipment A -activate yes"
#	NewButton $canvas [incr x 20] 2 Image_ButtonInfo "DoUnderlyingCommand C"
	NewButton $canvas [incr x 20] 2 Image_ButtonFood "InventoryPopup $menu $canvas inventory E -tval TV_FOOD"
	NewButton $canvas [incr x 20] 2 Image_ButtonPotion "InventoryPopup $menu $canvas inventory q -tval TV_POTION"
	NewButton $canvas [incr x 20] 2 Image_ButtonScroll "InventoryPopup $menu $canvas inventory r -tval TV_SCROLL"
	NewButton $canvas [incr x 20] 2 Image_ButtonRod "InventoryPopup $menu $canvas inventory z -tval TV_ROD"
	NewButton $canvas [incr x 20] 2 Image_ButtonWand "InventoryPopup $menu $canvas inventory a -tval TV_WAND"
	NewButton $canvas [incr x 20] 2 Image_ButtonStaff "InventoryPopup $menu $canvas inventory u -tval TV_STAFF"
	NewButton $canvas [incr x 20] 2 Image_ButtonUp "DoUnderlyingCommand <"
	NewButton $canvas [incr x 20] 2 Image_ButtonDown "DoUnderlyingCommand >"

	return $canvas
}

proc InitDisplay_Message parent {

	global Windows

	set font [Global font,fixed,normal]

	set text $parent.message
	text $text \
		-width 80 -height 1 -font $font -background Black \
		-foreground White -cursor {}
	bindtags $text [list $text $parent all]

$text tag configure tagMore -foreground turquoise2

	# Click to bypass the -more- prompt, or repeat the last message
	bind $text <ButtonPress-1> {
		if {[lsearch [angband inkey_flags] INKEY_MORE] != -1} {
			angband keypress " "
		} else {
			DoUnderlyingCommand ^o
		}
	}	
	bind $text <Double-ButtonPress-1> {
		DoUnderlyingCommand ^p
	}	

	NSStatusText::StatusText $text \
		$Windows(main,win).statusBar.label \
			"Click for last message. Double-click for message history."

	return $text
}

proc InitDisplay_MicroMap parent {

	global NSMainWindow
	global PYPX
	global Windows

	# I want this window as wide as the Misc Window
	set width [winfo reqwidth $Windows(misc,win).misc]

	set scale [Value micromap,scale]
	set cols [expr $width / $scale]

	# Make room for buttons on small monitors
	if {[winfo screenwidth .] < 800} {
		set rows [expr 38 * 4 / $scale]
	} else {
		set rows [expr 43 * 4 / $scale]
	}

	set widgetId [NSObject::New NSWidget $parent $cols $rows $scale $scale]
	NSWidget::Info $widgetId examineCmd ExamineLocation
	set widget [NSWidget::Info $widgetId widget]

	NSWidget::Info $widgetId scaleCmd \
		"Value micromap,scale \[NSWidget::Info $widgetId scale]"

	$parent configure -width $width \
		-height [NSWidget::Info $widgetId height]

	# Global access
	Global micromap,widgetId $widgetId
	Global micromap,widget $widget

	# Because the map supports different resolutions, and the
	# different resolutions result in slightly different-sized
	# maps, I make the background black so the edges don't show.
	$parent configure -background Black

	# See the Big Map when the Micro Map is clicked
	bind $widget <ButtonPress-1> {
		DoUnderlyingCommand M
	}

	# Display cursor when mouse enters the map. If the cursor is already
	# visible then don't reposition it. Perhaps I should have two cursors?
	bind $widget <Enter> {
		if {![Global cursor,visible]} {
			[Global main,widget] itemconfigure [Global cursor,itemId] \
				-visible yes
		}
	}

	# If the cursor was showing, recenter MainWidget on the cursor's
	# location. If the cursor was not showing (meaning we displayed it)
	# then hide the cursor and recenter on the character's location.
	bind $widget <Leave> {
		if {[Global cursor,visible]} {
			[Global main,widget] center [Global cursor,y] [Global cursor,x]
		} else {
			eval [Global main,widget] center [Global main,widget,center]
			[Global main,widget] itemconfigure [Global cursor,itemId] \
				-visible no
		}
	}

	bindtags $parent [concat [bindtags $parent] MicroMapBindTag]
	bind MicroMapBindTag <Configure> \
		NSMiscWindow::Configure

	# This Widget is used to display detail while examining the dungeon map
	set gsize [icon size]
	set width [expr [NSWidget::Info $widgetId width] / $gsize]
	set height [expr [NSWidget::Info $widgetId height] / $gsize]

	# Fiddle for odd number of rows/columns
	if {!($width & 1)} {incr width -1}
	if {!($height & 1)} {incr height -1}

	set widgetId [NSObject::New NSWidget $parent $width $height \
		$gsize $gsize]
	set widget2 [NSWidget::Info $widgetId widget]

	# Cursor for detail widget
	set itemId [$widget2 create cursor -color yellow -linewidth 2]

	# Global access
	Global mapdetail,widgetId $widgetId
	Global mapdetail,widget $widget2
	Global mapdetail,cursor $itemId

	# Keep the Micro Map centered on the character
	qebind $widget <Position> {NSMiscWindow::PositionChanged %W %y %x}

	qebind $widget <Dungeon-enter> {
		WidgetCenter %W
	}

	return $widget
}

proc InitDisplay_Misc parent {

	#
	# Font
	#

	set font [Global font,fixed,small]
	set fontWidth [font measure $font "W"]
	set fontHeight [font metrics $font -linespace]

	#
	# Main canvas
	#

	set c $parent.misc
	set wid [expr $fontWidth * 27 + 4]
	set hgt [expr $fontHeight * 9 + 4]
	canvas $c \
		-scrollregion "-2 -2 [expr $wid - 2] [expr $hgt - 2]" \
		-width $wid -height $hgt \
		-relief flat -highlightthickness 0 -background Black

	# Character image (center at 22,22)
	set inset [expr 22 - [icon size] / 2]
	set char_attr [angband player char_attr]
	$c create widget 22 22 -anchor center -index [lindex $char_attr 1] \
		-type [lindex $char_attr 0] -tags icon

	# Click the image to see character info
	$c bind icon <Enter> {
		%W itemconfigure focus -fill gray60
		NSMainWindow::StatusText $Windows(main,oop) \
			"Click for character info. Control-click for flags."
	}
	$c bind icon <Leave> {
		%W itemconfigure focus -fill Black
		NSMainWindow::StatusText $Windows(main,oop) ""
	}
	$c bind icon <ButtonPress-1> {
		DoUnderlyingCommand C
	}
	$c bind icon <Control-ButtonPress-1> {
		angband_display playerflags show
	}

	# Focus ring around character image (since user can click it)
	set left [expr $inset - 2]
	set top [expr $inset - 2]
	set right [expr $left + [icon size] + 2 + 1]
	set bottom [expr $top + [icon size] + 2 + 1]
	$c create line $left $top $right $top $right $bottom $left $bottom \
		$left $top -fill Black -tags focus

	# When the character is assigned a new icon, I need to update the
	# canvas Widget item. This binding does the trick.
	qebind $c <Assign-character> {
		%W itemconfigure icon -type %t -index %i
	}

	# Character name
	$c create text 45 14 -font $font -justify left -anchor w -fill turquoise2 -tags name

	# Title
	$c create text 45 28 -font $font -justify left -anchor w -fill turquoise2 -tags title

	# Stat titles
	set y 2
	foreach title {STR INT WIS DEX CON CHR} {
		CanvasAddTextItem $c $font 0 [incr y] 4 White $title: left {}
	}

	# Stats
	set y 2
	foreach stat {strength intelligence wisdom dexterity constitution charisma} {
		CanvasAddTextItem $c $font 4 [incr y] 8 LawnGreen --/--- right $stat
	}

	# Title
	set y 4
	foreach title {LEVEL EXP AU AC} {
		CanvasAddTextItem $c $font 14 [incr y] 0 White $title left $title
	}

	# Race and Class
	set y 2
	foreach info {race class} {
		CanvasAddTextItem $c $font 14 [incr y] 13 turquoise2 "" left $info
	}

	# Level, Experience, Gold and Armor Class
	set y 4
	foreach info {level exp gold armor_class} {
		CanvasAddTextItem $c $font 14 [incr y] 13 LawnGreen "" right $info
	}

	# Global access
	Global misc,canvas $c

	# Set some initial values.
	$c itemconfigure class -text [angband player class]

	# Synch with option
	if {![Value misc,mode,exp]} {
		$c itemconfigure EXP -text ADV
	}

	# When the mouse enters the "EXP" text, it toggles to ADV and shows
	# the points needed to reach the next level.
	$c bind EXP <Enter> {
		%W itemconfigure EXP -fill gray60
	}
	$c bind EXP <Leave> {
		%W itemconfigure EXP -fill White
	}
	$c bind EXP <ButtonPress-1> {
		Value misc,mode,exp [expr ![Value misc,mode,exp]]
		if {[Value misc,mode,exp]} {
			%W itemconfigure EXP -text EXP
		} else {
			%W itemconfigure EXP -text ADV
		}
		eval NSMiscWindow::bind_Py_exp %W [angband player exp]
	}

	$c bind AC <Enter> {
		%W itemconfigure AC -fill gray60
	}
	$c bind AC <Leave> {
		%W itemconfigure AC -fill White
	}
	$c bind AC <ButtonPress-1> {
		Value misc,mode,armor_class [expr ![Value misc,mode,armor_class]]
		eval NSMiscWindow::bind_Py_ac %W [angband player armor_class]
	}

	qebind $c <Stat> {
		NSMiscWindow::bind_Stat %W %d
	}

	qebind $c <Py-armor_class> {
		NSMiscWindow::bind_Py_ac %W %c %t
	}

	qebind $c <Py-exp> {
		NSMiscWindow::bind_Py_exp %W %c %m %a
	}

	qebind $c <Py-gold> {
		NSMiscWindow::bind_Py_value %W %d %c
	}

	qebind $c <Py-level> {
		NSMiscWindow::bind_Py_level %W %c %m
	}

	qebind $c <Py-name> {
		NSMiscWindow::bind_Py_value %W %d %c
	}

	qebind $c <Py-title> {
		NSMiscWindow::bind_Py_value %W %d %c
	}

	# Race can change in ZAngband!
	qebind $c <Py-race> {
		NSMiscWindow::bind_Py_value %W %d %c
		if {[Global autoAssign]} {
			AutoAssignCharacterIcon
		}
	}

if {1 || [winfo screenwidth .] >= 800} {
	$c configure -height [expr $hgt + 20]
	$c move all 0 20
	$c create line 0 21 [expr $wid - 4] 21 -fill #333366
	set tools [InitDisplay_Toolbar $c]
	pack $tools -side top -anchor n
	pack propagate $c no
}

	return $c
}

proc InitDisplay_Progress parent {

	global Notice
	global NSProgress
	global Windows

	set mainId $Windows(main,oop)

	# I want this window as wide as the Misc Window
	set width [winfo reqwidth $Windows(misc,win).misc]

	set font [Global font,fixed,small]
	set fontWidth [font measure $font "W"]

	# Space at ends of each bar
	set barSpace 8

	# Width of each bar
	set progWidth [expr $width - $fontWidth * 8 - $barSpace * 2 - 2]

	set barLeft [expr $width - $barSpace - $progWidth]
	set barRight [expr $width - $barSpace]

	set c $parent.progress
	canvas $c \
		-scrollregion "0 0 $width 60" \
		-width $width -height 60 \
		-relief flat -highlightthickness 0 -background Black

	# Border around first bar
	set x [expr $barLeft - 1]
	set y1 12
	set y2 19
	$c create line $x $y2 $x $y1 $barRight $y1 -fill #666699 \
		-tags {borderHP1 border}
	$c create line $barRight $y1 $barRight $y2 $x $y2 -fill #333366 \
		-tags {borderHP2 border}

	# Border around second bar
	incr y1 15
	incr y2 15
	$c create line $x $y2 $x $y1 $barRight $y1 -fill #666699 \
		-tags {borderSP1 border}
	$c create line $barRight $y1 $barRight $y2 $x $y2 -fill #333366 \
		-tags {borderSP2 border}

	# Border around third bar
	incr y1 15
	incr y2 15
	$c create line $x $y2 $x $y1 $barRight $y1 -fill #666699 \
		-tags {borderFD1 border}
	$c create line $barRight $y1 $barRight $y2 $x $y2 -fill #333366 \
		-tags {borderFD2 border}

	# Bar titles
	set y 7
	foreach title {HP SP FD} {
		$c create text 2 [incr y 15] -text $title -font $font \
			-fill White -anchor sw
	}

	# Hit Points
	set y 13
	set progHP [NSObject::New NSProgress $c $progWidth 6 "Red" gray60]
	place $NSProgress($progHP,frame) \
		-x $barLeft -y $y -anchor nw
	Info proghitpoints $progHP

	NSStatusText::StatusText $NSProgress($progHP,frame) \
		$Windows(main,win).statusBar.label "Your character's hit points."

	set x [expr $barLeft - $barSpace] 
	set y2 7
	$c create text $x [incr y2 15] -text 99999 -font $font \
		-fill LawnGreen -anchor se -tags hitpoints

	# Spell Points
	set progSP [NSObject::New NSProgress $c $progWidth 6 "Blue" gray60]
	place $NSProgress($progSP,frame) \
		-x $barLeft -y [incr y 15] -anchor nw
	Info progmana $progSP

	NSStatusText::StatusText $NSProgress($progSP,frame) \
		$Windows(main,win).statusBar.label "Your character's spell points (mana)."

	$c create text $x [incr y2 15] -text 99999 -font $font \
		-fill LawnGreen -anchor se -tags mana

	# Food
	set progFD [NSObject::New NSProgress $c $progWidth 6 "Brown" gray60]
	place $NSProgress($progFD,frame) \
		-x $barLeft -y [incr y 15] -anchor nw
	Info progfood $progFD

	NSStatusText::StatusText $NSProgress($progFD,frame) \
		$Windows(main,win).statusBar.label "Your character's food level."

	$c create text $x [incr y2 15] -text 99999 -font $font \
		-fill LawnGreen -anchor se -tags food

	qebind $c <Py-hitpoints> {
		NSMiscWindow::UpdateProgress hitpoints %c %m %f
	}

	qebind $c <Py-mana> {
		NSMiscWindow::UpdateProgress mana %c %m %f
	}

	qebind $c <Py-food> {
		NSMiscWindow::UpdateProgress food %c %m %f
	}

	#
	# Context Menu
	#
	
	set menu $parent.context
	menu $menu -tearoff 0
	bind $parent <ButtonPress-3> "NSMiscWindow::ContextMenu $menu %X %Y"

	# Global access
	Global progress,canvas $c

	# Hack -- Synch with options
	ConfigProgress

	# Hack -- Manually updated the food bar
	eval UpdateProgress food [angband player food]

	return $c
}

proc UpdateProgress {which cur max frac} {

	set canvas [Global progress,canvas]

	if {$which != "food"} {

		set hitpoint_warn [lindex [angband setting hitpoint_warn] 0]
	
		if {$cur >= $max} {
			set fill LawnGreen
		} elseif {$cur > ($max * $hitpoint_warn) / 10} {
			set fill Yellow
		} else {
			if {$cur < 0} {set cur 0}
			set fill Red
		}

	} else {
		set fill LawnGreen
	}

	if {[Value progress,showNumbers]} {
		if {[Value progress,showBars]} {
			$canvas itemconfigure $which -fill $fill -text $cur
		} else {
			$canvas itemconfigure $which -fill $fill -text $cur/$max
		}
	}

	if {[Value progress,showBars]} {
		NSProgress::SetDoneRatio [NSMiscWindow::Info prog$which] $frac
	}
}

# NSMiscWindow::ConfigProgress --
#
#	Arranges stuff in the Progress Window depending on the current
#	configuration options.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ConfigProgress {} {

	global NSProgress

	set showBars [Value progress,showBars]
	set showNumbers [Value progress,showNumbers]

	set font [Global font,fixed,normal]
	set fontWidth [font measure $font "W"]

	set canvas [Global progress,canvas]
	set width [winfo reqwidth $canvas]

	# Width of the text. Includes 3 for "HP " or "SP " or "FD ", plus
	# "12345" if the bars are showing, or "12345/12345" if the bars
	# are not showing. Inset 2 pixels from the left.
	set textLength 2
	if $showNumbers {incr textLength 6}
	if !$showBars {incr textLength 6}
	set textWidth [expr $textLength * $fontWidth + 2]

	# Space at ends of each bar
	set barSpace 8

	# Width of each bar. This is the width of the window minus the space
	# at each end of a bar, minus the bar title "HP " etc, minus the
	# numbers "12345" if numbers are showing.
	set barWidth [expr $width - $textWidth - $barSpace * 2]

	set barLeft [expr $width - $barSpace - $barWidth]
	set barRight [expr $width - $barSpace]

	if $showNumbers {
		set x [expr $barLeft - $barSpace] 
		set y2 7
		$canvas coords hitpoints $x [incr y2 15]

		set x [expr $barLeft - $barSpace] 
		$canvas coords mana $x [incr y2 15]

		set x [expr $barLeft - $barSpace] 
		$canvas coords food $x [incr y2 15]

	} else {
		$canvas move hitpoints $width 0
		$canvas move mana $width 0
		$canvas move food $width 0
	}
		
	if $showBars {

		# Border around first bar
		set x [expr $barLeft - 1]
		set y1 12
		set y2 19
		$canvas coords borderHP1 $x $y2 $x $y1 $barRight $y1
		$canvas coords borderHP2 $barRight $y1 $barRight $y2 $x $y2
	
		# Border around second bar
		incr y1 15
		incr y2 15
		$canvas coords borderSP1 $x $y2 $x $y1 $barRight $y1
		$canvas coords borderSP2 $barRight $y1 $barRight $y2 $x $y2
	
		# Border around third bar
		incr y1 15
		incr y2 15
		$canvas coords borderFD1 $x $y2 $x $y1 $barRight $y1
		$canvas coords borderFD2 $barRight $y1 $barRight $y2 $x $y2
	
		# Hit Points
		set y 13
		set progHP [Info proghitpoints]
		place $NSProgress($progHP,frame) \
			-x $barLeft -y $y -anchor nw
		$NSProgress($progHP,frame) configure -width $barWidth
		
		# Spell Points
		set progSP [Info progmana]
		place $NSProgress($progSP,frame) \
			-x $barLeft -y [incr y 15] -anchor nw
		$NSProgress($progSP,frame) configure -width $barWidth
	
		# Food
		set progFD [Info progfood]
		place $NSProgress($progFD,frame) \
			-x $barLeft -y [incr y 15] -anchor nw
		$NSProgress($progFD,frame) configure -width $barWidth

	} else {
		$canvas move border $width 0
		foreach bar [winfo children $canvas] {
			place forget $bar
		}
	}
}

# NSMiscWindow::ContextMenu --
#
#	Pop up a context menu in the Progress Window to configure it's
# 	appearance.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ContextMenu {menu x y} {

	$menu delete 0 end

	set showBars [Value progress,showBars]
	set showNumbers [Value progress,showNumbers]

	set state normal
	set action Show
	if $showBars {
		set action Hide
		if !$showNumbers {
			set state disabled
		}
	}
	$menu add command -label "$action Bars" -state $state \
		-command "NSMiscWindow::ToggleProgress showBars"

	set state normal
	set action Show
	if $showNumbers {
		set action Hide
		if !$showBars {
			set state disabled
		}
	}
	$menu add command -label "$action Numbers" -state $state \
		-command "NSMiscWindow::ToggleProgress showNumbers"

	tk_popup $menu $x $y
}

# NSMiscWindow::ToggleProgress --
#
#	Hide or show the bars or numbers in the Progress Window.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ToggleProgress which {

	set showIt [Value progress,$which]
	Value progress,$which [expr !$showIt]

	ConfigProgress

	eval UpdateProgress hitpoints [angband player hitpoints]
	eval UpdateProgress mana [angband player mana]
	eval UpdateProgress food [angband player food]
}

proc Configure {} {

	global Windows

	set win $Windows(micromap,win)

	# Resize micromap,widget
	set widgetId [Global micromap,widgetId]
	set cols [expr [winfo width $win] / [Value micromap,scale]]
	set rows [expr [winfo height $win] / [Value micromap,scale]]
	if {[NSWidget::Resize $widgetId $cols $rows]} {
	}

	# Resize mapdetail,widget
	set widgetId [Global mapdetail,widgetId]
	set cols [expr [winfo width $win] / [icon size]]
	set rows [expr [winfo height $win] / [icon size]]
	if {!($cols & 1)} {incr cols -1}
	if {!($rows & 1)} {incr rows -1}
	if {[NSWidget::Resize $widgetId $cols $rows]} {
	}

}

# NSMiscWindow::CanvasAddTextItem --
#
#	Create a new canvas text item at the given "row" and "column".
#	Item positioning is determined by the size of the given font.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc CanvasAddTextItem {canvas font col row width fill text justify tags} {

	set fontWidth [font measure $font "W"]
	set fontHeight [font metrics $font -linespace]

	if {$justify == "right"} {
		incr col $width
		set anchor ne
	} else {
		set anchor nw
	}
	set x [expr $col * $fontWidth]
	set y [expr $row * $fontHeight]
	set width [expr $width * $fontWidth]
	return [$canvas create text $x $y -fill $fill -text $text \
		-font $font -justify $justify \
		-anchor $anchor -tags $tags]
}

# NSMiscWindow::bind_Stat --
#
#	Called in response to a <Stat> event. Updates the Misc Window.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc bind_Stat {canvas stat} {

	angband player stat $stat statInfo
	set max $statInfo(use)
	set top $statInfo(top)
	if {$max < $top} {
		set fill Yellow
	} else {
		set fill LawnGreen
	}
	set value [cnv_stat_disp $max]
	set text [$canvas itemcget $stat -text]
	if {[string compare $text $value]} {
		$canvas itemconfigure $stat -text $value -fill $fill
		FlashCanvasText $canvas $stat White 4
	}
}

# bind_Py --
#
#	Called in response to a <Py-armor_class> event.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc bind_Py_ac {canvas dis_ac dis_to_a} {

	if {[Value misc,mode,armor_class]} {
		set ac [expr $dis_ac + $dis_to_a]
	} else {
		set ac [format "%d,%+d" $dis_ac $dis_to_a]
	}
	set text [$canvas itemcget armor_class -text]
	if {[string compare $text $ac]} {
		$canvas itemconfigure armor_class -text $ac -fill LawnGreen
		FlashCanvasText $canvas armor_class White 4
	}
}

# bind_Py_exp --
#
#	Called in response to a <Py-exp> event.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc bind_Py_exp {canvas cur max adv} {

	if {$cur < $max} {
		set fill Yellow
	} else {
		set fill LawnGreen
	}
	if {![Value misc,mode,exp]} {
		if {$adv == 999999999} {
			set cur *****
		} else {
			set cur [expr $adv - $cur]
		}
	}
	set text [$canvas itemcget exp -text]
	if {$text != $cur} {
		$canvas itemconfigure exp -text $cur -fill $fill
		FlashCanvasText $canvas exp White 4
	}
}

# bind_Py_gold --
#
#	Called in response to a <Py-> event.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc bind_Py_value {canvas tag value} {

	set text [$canvas itemcget $tag -text]
	if {[string compare $text $value]} {
		$canvas itemconfigure $tag -text $value
		FlashCanvasText $canvas $tag White 4
	}
}

# bind_Py_level --
#
#	Called in response to a <Py-level> event.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc bind_Py_level {canvas cur max} {

	if {$cur < $max} {
		set fill Yellow
	} else {
		set fill LawnGreen
	}
	set text [$canvas itemcget level -text]
	if {$text != $cur} {
		$canvas itemconfigure level -text $cur -fill $fill
		FlashCanvasText $canvas level White 4
	}

# Hack -- Display experience as well
eval bind_Py_exp $canvas [angband player exp]
}

# PositionChanged --
#
#	Called as a qebind <Position> script. Update the Micro Map Window
#	when the character's position changes. Handles the "scroll_follow"
#	option.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc PositionChanged {widget y x} {

	global PYPX

	# Option: Keep character centered in the display
	if {[Value scroll_follow]} {
		$widget center $y $x

	# Scroll when character crosses the edges of the display
	} else {
		scan [$widget center] "%d %d" oy ox

		set ny $y
		set yscroll [ClipCenter ny $oy [angband cave height] [$widget cget -height]]

		set nx $x
		set xscroll [ClipCenter nx $ox [angband cave width] [$widget cget -width]]

		# Center the widget if needed
		if {$xscroll || $yscroll} {
			scan $PYPX "%d %d" opy opx
			if {abs($y - $opy) > 1 || abs($x - $opx) > 1} {
				set ny $y
				set nx $x
			}
		}
		$widget center $ny $nx
	}
}

# namespace eval NSMiscWindow
}

proc InventoryPopup {menu canvas invOrEquip cmdChar args} {

	if {[lsearch -exact [angband inkey_flags] INKEY_CMD] == -1} return

	# Hack -- Turn on plain_descriptions
	set setting [lindex [angband setting plain_descriptions] 0]
	angband setting plain_descriptions yes

	$menu delete 0 end

	# Can't call DoUnderlyingCommand because INKEY_CMD is not set
	# while looking at inventory! Can't simply prepend a slash to
	# get the underlying command because request_command() isn't
	# being called. So I have to find a keymap for the given
	# underlying command.
	set cmdChar [angband keymap find $cmdChar]
	
	set num 0
	foreach item [eval angband $invOrEquip find $args] {
		angband $invOrEquip info $item attrib
		$menu add command -label "$attrib(char) $attrib(name)" \
			-command "angband keypress $cmdChar$attrib(char)" \
			-underline 0
		incr num
	}

	if $num {$menu add separator}
	$menu add command -label Cancel

	# Hack -- Restore plain_descriptions
	angband setting plain_descriptions $setting

	set x [winfo rootx $canvas]
	set y [expr [winfo rooty $canvas] + [winfo height $canvas]]

	tk_popup $menu $x $y
}

# ExamineLocation --
#
#	Called when the mouse moves over the MicroMap. Displays
#	a description (if any) of what the character knows about
#	the given cave location. If the screen cursor is not
#	currently visible, it is displayed at the given location.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ExamineLocation {widgetId y x} {

	global Windows

	set widget [Global main,widget]
	set mainId $Windows(main,oop)

	$widget center $y $x
	NSMainWindow::StatusText $mainId [angband cave examine $y $x]

	# If the cursor isn't already visible because of target/look, then
	# center the cursor at the given location.
	if {![Global cursor,visible]} {
		$widget itemconfigure [Global cursor,itemId] -y $y -x $x
	}
}

proc InitHeader {parent inset font elemList} {

	set cw [font measure $font "W"]

	# How to calculate the height?
	set frame $parent.header
	frame $frame \
		-borderwidth 2 -relief raised -height 22

	set i 0
	foreach {title offset anchor} $elemList {
		set label $frame.label[incr i]
		label $label \
			-text $title -borderwidth 0 -font $font
		set x [expr $inset + $cw * $offset]
		if {$anchor == "ne"} {incr x 1}
		place $label -x $x -y 0 -anchor $anchor
	}
}
