# info-window.tcl --
#
#	Implements the Info Window and related commands. The Info Window
#	is used to display a list of strings (such as Self Knowledge).
#

namespace eval NSInfoWindow {

# NSInfoWindow::NSInfoWindow --
#
#	Object constructor called by NSObject::New().
#
# Arguments:
#	oop					OOP ID.
#
# Results:
#	.

proc NSInfoWindow oop {

	global Windows

	InitWindow $oop

	#
	# Global list of application windows
	#

	set Windows(info,win) [Info $oop win]
	set Windows(info,class) NSInfoWindow
	set Windows(info,oop) $oop
}

# NSInfoWindow::Info --
#
#	Query and modify info.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Info {oop info args} {

	global NSInfoWindow

	# Set info
	if {[llength $args]} {
		switch -- $info {
			default {
				set NSInfoWindow($oop,$info) [lindex $args 0]
			}
		}

	# Get info
	} else {
		switch -- $info {
			default {
				return $NSInfoWindow($oop,$info)
			}
		}
	}
}

# NSInfoWindow::InitWindow --
#
#	Create the window associated with the object.
#
# Arguments:
#	oop					OOP ID. See above.
#
# Results:
#	.

proc InitWindow oop {

	global Windows

	set win .info$oop
	toplevel $win
	wm title $win XXXXX

	wm transient $win $Windows(main,win)

	# Start out withdrawn (hidden)
	wm withdraw $win

	# Do stuff when window closes
	wm protocol $win WM_DELETE_WINDOW "NSInfoWindow::Close $oop"

	# Set instance variables
	Info $oop win $win

	set frame $win.frameList
	frame $frame \
		-relief sunken -borderwidth 1

	set font [Global font,fixed,normal]
	set canvistId [NSObject::New NSTexist $frame $font 60 10]
	set canvas [NSTexist::Info $canvistId text]
	$canvas configure -background [NSColorPreferences::Get listBG]
	$canvas configure -yscrollcommand "$frame.yscroll set"
	$canvas configure -xscrollcommand "$frame.xscroll set"

	scrollbar $frame.yscroll \
		-orient vertical -command "$canvas yview" \
		-highlightthickness 0

	scrollbar $frame.xscroll \
		-orient horizontal -command "$canvas xview" \
		-highlightthickness 0

	# This call updates the list background color whenever the
	# global list background color changes
	Info $oop list,clientId \
		[NSValueManager::AddClient listBG "$canvas configure \
			-background \[NSColorPreferences::Get listBG]"]

	Info $oop canvistId $canvistId

	pack $win.frameList \
		-expand yes -fill both

	#
	# Geometry
	#

	grid rowconfig $win.frameList 0 -weight 1
	grid rowconfig $win.frameList 1 -weight 0
	grid columnconfig $win.frameList 0 -weight 1
	grid columnconfig $win.frameList 1 -weight 0

	grid $canvas \
		-row 0 -column 0 -rowspan 1 -columnspan 1 -sticky news
	grid $frame.yscroll \
		-row 0 -column 1 -rowspan 1 -columnspan 1 -sticky ns
	grid $frame.xscroll \
		-row 1 -column 0 -rowspan 1 -columnspan 1 -sticky ew

	#
	# Feed Term when keys pressed
	#

	bind $win <KeyPress> {
		angband keypress %A
	}

	# Synch the scrollbars when window is shown.
	bind $canvas <Map> "NSInfoWindow::SynchScrollBars $oop"
}

# NSInfoWindow::Close --
#
#	Description. 
#
# Arguments:
#	oop					OOP ID. See above.
#
# Results:
#	.

proc Close oop {

	angband keypress \033
}

# NSInfoWindow::SetList --
#
#	Display a list of items. 
#
# Arguments:
#	oop					OOP ID. See above.
#
# Results:
#	.

proc SetList {oop title info} {

	set win [Info $oop win]
	set canvistId [Info $oop canvistId]

	wm title $win $title

	NSTexist::Delete $canvistId 0 end

	foreach elem $info {
		NSTexist::Insert $canvistId end $elem White
	}
}

# NSInfoWindow::SynchScrollBars --
#
#	There is a bug (my bug or in Tk?) which prevents the scroll bars
#	from synchronizing when the window is not mapped. So I bind to
#	the <Map> event and synch the scroll bars here.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc SynchScrollBars oop {

	set win [Info $oop win]
	set canvistId [Info $oop canvistId]
	set canvas [NSTexist::Info $canvistId text]

	eval $win.frameList.yscroll set [$canvas yview]
	eval $win.frameList.xscroll set [$canvas xview]
}

# namespace eval NSInfoWindow
}

