/************************************************************************
**
**	idtmem.s - memory and cache functions
**
**	Copyright 1991 Integrated Device Technology, Inc.
**	All Rights Reserved
**
**************************************************************************/

#include "iregdef.h"
#include "idtcpu.h"  
#include "idtmon.h"  
#include "saunder.h"


	.data
mem_size:
	.word	0	
dcache_size:
	.word 	0		
icache_size:
	.word	MINCACHE	

		.text

#define CONFIGFRM ((2*4)+4)	

/*************************************************************************
**
** Config_Dcache() -- determine size of Data cache 
**
**************************************************************************/

FRAME(config_Dcache,sp, CONFIGFRM, ra)
	.set	noreorder
	subu	sp,CONFIGFRM
	sw	ra,CONFIGFRM-4(sp)	/* save return address */
	sw	s0,4*4(sp)		/* save s0 in first regsave slot */
	mfc0	s0,C0_SR		/* save SR */
	mtc0	zero,C0_SR		/* disable interrupts */
	.set	reorder
	jal	_size_cache		/* returns Data cache size in v0 */
	sw	v0, dcache_size		/* save it */
	and 	s0, ~SR_PE		/* do not clear PE */
	.set	noreorder
	mtc0	s0,C0_SR		/* restore SR */
	.set	reorder
	lw	s0, 4*4(sp)		/* restore s0 */
	lw	ra,CONFIGFRM-4(sp)	/* restore ra */
	addu	sp,CONFIGFRM		/* pop stack */
	j	ra
ENDFRAME(config_Dcache)
 

/*************************************************************************
**
** Config_Icache() -- determine size of Instruction cache
**		      MUST be run in uncached mode/handled in idt_csu.s
**		      a0 => memory size, save it in mem_size
**
**************************************************************************/

FRAME(config_Icache,sp, CONFIGFRM, ra)
	.set	noreorder
	subu	sp,CONFIGFRM
	sw	ra,CONFIGFRM-4(sp)	/* save return address */
	sw	s0,4*4(sp)		/* save s0 in first regsave slot */
	sw	a0, mem_size		/* save memory size */
	mfc0	s0,C0_SR		/* save SR */
 	mtc0	zero, C0_SR		/* disable interrupts */
	li	v0,SR_SWC		/* swap caches/disable ints  */
	mtc0	v0,C0_SR
	.set	reorder
	jal	_size_cache		/* returns instruction cache size */
	.set	noreorder
	mtc0	zero,C0_SR		/* swap back caches */  
	and	s0,~SR_PE		/* do not inadvertantly clear PE */
	mtc0	s0,C0_SR		/* restore SR */
	.set	reorder
	sw	v0, icache_size		/* save it AFTER caches back */
	lw	s0,4*4(sp)		/* restore s0 */
	lw	ra,CONFIGFRM-4(sp)	/* restore ra */
	addu	sp,CONFIGFRM		/* pop stack */
	j	ra
ENDFRAME(config_Icache)

/************************************************************************
**
** _size_cache()
** returns cache size in v0
**
************************************************************************/

FRAME(_size_cache,sp,0,ra)
	.set	noreorder
	mfc0	t0,C0_SR		/* save current sr */
	and	t0,~SR_PE		/* do not inadvertently clear PE */
	or	v0,t0,SR_ISC		/* isolate cache */
	mtc0	v0,C0_SR
	/*
	 * First check if there is a cache there at all
	 */
	move	v0,zero
	li	v1,0xa5a5a5a5		/* distinctive pattern */
	sw	v1,K0BASE		/* try to write into cache */
	lw	t1,K0BASE		/* try to read from cache */
	nop
	mfc0	t2,C0_SR
	nop
	.set	reorder
	and	t2,SR_CM
	bne	t2,zero,3f		/* cache miss, must be no cache */
	bne	v1,t1,3f		/* data not equal -> no cache */
	/*
	 * Clear cache size boundries to known state.
	 */
	li	v0,MINCACHE
1:
	sw	zero,K0BASE(v0)
	sll	v0,1
	ble	v0,MAXCACHE,1b

	li	v0,-1
	sw	v0,K0BASE(zero)		/* store marker in cache */
	li	v0,MINCACHE		/* MIN cache size */

2:	lw	v1,K0BASE(v0)		/* Look for marker */
	bne	v1,zero,3f		/* found marker */
	sll	v0,1			/* cache size * 2 */
	ble	v0,MAXCACHE,2b		/* keep looking */
	move	v0,zero			/* must be no cache */
	.set	noreorder
3:	mtc0	t0,C0_SR		/* restore sr */
	j	ra
	nop
ENDFRAME(_size_cache)
	.set	reorder


#define FLUSHFRM (2*4)

/***************************************************************************
**
** flush_Dcache() -  flush entire Data cache
**
****************************************************************************/
FRAME(flush_Dcache,sp,FLUSHFRM,ra)
	lw      t2, dcache_size		
	.set	noreorder
	mfc0	t3,C0_SR		/* save SR */
	nop
	and	t3,~SR_PE		/* dont inadvertently clear PE */
	beq	t2,zero,_Dflush_done	/* no D cache, get out! */
	nop
	li	v0, SR_ISC		/* isolate cache */
	mtc0	v0, C0_SR
	nop
	.set	reorder
	li	t0,K0BASE		/* set loop registers  */
	or	t1,t0,t2

2:	sb	zero,0(t0)
	sb	zero,4(t0)
	sb	zero,8(t0)
	sb	zero,12(t0)
	sb	zero,16(t0)
	sb	zero,20(t0)
	sb	zero,24(t0)
	addu	t0,32
	sb	zero,-4(t0)
	bne	t0,t1,2b

	.set	noreorder
_Dflush_done:
	mtc0	t3,C0_SR		/* restore Status Register */
	.set	reorder
	j	ra
ENDFRAME(flush_Dcache)


/***************************************************************************
**
** flush_Icache() -  flush entire Instruction cache
**
**	NOTE: Icache can only be flushed/cleared when uncached
**	      Code forces into uncached memory regardless of calling mode
**
****************************************************************************/
FRAME(flush_Icache,sp,FLUSHFRM,ra)
	lw	t1,icache_size 
	.set	noreorder
	mfc0	t3,C0_SR		/* save SR */
	nop
	la	v0,1f
	li	v1,K1BASE
	or	v0,v1
	j	v0			/* force into non-cached space */
	nop
1:
	and	t3,~SR_PE		/* dont inadvertently clear PE */
	beq	t1,zero,_Iflush_done	/* no i-cache get out */
	nop
	li	v0,SR_ISC|SR_SWC	/* disable intr, isolate and swap */
	mtc0	v0,C0_SR
	li	t0,K0BASE
	.set	reorder
	or	t1,t0,t1

1:	sb	zero,0(t0)
	sb	zero,4(t0)
	sb	zero,8(t0)
	sb	zero,12(t0)
	sb	zero,16(t0)
	sb	zero,20(t0)
	sb	zero,24(t0)
	addu	t0,32
	sb	zero,-4(t0)
	bne	t0,t1,1b
	.set	noreorder
_Iflush_done:
	mtc0	t3,C0_SR		/* un-isolate, enable interrupts */
	.set	reorder
	j	ra
ENDFRAME(flush_Icache)

FRAME(cacheflush,sp,FLUSHFRM,ra)
	jal	flush_Icache
	j	ra
ENDFRAME(cacheflush)

/**************************************************************************
**
** clear_Dcache(base_addr, byte_count) - flush portion of Data cache
**
**      a0 = base address of portion to be cleared
**      a1 = byte count of length
**
***************************************************************************/
FRAME(clear_Dcache,sp,0,ra)

	lw  	t2, dcache_size		/* Data cache size */
	.set	noreorder
	mfc0	t3,C0_SR		/* save SR */
	and	t3,~SR_PE		/* dont inadvertently clear PE */
	nop
	nop
	.set	reorder
	/*
	 * flush data cache
	 */

	.set	noreorder
	nop
	li	v0,SR_ISC		/* isolate data cache */
	mtc0	v0,C0_SR
	.set	reorder
	bltu	t2,a1,1f		/* cache is smaller than region */
	move	t2,a1
1:	addu	t2,a0			/* ending address + 1 */
	move	t0,a0

1:	sb	zero,0(t0)
	sb	zero,4(t0)
	sb	zero,8(t0)
	sb	zero,12(t0)
	sb	zero,16(t0)
	sb	zero,20(t0)
	sb	zero,24(t0)
	addu	t0,32
	sb	zero,-4(t0)
	bltu	t0,t2,1b

	.set	noreorder
	mtc0	t3,C0_SR		/* un-isolate, enable interrupts */
	nop
	.set	reorder
	j	ra
ENDFRAME(clear_Dcache)


/**************************************************************************
**
** clear_Icache(base_addr, byte_count) - flush portion of Instruction cache
**
**      a0 = base address of portion to be cleared
**      a1 = byte count of length
**
**	NOTE: Icache can only be flushed/cleared when uncached
**	      Code forces into uncached memory regardless of calling mode
**
***************************************************************************/
FRAME(clear_Icache,sp,0,ra)

	lw      t1, icache_size		/* Instruction cache size */
	/*
	 * flush text cache
	 */
	.set	noreorder
	mfc0	t3,C0_SR		/* save SR */
	nop 
	la	v0,1f
	li	v1,K1BASE
	or	v0,v1
	j	v0			/* force into non-cached space */
	nop
1:
	and	t3,~SR_PE		/* dont inadvertently clear PE */
	nop
	nop
	li	v0,SR_ISC|SR_SWC	/* disable intr, isolate and swap */
	mtc0	v0,C0_SR
	.set	reorder
	bltu	t1,a1,1f		/* cache is smaller than region */
	move	t1,a1
1:	addu	t1,a0			/* ending address + 1 */
	move	t0,a0

	sb	zero,0(t0)
	sb	zero,4(t0)
	sb	zero,8(t0)
	sb	zero,12(t0)
	sb	zero,16(t0)
	sb	zero,20(t0)
	sb	zero,24(t0)
	addu	t0,32
	sb	zero,-4(t0)
	bltu	t0,t1,1b
	.set	noreorder
	mtc0	t3,C0_SR		/* un-isolate, enable interrupts */
	nop
	nop
	nop				/* allow time for caches to swap */
	.set	reorder
	j	ra
ENDFRAME(clear_Icache)


/**************************************************************************
**
**  get_mem_conf - get memory configuration 
**
***************************************************************************/


FRAME(get_mem_conf,sp,0,ra)

	lw	t6, mem_size
	sw	t6, 0(a0)
	lw	t7, icache_size
	sw	t7, 4(a0)
	lw	t8, dcache_size
	sw	t8, 8(a0) 
	j	ra

ENDFRAME(get_mem_conf)
