/*
 * Draw the list of entries and everything in it. The list is drawn into
 * an array of text and image buttons. The first row is reserved for the
 * title; it cannot be pressed. This means schedule entries 0..n are drawn
 * into columns 1..n+1.
 *
 *	draw_list(w)			Draw or redraw a list of entries
 *					into a list menu.
 *	mkbuttonstring(lw, x, y)	Return a string to draw into a text
 *					button in the list menu array.
 *	mknotestring(ep)		Return the string to draw into the
 *					note button in the menu list array,
 *					or into the day box in the calendar.
 */

#include <time.h>
#include <Xm/Xm.h>
#include "cal.h"

extern struct edit	edit;		/* info about entry being edited */
extern struct config	config;		/* global configuration data */

char *mkbuttonstring(), *mknotestring();
extern char *mkdatestring(), *mktimestring();


/*
 * draw a list into a list popup. Always call create_sublist() for the
 * menu first, so that there is something to draw. Clear now-unused rows
 * under the current last row that may have been drawn earlier.
 */

draw_list(w)
	struct listmenu		*w;		/* draw into which menu */
{
	register Widget		(*wp)[SC_N];	/* button widget pointer */
	struct entry		**epp;		/* entry pointer pointer */
	register struct entry	*ep;		/* entry pointer */
	int			n = 0;		/* entry counter */
	Arg			args;		/* for toggle button */

	wp = w->entry + 1;				/* [0] is header */
	if (w->sublist) {				/* draw entries */
	    create_entry_widgets(w, w->sublist->nentries);
	    wp  = w->entry + 1;
	    epp = w->sublist->entry;
	    for (n=0; n <= w->sublist->nentries; n++, wp++, epp++) {
		if (edit.editing && edit.menu == w && edit.y-1 == n)
			ep = &edit.entry;
		else if (n < w->sublist->nentries)
			ep = *epp;
		else
			break;
		XtSetArg(args, XmNset, !ep->suspended);
		XtSetValues ((*wp)[SC_ENABLE], &args, 1);

		print_button((*wp)[SC_DATE],  mkbuttonstring(w,SC_DATE,  n+1));
		print_button((*wp)[SC_TIME],  mkbuttonstring(w,SC_TIME,  n+1));
		print_button((*wp)[SC_LENGTH],mkbuttonstring(w,SC_LENGTH,n+1));
		print_button((*wp)[SC_NOTE],  mkbuttonstring(w,SC_NOTE,  n+1));

		if (config.bigwarning)
		    print_button((*wp)[SC_ADVANCE],
					     mkbuttonstring(w,SC_ADVANCE,n+1));
		else
		    print_pixmap((*wp)[SC_ADVANCE],
			!ep->notime && (ep->noalarm ||
			ep->early_warn || ep->late_warn)  ? PIC_ADVANCE : -1);
		print_pixmap((*wp)[SC_RECYCLE],
			ep->rep_every || ep->rep_days ||
			ep->rep_yearly ||
			ep->rep_weekdays		  ? PIC_RECYCLE : -1);
		print_pixmap((*wp)[SC_MESSAGE],
			ep->message			  ? PIC_MESSAGE : -1);
		print_pixmap((*wp)[SC_SCRIPT],
			!ep->notime && ep->script	  ? PIC_SCRIPT  : -1);
		print_pixmap((*wp)[SC_MEETING],
			ep->meeting			  ? PIC_MEETING : -1);
		print_pixmap((*wp)[SC_PRIVATE],
			ep->private			  ? PIC_PRIVATE  : -1);
	    }
	}
	for (; n < w->nentries; n++, wp++) {		/* draw blank rows */
		XtSetArg(args, XmNset, FALSE);
		XtSetValues ((*wp)[SC_ENABLE], &args, 1);
		print_button((*wp)[SC_DATE],    "");
		print_button((*wp)[SC_TIME],    "");
		print_button((*wp)[SC_LENGTH],  "");
		print_button((*wp)[SC_NOTE],    "");
		print_pixmap((*wp)[SC_RECYCLE], -1);
		print_pixmap((*wp)[SC_MESSAGE], -1);
		print_pixmap((*wp)[SC_SCRIPT],  -1);
		print_pixmap((*wp)[SC_MEETING], -1);
		print_pixmap((*wp)[SC_PRIVATE], -1);
		if (config.bigwarning)
			print_button((*wp)[SC_ADVANCE], "");
		else
			print_pixmap((*wp)[SC_ADVANCE], -1);
	}
}


/*
 * return some sensible string representation of a button. This doesn't work
 * for pixmap labels.
 */

char *mkbuttonstring(lw, x, y)
	struct listmenu		*lw;		/* menu the button is in */
	int			x, y;		/* column and row (y >= 1) */
{
	static char		buf[40];	/* string representation */
	register struct entry	*ep;		/* entry pointer */

	if (edit.editing && edit.menu == lw && edit.y == y)
		ep = &edit.entry;
	else if (!lw->sublist || y > lw->sublist->nentries)
		return("");
	else
		ep = lw->sublist->entry[y-1];

	switch(x) {
	  case SC_DATE:
		return(mkdatestring(ep->time));

	  case SC_TIME:
		return(ep->notime ? "-" : mktimestring(ep->time, FALSE));

	  case SC_LENGTH:
		return(ep->length > 0 && !ep->notime ?
					mktimestring(ep->length, TRUE) : "");
	  case SC_ADVANCE:
		*buf = 0;
		if (ep->notime)
			break;
		if (ep->early_warn)
			sprintf(buf, "%d", (int)ep->early_warn/60);
		if (ep->late_warn) {
			if (ep->early_warn)
				strcat(buf, ", ");
			sprintf(buf+strlen(buf), "%d", (int)ep->late_warn/60);
		}
		if (ep->noalarm) {
			if (ep->early_warn || ep->late_warn)
				strcat(buf, ", ");
			strcat(buf, "-");
		}
		break;

	  case SC_NOTE:
		return(mknotestring(ep));
	}
	return(buf);
}


/*
 * return the entries's note string. This is separate from mkbuttonstring
 * because it is also used to print notes into the calendar day boxes.
 */

char *mknotestring(ep)
	register struct entry	*ep;		/* entry pointer */
{
	static char		buf[256];	/* string representation */
	char			*note;		/* note to print */
	int			i;		/* string index */

	if ((note = ep->note)    == 0 &&
	    (note = ep->message) == 0 &&
	    (note = ep->script)  != 0)
		if (note[0] == '#' && note[1] == '!') {
			while (*note && *note++ != '\n');
			if (!*note)
				note = ep->script;
		}
	i = 0;
	if (note)
		while (i < 255 && *note && *note != '\n')
			buf[i++] = *note++;
	buf[i] = 0;
	return(buf);
}
