//
// Copyright (c) 2019-2025 Ruben Perez Hidalgo (rubenperez038 at gmail dot com)
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//

#include <boost/mysql/client_errc.hpp>
#include <boost/mysql/error_code.hpp>
#include <boost/mysql/string_view.hpp>

#include <boost/mysql/impl/internal/sansio/csha2p_encrypt_password.hpp>

#include <boost/asio/ssl/error.hpp>
#include <boost/core/span.hpp>
#include <boost/test/unit_test.hpp>

#include <cstddef>
#include <openssl/bio.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <string>
#include <vector>

#include "handshake_csha2p_keys.hpp"
#include "test_common/assert_buffer_equals.hpp"
#include "test_common/printing.hpp"

using namespace boost::mysql;
using namespace boost::mysql::test;
using boost::span;
using boost::asio::error::ssl_category;
using detail::csha2p_encrypt_password;

using buffer_type = boost::container::small_vector<std::uint8_t, 512>;

namespace {

// Encrypting with RSA and OAEP padding involves random numbers for padding.
// There isn't a reliable way to seed OpenSSL's random number generators so that
// the output is deterministic. So we do the following:
//    1. We know the server's public and private keys and the password (the constants below).
//    2. We capture a scramble and a corresponding ciphertext using Wireshark.
//    3. We decrypt the ciphertext with OpenSSL to obtain the expected plaintext (the salted password).
//    4. Tests run csha2p_encrypt_password, decrypt its output, and verify that it matches the expected
//    plaintext.
BOOST_AUTO_TEST_SUITE(test_handshake_csha2p_encrypt_password)

constexpr std::uint8_t scramble[] = {
    0x0f, 0x64, 0x4f, 0x2f, 0x2b, 0x3b, 0x27, 0x6b, 0x45, 0x5c,
    0x53, 0x01, 0x13, 0x7e, 0x4f, 0x10, 0x26, 0x23, 0x5d, 0x27,
};

// Decrypts the output of
std::vector<std::uint8_t> decrypt(span<const std::uint8_t> private_key, span<const std::uint8_t> ciphertext)
{
    // RAII helpers
    struct bio_deleter
    {
        void operator()(BIO* bio) const noexcept { BIO_free(bio); }
    };
    using unique_bio = std::unique_ptr<BIO, bio_deleter>;

    struct evp_pkey_deleter
    {
        void operator()(EVP_PKEY* pkey) const noexcept { EVP_PKEY_free(pkey); }
    };
    using unique_evp_pkey = std::unique_ptr<EVP_PKEY, evp_pkey_deleter>;

    struct evp_pkey_ctx_deleter
    {
        void operator()(EVP_PKEY_CTX* ctx) const noexcept { EVP_PKEY_CTX_free(ctx); }
    };
    using unique_evp_pkey_ctx = std::unique_ptr<EVP_PKEY_CTX, evp_pkey_ctx_deleter>;

    // Create a BIO with the key
    unique_bio bio(BIO_new_mem_buf(private_key.data(), static_cast<int>(private_key.size())));
    BOOST_TEST_REQUIRE((bio != nullptr), "Creating a BIO failed: " << ERR_get_error());

    // Load the key
    unique_evp_pkey pkey(PEM_read_bio_PrivateKey(bio.get(), nullptr, nullptr, nullptr));
    BOOST_TEST_REQUIRE((bio != nullptr), "Loading the key failed: " << ERR_get_error());

    // Create a decryption context
    unique_evp_pkey_ctx ctx(EVP_PKEY_CTX_new(pkey.get(), nullptr));
    BOOST_TEST_REQUIRE((ctx != nullptr), "Creating a context failed: " << ERR_get_error());

    // Initialize it
    BOOST_TEST_REQUIRE(
        EVP_PKEY_decrypt_init(ctx.get()) > 0,
        "Initializing decryption failed: " << ERR_get_error()
    );

    // Set the padding scheme
    BOOST_TEST_REQUIRE(
        EVP_PKEY_CTX_set_rsa_padding(ctx.get(), RSA_PKCS1_OAEP_PADDING) > 0,
        "Setting the padding scheme failed: " << ERR_get_error()
    );

    // Determine the size of the decrypted buffer
    std::size_t out_size = 0;
    BOOST_TEST_REQUIRE(
        EVP_PKEY_decrypt(ctx.get(), nullptr, &out_size, ciphertext.data(), ciphertext.size()) > 0,
        "Determining decryption size failed: " << ERR_get_error()
    );
    std::vector<std::uint8_t> res(out_size, 0);

    // Actually decrypt
    BOOST_TEST_REQUIRE(
        EVP_PKEY_decrypt(ctx.get(), res.data(), &out_size, ciphertext.data(), ciphertext.size()) > 0,
        "Decrypting failed: " << ERR_get_error()
    );
    res.resize(out_size);

    // Done
    return res;
}

BOOST_AUTO_TEST_CASE(success)
{
    // A password with all the possible ASCII values
    std::string all_chars_password;
    for (int i = 0; i < 256; ++i)
        all_chars_password.push_back(static_cast<char>(i));

    struct
    {
        string_view name;
        std::string password;
        span<const std::uint8_t> public_key;
        span<const std::uint8_t> private_key;
        std::vector<std::uint8_t> expected_decrypted;
    } test_cases[] = {
        // clang-format off
        // An empty password does not cause trouble. This is an edge case, since
        // an empty password should employ a different workflow
        {
            "password_empty",
            {},
            span<const unsigned char>(public_key_2048),
            span<const unsigned char>(private_key_2048),
            {0x0f}
        },

        // Password is < length of the scramble
        {
            "password_shorter_scramble",
            "csha2p_password",
            span<const unsigned char>(public_key_2048),
            span<const unsigned char>(private_key_2048), 
            {
                0x6c, 0x17, 0x27, 0x4e, 0x19, 0x4b, 0x78, 0x1b, 0x24, 0x2f, 0x20, 0x76, 0x7c, 0x0c, 0x2b, 0x10
            }
        },

        // Password (with NULL byte) is equal to the length of the scramble
        {
            "password_same_size_scramble",
            "hjbjd923KKLiosoi90J",
            span<const unsigned char>(public_key_2048),
            span<const unsigned char>(private_key_2048),
            {
                0x67, 0x0e, 0x2d, 0x45, 0x4f, 0x02, 0x15, 0x58, 0x0e, 0x17,
                0x1f, 0x68, 0x7c, 0x0d, 0x20, 0x79, 0x1f, 0x13, 0x17, 0x27
            }
        },

        // Passwords longer than the scramble use it cyclically
        {
            "password_longer_scramble",
            "kjaski829380jvnnM,ka;::_kshf93IJCLIJO)jcjsnaklO?a",
            span<const unsigned char>(public_key_2048),
            span<const unsigned char>(private_key_2048),
            {
                0x64, 0x0e, 0x2e, 0x5c, 0x40, 0x52, 0x1f, 0x59, 0x7c, 0x6f, 0x6b, 0x31, 0x79, 0x08, 0x21, 0x7e, 0x6b,
                0x0f, 0x36, 0x46, 0x34, 0x5e, 0x75, 0x70, 0x40, 0x48, 0x4f, 0x0d, 0x7c, 0x6f, 0x1a, 0x4b, 0x50, 0x32,
                0x06, 0x5a, 0x69, 0x0a, 0x37, 0x44, 0x65, 0x17, 0x21, 0x4e, 0x40, 0x57, 0x68, 0x54, 0x24, 0x5c,
            }
        },

        // The longest password that a 2048 RSA key can encrypt with the OAEP scheme
        {
            "password_max_size_2048",
            "hjbjd923KKLkjbdkcjwhekiy8393ou2weusidhiahJBKJKHCIHCKJIu9KHO09IJIpojaf0w39jalsjMMKjkjhiue93I=))"
                "UXIOJKXNKNhkai8923oiawiakssaknhakhIIHICHIO)CU)"
                "IHCKHCKJhisiweioHHJHUCHIIIJIOPkjgwijiosoi9jsu84HHUHCHI9839hdjsbsdjuUHJjbJ",
            span<const unsigned char>(public_key_2048),
            span<const unsigned char>(private_key_2048),
            {
                0x67, 0x0e, 0x2d, 0x45, 0x4f, 0x02, 0x15, 0x58, 0x0e, 0x17, 0x1f, 0x6a, 0x79, 0x1c, 0x2b, 0x7b, 0x45,
                0x49, 0x2a, 0x4f, 0x6a, 0x0f, 0x26, 0x56, 0x13, 0x08, 0x1e, 0x58, 0x2a, 0x29, 0x61, 0x76, 0x76, 0x0b,
                0x3c, 0x79, 0x42, 0x4b, 0x34, 0x46, 0x67, 0x2e, 0x0d, 0x64, 0x61, 0x70, 0x6f, 0x28, 0x0c, 0x14, 0x10,
                0x4a, 0x59, 0x37, 0x3a, 0x29, 0x6d, 0x6b, 0x12, 0x17, 0x36, 0x2d, 0x05, 0x66, 0x5b, 0x54, 0x4d, 0x0a,
                0x23, 0x6c, 0x24, 0x32, 0x2a, 0x14, 0x2e, 0x7c, 0x55, 0x49, 0x10, 0x6a, 0x44, 0x0e, 0x24, 0x45, 0x43,
                0x52, 0x52, 0x0e, 0x7c, 0x6f, 0x1a, 0x3c, 0x3a, 0x57, 0x1a, 0x48, 0x6f, 0x6c, 0x17, 0x6c, 0x57, 0x2a,
                0x04, 0x61, 0x43, 0x50, 0x46, 0x02, 0x7d, 0x65, 0x61, 0x32, 0x7c, 0x17, 0x2e, 0x67, 0x4f, 0x42, 0x36,
                0x54, 0x7c, 0x05, 0x24, 0x41, 0x43, 0x5a, 0x4c, 0x03, 0x0c, 0x15, 0x1b, 0x48, 0x50, 0x36, 0x06, 0x5f,
                0x0f, 0x60, 0x08, 0x0e, 0x46, 0x2c, 0x0c, 0x64, 0x63, 0x78, 0x6c, 0x21, 0x2d, 0x35, 0x20, 0x68, 0x64,
                0x1b, 0x26, 0x7f, 0x6e, 0x6b, 0x17, 0x6f, 0x5a, 0x27, 0x07, 0x66, 0x62, 0x72, 0x6d, 0x22, 0x0a, 0x0c,
                0x38, 0x6b, 0x74, 0x09, 0x26, 0x7a, 0x4f, 0x4c, 0x2e, 0x48, 0x66, 0x5d, 0x25, 0x5c, 0x5e, 0x03, 0x13,
                0x23, 0x0d, 0x09, 0x1b, 0x42, 0x5b, 0x37, 0x76, 0x28, 0x15, 0x1a, 0x35, 0x43, 0x65, 0x17, 0x2d, 0x5c,
                0x4f, 0x51, 0x52, 0x3e, 0x0d, 0x16, 0x39, 0x63, 0x59, 0x7e,
            }
        },

        // Longer passwords are accepted if the key supports them.
        // Concretely, passwords longer than 512 (size of our SBO) are supported
        {
            "password_longer_sbo",
            std::string(600, '5'),
            span<const unsigned char>(public_key_8192),
            span<const unsigned char>(private_key_8192),
            {
                0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13,
                0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b,
                0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66,
                0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e,
                0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e,
                0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51,
                0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68,
                0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25,
                0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26,
                0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69,
                0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12,
                0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a,
                0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a,
                0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16,
                0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a,
                0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34,
                0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70,
                0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e,
                0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a,
                0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12,
                0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13,
                0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b,
                0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66,
                0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e,
                0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e,
                0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51,
                0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68,
                0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25,
                0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26,
                0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69,
                0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12,
                0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a,
                0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16, 0x68, 0x12, 0x3a,
                0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a, 0x25, 0x13, 0x16,
                0x68, 0x12, 0x3a, 0x51, 0x7a, 0x1a, 0x1e, 0x0e, 0x12, 0x5e, 0x70, 0x69, 0x66, 0x34, 0x26, 0x4b, 0x7a,
                0x25, 0x13, 0x16, 0x68, 0x12, 0x0f
            }
        },

        // No character causes trouble
        {
            "password_all_characters",
            all_chars_password,
            span<const unsigned char>(public_key_8192),
            span<const unsigned char>(private_key_8192),
            {
                0x0f, 0x65, 0x4d, 0x2c, 0x2f, 0x3e, 0x21, 0x6c, 0x4d, 0x55, 0x59, 0x0a, 0x1f, 0x73, 0x41, 0x1f, 0x36,
                0x32, 0x4f, 0x34, 0x1b, 0x71, 0x59, 0x38, 0x33, 0x22, 0x3d, 0x70, 0x59, 0x41, 0x4d, 0x1e, 0x33, 0x5f,
                0x6d, 0x33, 0x02, 0x06, 0x7b, 0x00, 0x27, 0x4d, 0x65, 0x04, 0x07, 0x16, 0x09, 0x44, 0x75, 0x6d, 0x61,
                0x32, 0x27, 0x4b, 0x79, 0x27, 0x1e, 0x1a, 0x67, 0x1c, 0x33, 0x59, 0x71, 0x10, 0x6b, 0x7a, 0x65, 0x28,
                0x01, 0x19, 0x15, 0x46, 0x5b, 0x37, 0x05, 0x5b, 0x6a, 0x6e, 0x13, 0x68, 0x5f, 0x35, 0x1d, 0x7c, 0x7f,
                0x6e, 0x71, 0x3c, 0x1d, 0x05, 0x09, 0x5a, 0x4f, 0x23, 0x11, 0x4f, 0x46, 0x42, 0x3f, 0x44, 0x6b, 0x01,
                0x29, 0x48, 0x43, 0x52, 0x4d, 0x00, 0x29, 0x31, 0x3d, 0x6e, 0x63, 0x0f, 0x3d, 0x63, 0x52, 0x56, 0x2b,
                0x50, 0x77, 0x1d, 0x35, 0x54, 0x57, 0x46, 0x59, 0x14, 0xc5, 0xdd, 0xd1, 0x82, 0x97, 0xfb, 0xc9, 0x97,
                0xae, 0xaa, 0xd7, 0xac, 0x83, 0xe9, 0xc1, 0xa0, 0xbb, 0xaa, 0xb5, 0xf8, 0xd1, 0xc9, 0xc5, 0x96, 0x8b,
                0xe7, 0xd5, 0x8b, 0xba, 0xbe, 0xc3, 0xb8, 0xaf, 0xc5, 0xed, 0x8c, 0x8f, 0x9e, 0x81, 0xcc, 0xed, 0xf5,
                0xf9, 0xaa, 0xbf, 0xd3, 0xe1, 0xbf, 0x96, 0x92, 0xef, 0x94, 0xbb, 0xd1, 0xf9, 0x98, 0x93, 0x82, 0x9d,
                0xd0, 0xf9, 0xe1, 0xed, 0xbe, 0xd3, 0xbf, 0x8d, 0xd3, 0xe2, 0xe6, 0x9b, 0xe0, 0xc7, 0xad, 0x85, 0xe4,
                0xe7, 0xf6, 0xe9, 0xa4, 0x95, 0x8d, 0x81, 0xd2, 0xc7, 0xab, 0x99, 0xc7, 0xfe, 0xfa, 0x87, 0xfc, 0xd3,
                0xb9, 0x91, 0xf0, 0xcb, 0xda, 0xc5, 0x88, 0xa1, 0xb9, 0xb5, 0xe6, 0xfb, 0x97, 0xa5, 0xfb, 0xca, 0xce,
                0xb3, 0xc8, 0xff, 0x95, 0xbd, 0xdc, 0xdf, 0xce, 0xd1, 0x9c, 0xbd, 0xa5, 0xa9, 0xfa, 0xef, 0x83, 0xb1,
                0xef, 0x26
            }
        }

        // clang-format on
    };

    for (const auto& tc : test_cases)
    {
        BOOST_TEST_CONTEXT(tc.name)
        {
            // Setup
            buffer_type buff;

            // Call the function
            auto ec = csha2p_encrypt_password(tc.password, scramble, tc.public_key, buff, ssl_category);

            // Verify
            BOOST_TEST_REQUIRE(ec == error_code());
            BOOST_MYSQL_ASSERT_BUFFER_EQUALS(decrypt(tc.private_key, buff), tc.expected_decrypted);
        }
    }
}

//
// Errors. For OpenSSL errors, it's not defined what exact code will each function return.
//

// We passed an empty buffer to the key parser
BOOST_AUTO_TEST_CASE(error_key_buffer_empty)
{
    buffer_type buff;
    auto ec = csha2p_encrypt_password("csha2p_password", scramble, {}, buff, ssl_category);
    BOOST_TEST((ec.category() == ssl_category));
    BOOST_TEST(ec.value() > 0);      // is an error
    BOOST_TEST(ec.message() != "");  // produces some output
    BOOST_TEST(ec.has_location());
}

BOOST_AUTO_TEST_CASE(error_key_malformed)
{
    constexpr std::uint8_t key_buffer[] = "-----BEGIN PUBLIC KEY-----zwIDAQAB__kaj0?))=";
    buffer_type buff;
    auto ec = csha2p_encrypt_password("csha2p_password", scramble, key_buffer, buff, ssl_category);
    BOOST_TEST((ec.category() == ssl_category));
    BOOST_TEST(ec.value() > 0);      // is an error
    BOOST_TEST(ec.message() != "");  // produces some output
    BOOST_TEST(ec.has_location());
}

// Passing in a public key type that does not support encryption operations
// (like ECDSA) fails
BOOST_AUTO_TEST_CASE(error_key_doesnt_support_encryption)
{
    constexpr unsigned char key_buffer[] = R"%(-----BEGIN PUBLIC KEY-----
MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAERDkCI/degPJXEIYYncyvGsTdj9YI
4xQ6KPTzoF+DY2jM09w1TCncxk9XV1eOo44UMDUuK9K01halQy70mohFSQ==
-----END PUBLIC KEY-----
)%";

    buffer_type buff;
    auto ec = csha2p_encrypt_password("csha2p_password", scramble, key_buffer, buff, ssl_category);
    BOOST_TEST((ec.category() == ssl_category));
    BOOST_TEST(ec.value() > 0);      // is an error
    BOOST_TEST(ec.message() != "");  // produces some output
    BOOST_TEST(ec.has_location());
}

// Passing in a public key type that allows encryption but is not RSA fails as expected.
// This is a SM2 key, generated by:
//   openssl genpkey -algorithm EC -pkeyopt ec_paramgen_curve:sm2 -out public.pem
//   openssl pkey -in private.pem -pubout -out public.pem
BOOST_AUTO_TEST_CASE(error_key_not_rsa)
{
    // ECDSA public key set up for SM2 encryption
    constexpr unsigned char public_key_sm2[] = R"%(-----BEGIN PUBLIC KEY-----
MFkwEwYHKoZIzj0CAQYIKoEcz1UBgi0DQgAEWGAXSpPHb2bWQjROuegjWPcuVwNW
mVpTh++3j7pnpWUjnFuarvWmWh/H6t96/pTx566FKGxZpLn3H9TLHZJsog==
-----END PUBLIC KEY-----
)%";

    buffer_type buff;
    auto ec = csha2p_encrypt_password("csha2p_password", scramble, public_key_sm2, buff, ssl_category);
    BOOST_TEST(ec.failed());  // OpenSSL might or might not provide an error code here
    BOOST_TEST(ec.has_location());
}

BOOST_AUTO_TEST_CASE(error_key_too_big)
{
    buffer_type buff;
    std::vector<std::uint8_t> key_buffer(2u * 1024u * 1024u, 0x0a);
    auto ec = csha2p_encrypt_password("csha2p_password", scramble, key_buffer, buff, ssl_category);
    BOOST_TEST(ec == client_errc::protocol_value_error);
    BOOST_TEST(ec.has_location());
}

// If a password longer than the longest allowed plaintext is provided, we fail
BOOST_AUTO_TEST_CASE(error_password_too_big)
{
    buffer_type buff;
    std::string passwd(214u, 'a');  // 213 is the max RSA/OAEP plaintext size
    auto ec = csha2p_encrypt_password(passwd, scramble, public_key_2048, buff, ssl_category);
    BOOST_TEST((ec.category() == ssl_category));
    BOOST_TEST(ec.value() > 0);      // is an error
    BOOST_TEST(ec.message() != "");  // produces some output
    BOOST_TEST(ec.has_location());
}

BOOST_AUTO_TEST_SUITE_END()

}  // namespace