/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 *
 *	Copyright 1996 Matti Aarnio
 */

/* LINTLIBRARY */

#include "mailer.h"
#ifdef	HAVE_DB_H
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#include <db.h>
#include <sys/file.h>
#include "search.h"
#include "io.h"

extern int errno;

extern int deferit;
extern void v_set();


static BTREEINFO BINFO = { 0, 2560, 0, 0, 0, NULL,  NULL, 0 };

/*
 * Flush buffered information from this database, close any file descriptors.
 */

void
close_btree(sip)
	struct search_info *sip;
{
	DB *db;
	struct spblk *spl;
	spkey_t symid;

	if (sip->file == NULL)
		return;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_modcheck);
	if (spl != NULL)
		sp_delete(spl, spt_modcheck);
	spl = sp_lookup(symid, spt_files);
	if (spl == NULL || (db = (DB *)spl->data) == NULL)
		return;
	(db->close)(db);
	sp_install(symid, (u_char *)NULL, 0, spt_files);
}


DB *
open_btree(sip, flag, comment)
	struct search_info *sip;
	int flag;
	char *comment;
{
	DB *db;
	struct spblk *spl;
	spkey_t symid;

	if (sip->file == NULL)
		return NULL;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_files);
	if (spl != NULL && flag == O_RDWR && spl->mark != O_RDWR)
		close_btree(sip);
	if (spl == NULL || (db = (DB *)spl->data) == NULL) {
		db = dbopen(sip->file, flag, 0, DB_BTREE, &BINFO);
		if (db == NULL) {
			++deferit;
			v_set(DEFER, DEFER_IO_ERROR);
			fprintf(stderr, "%s: cannot open %s!\n",
					comment, sip->file);
			return NULL;
		}
		if (spl == NULL)
			sp_install(symid, (u_char *)db, flag, spt_files);
		else
			spl->data = (u_char *)db;
	}
	return db;
}


/*
 * Search an B-TREE format database for a key pair.
 */

struct conscell *
search_btree(sip)
	struct search_info *sip;
{
	DB *db;
	DBT val, key;
	struct conscell *tmp;
	struct spblk *spl;
	int retry, rc;
	spkey_t symid;
	u_char *us;

	if (sip->file == NULL)
		return NULL;
	retry = 0;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_files);
	if (spl == NULL || (db = (DB *)spl->data) == NULL) {
reopen:
		db = open_btree(sip, O_RDONLY, "search_btree");
	}
	key.data = (char *)(sip->key);
	key.size = strlen((char *)(sip->key)) + 1;
	rc = (db->get)(db, &key, &val, 0);
	if (rc != 0) {
		if (!retry && rc < 0) {
			close_btree(sip);
			++retry;
			goto reopen;
		}
		return NULL;
	}
	us = (u_char *)strnsave(val.data, val.size);
	return newstring(us);
}


/*
 * Add the indicated key/value pair to the database.
 */

int
add_btree(sip, value)
	struct search_info *sip;
	char *value;
{
	DB *db;
	DBT val, key;
	int rc;

	if ((db = open_btree(sip, O_RDWR, "add_btree")) == NULL)
		return EOF;
	key.data = (char *)(sip->key);
	key.size = strlen((char *)(sip->key)) + 1;
	val.data = value;
	val.size = strlen(value)+1;
	rc = (db->put)(db, &key, &val, 0);
	if (rc < 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		fprintf(stderr, "add_btree: cannot store (\"%s\",\"%s\")\n",
				sip->key, value);
		return EOF;
	}
	return 0;
}

/*
 * Remove the indicated key from the database.
 */

int
remove_btree(sip)
	struct search_info *sip;
{
	DB *db;
	DBT key;
	int rc;

	if ((db = open_btree(sip, O_RDWR, "remove_btree")) == NULL)
		return EOF;
	key.data = (char *)(sip->key);
	key.size = strlen((char *)(sip->key)) + 1;
	rc = (db->del)(db, &key, 0);
	if (rc < 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		fprintf(stderr, "remove_btree: cannot remove \"%s\"\n",
				sip->key);
		return EOF;
	}
	return 0;
}

/*
 * Print the database.
 */

void
print_btree(sip, outfp)
	struct search_info *sip;
	FILE *outfp;
{
	DB *db;
	DBT key, val;
	int rc;

	if ((db = open_btree(sip, O_RDONLY, "print_btree")) == NULL)
		return;

	rc = (db->seq)(db, &key, &val, R_FIRST);
	for ( ; rc != 0 ; ) {
		if (val.data == NULL)
			continue;
		if (*(char*)val.data == '\0')
			fprintf(outfp, "%s\n", key.data);
		else
			fprintf(outfp, "%s\t%s\n", key.data, val.data);
		rc = (db->seq)(db, &key, &val, R_NEXT);
	}
	fflush(outfp);
}

/*
 * Count the database.
 */

void
count_btree(sip, outfp)
	struct search_info *sip;
	FILE *outfp;
{
	DB *db;
	DBT key, val;
	int cnt = 0;
	int rc;

	if ((db = open_btree(sip, O_RDONLY, "count_btree")) != NULL) {
	  rc = (db->seq)(db, &key, &val, R_FIRST);
	  for ( ; rc != 0 ; ) {
	    if (val.data == NULL)
	      continue;
	    ++cnt;
	    rc = (db->seq)(db, &key, &val, R_NEXT);
	  }
	}
	fprintf(outfp,"%d\n",cnt);
	fflush(outfp);
}

/*
 * Print the uid of the owner of the database.  Note that for db-style
 * databases there are several files involved so picking one of them for
 * security purposes is very dubious.
 */

void
owner_btree(sip, outfp)
	struct search_info *sip;
	FILE *outfp;
{
	DB *db;
	struct stat stbuf;

	if ((db = open_btree(sip, O_RDONLY, "owner_btree")) == NULL)
		return;
	/* There are more timing hazards, when the internal fd is not
	   available for probing.. */
	if (fstat((db->fd)(db), &stbuf) < 0) {
		fprintf(stderr, "owner_btree: cannot fstat(\"%s\")!\n",
				sip->file);
		return;
	}
	fprintf(outfp, "%d\n", stbuf.st_uid);
	fflush(outfp);
}

int
modp_btree(sip)
	struct search_info *sip;
{
	DB *db;
	struct stat stbuf;
	struct spblk *spl;
	spkey_t symid;
	int rval;

	if (sip->file == NULL
	    || (db = open_btree(sip, O_RDONLY, "owner_btree")) == NULL)
		return 0;
	if (fstat((db->fd)(db), &stbuf) < 0) {
		fprintf(stderr, "modp_btree: cannot fstat(\"%s\")!\n",
				sip->file);
		return 0;
	}
	if (stbuf.st_nlink == 0)
		return 1;	/* Unlinked underneath of us! */
	
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_modcheck);
	if (spl != NULL) {
		rval = ((long)stbuf.st_mtime != (long)spl->data
			|| (long)stbuf.st_nlink != (long)spl->mark);
	} else
		rval = 0;
	sp_install(symid, (u_char *)((long)stbuf.st_mtime),
		   stbuf.st_nlink, spt_modcheck);
	return rval;
}
#endif	/* HAVE_DB */
