;;!emacs
;; mhg.el --  Rule-based MH mail categorization and reading under GNUS.
;; USAGE           : GNU Emacs Lisp Library
;; Author          : Bob Weiner (Brown U.)
;; Created On      : 8-Jan-92 at 02:21:39
;; Last Modified By: Ulrich Pfeifer
;; Last Modified On: Sat Apr 30 17:02:52 1994
;; Update Count    : 32
;; Status          : Unknown, Use with caution!
;; $Locker: pfeifer $
;; $Log: mhg.el,v $
;; Revision 1.8  1994/04/19  06:53:37  pfeifer
;; Funktioniert, jetzt versuche ich das refile fuer mehrere ordner
;;
;; Revision 1.7  1993/10/15  15:15:02  pfeifer
;; Sicherung vor elimination von vm
;;
;; Revision 1.6  1993/10/14  09:56:42  pfeifer
;; Made usage of vm optional
;;
;; Revision 1.5  1993/10/14  09:12:07  pfeifer
;; Sicherung vor reintegration der mh-e Routinen
;;
;; Revision 1.4  1993/10/08  15:00:30  pfeifer
;; Ende hyperbole version
;;
;;
;; LCD-ENTRY:    
;;  mhg.el|Bob Weiner|None|MH-based mail reading under GNUS.|93-02-23|1.0
;;
;; This file is not part of GNU Emacs.
;;
;; Copyright (C) 1991, Brown University, Providence, RI
;; Developed with support from Motorola Inc.
;; 
;; Permission to use, modify and redistribute this software and its
;; documentation for any purpose other than its incorporation into a
;; commercial product is hereby granted without fee.  A distribution fee
;; may be charged with any redistribution.  Any distribution requires
;; that the above copyright notice appear in all copies, that both that
;; copyright notice and this permission notice appear in supporting
;; documentation, and that neither the name of Brown University nor the
;; author's name be used in advertising or publicity pertaining to
;; distribution of the software without specific, written prior permission.
;; 
;; Brown University makes no representations about the suitability of this
;; software for any purpose.  It is provided "as is" without express or
;; implied warranty.
;;
;;
;; DESCRIPTION:  
;;   rm       - Read Mail; Get new mail (Checks for existing gnus-buffers)
;;
;;   rn       - Read News; Get new News (Checks for existing gnus-buffers)
;;
;;   mhg-read - Command to read mail with GNUS and Mh support required.
;;
;;   mhg-news - Read news with same GNUS interface.
;;
;; DESCRIP-END.

;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

(require 'gnus)
;;; Prevent interactive execution of GNUS.  Use functions below instead.
;; (fset 'gnus (set:remove '(interactive "P") (symbol-function 'gnus)))
;; Not the same since original-gnus may be called but works with bye compiled 
;; gnus:
(if (fboundp 'original-gnus) ()
  (progn
    (fset 'original-gnus (symbol-function 'gnus))
    (fmakunbound 'gnus)
    (defun gnus (&optional prompt)
      (original-gnus prompt))))

;; (autoload 'gnus-post-news "gnuspost" "Post news." t)
(defvar hyperb:host-domain "@informatik.uni-dortmund.de")
(defvar mhg-clear-marked-assoc nil "clear the gnus-marked-assoc on startup")
(require 'hypb)
(require 'set)
(require 'comint)
(require 'shell)
;;; ************************************************************************
;;; Public variables
;;; ************************************************************************


(defvar mhg-use-vm nil 
  "Set to non nil to use vm in Article mode")

(defvar mhg-default-nntp-server "fbi-news"
  "The default NNTP server")

(defvar mhg-default-mail-server ":Mail"
  "The default mh Mail dir read with mhg.")

(defvar mhg-edit-abort-key "\C-c\C-k"
  "Key to use to abort an edit of a mail message.")

(defvar mhg-edit-end-key "\C-c\C-c"
  "Key to use to end an edit of a mail message and to save it.")

(defvar mhg-filed-to nil
  "Locations current e-mail message is to be filed to.")

(defvar mhg-root-dir nil
  "Root directory below which all auto-filed mail is to be saved.")

;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

(defun mhg-news (&optional prompt)
  "Reads news with Gnus.  With optional prefix arg PROMPT, reads server."
  (interactive "P")
  (setq gnus-nntp-server mhg-default-nntp-server
	gnus-nntp-service "nntp")
  (define-key gnus-group-mode-map   "g" 'gnus-group-get-new-news)
  (define-key gnus-summary-mode-map   " " 'gnus-summary-next-page)
  (define-key gnus-summary-mode-map "k"
    'gnus-summary-kill-same-subject-and-select)
  (gnus prompt))

(defun mhg-read (&optional prompt mhg-server)
  "Incorporates new mail and reads under Gnus in Mh mail.
Optional prefix argument forces PROMPT for mail-path under which to read
folders."
  (interactive (list current-prefix-arg
                     (read-string "Server: " mhg-default-mail-server)))
  (require 'mh-e) (mh-find-path)
  (if (null mh-folder-list)
      (setq mh-folder-list (mh-make-folder-list)))
  (if (call-process "inc" nil nil "-silent")
      ;; New mail was read, so refile it using personal rules.
      (progn (message "Categorizing new messages...")
             (call-process "inc" nil nil "-silent" "-file" 
                           (expand-file-name "~/vmail/INBOX") "-truncate")
	     ;;(if mhg-use-vm (mhg-refile-vm) (mhg-refile))
             (mhg-refile)
	     (message "Categorizing new messages...Done")
	     (if (equal (buffer-name) gnus-group-buffer)
		 (gnus-group-restart))
	     )
    (if (equal (buffer-name) gnus-group-buffer)
	(gnus-group-get-new-news))
    )
  (if (equal (buffer-name) gnus-group-buffer)
      nil
    (setq gnus-nntp-server mhg-server
	  gnus-nntp-service nil)
    (define-key gnus-summary-mode-map "e" 'mhg-msg-edit)
    (define-key gnus-article-mode-map "e" 'mhg-msg-edit)
    (define-key gnus-summary-mode-map "\M-r" 'mhg-refile-to1)
    (define-key gnus-article-mode-map "\M-R" 'mhg-refile)
    (define-key gnus-article-mode-map "v" 'mhg-msg-visit)
    (define-key gnus-summary-mode-map "v" 'mhg-msg-visit)
    (define-key gnus-group-mode-map   "g" 'mhg-read)
    (define-key gnus-summary-mode-map "k" 'mhg-msg-delete)
    (define-key gnus-summary-mode-map "W" 'mhg-where-refile)
    (define-key gnus-article-mode-map "W" 'mhg-where-refile)
    (define-key gnus-article-mode-map "w" 'mhg-where-refile)
    (if mhg-clear-marked-assoc
        (setq gnus-marked-assoc nil))
    (gnus prompt)))


(defun mhg-msg-delete ()
  "Removes message that point is on."
  (interactive)
  (let ((msg (gnus-summary-article-number)))
    (if (= 0 (call-process "rmm" nil nil t
                           (concat "+" mhspool-current-directory) 
                           (int-to-string msg)))
        (message "(msg-delete): Deleted msg %d." msg)
      (error "(msg-delete): Could not delete msg %d." msg))
    (gnus-summary-mark-as-unread-forward 1)))

(defun mhg-msg-visit ()
  "Calls vm-visit-buffer on message."
  (interactive)
  (require 'vm)
  (let* ((msg (gnus-summary-article-number))
         (file (concat mhspool-current-directory 
                 (int-to-string msg)))
         (vm-startup-with-summary nil)
         (vm-mutable-windows nil)
         )
    (if (file-exists-p file)
        (progn
          (gnus-summary-mark-as-read msg "D")
          (gnus-summary-next-subject 1 'unread-only)
          (mhg-fix-msg file)
          (vm file))
      (error "Message \"%s: %d\" not found!" gnus-newsgroup-name msg))))

(defun mhg-fix-msg (file)
  (find-file file)
  (beginning-of-buffer)
  (if (looking-at "^From ") ()
    (if (re-search-forward "^From: \\(.*\\)" (point-max) t)
        (progn
          (beginning-of-buffer)
          (insert "From " 
                  (buffer-substring (match-beginning 1)  (match-end 1))
                  "\n"))
      (insert "From anonymous@anywhere\n"))))
  

(defun mhg-msg-edit ()
  "Edits current message.  Use 'mhg-edit-end-key' to end edit."
  (interactive)
  (let* ((msg (gnus-summary-article-number))
	 (name (concat (file-name-as-directory mhspool-current-directory)
		       (int-to-string msg)))
	 (path (concat (file-name-as-directory mh-user-path) name)))
    (cond ((not (file-exists-p path))
	   (error "(msg-edit): \"%s\" does not exist."))
	  ((not (file-readable-p path))
	   (error "(msg-edit): \"%s\" is not readable."))
	  ((not (file-writable-p path))
	   (error "(msg-edit): \"%s\" is not writable."))
	  (t (setq mhg-buffer (current-buffer))
	     (funcall (if (equal (buffer-name) gnus-summary-buffer)
			  'find-file-other-window 'find-file)
		      path)
	     (buffer-flush-undo (current-buffer))
	     (auto-save-mode -1)
	     (rename-buffer name)
	     (local-set-key mhg-edit-abort-key 'mhg-msg-edit-abort)
	     (local-set-key mhg-edit-end-key 'mhg-msg-edit-end)
	     (message "Use {%s} to finish editing, {%s} to abort."
		      (key-description mhg-edit-end-key)
		      (key-description mhg-edit-abort-key)
		      )))))

(defun mhg-msg-edit-abort ()
  "Abort in progress edit of a mail message and return to previous buffer."
  (interactive)
  (set-buffer-modified-p nil)
  (kill-buffer nil)
  (let ((wind (get-buffer-window mhg-buffer)))
    (if wind (select-window wind)
      (switch-to-buffer mhg-buffer))))

(defun mhg-msg-edit-end ()
  "End an edit of a saved mail message and return to previous buffer."
  (interactive)
  (save-buffer) (kill-buffer nil)
  (let ((wind (get-buffer-window mhg-buffer)))
    (if wind (select-window wind)
      (switch-to-buffer mhg-buffer))))

(defun mhg-get-folder-names (auto-fdrs)
  (interactive)
  (let* ((maildir (concat "~/" 
                          (substring gnus-nntp-server 1)))
         (defaultf (if auto-fdrs 
                       (concat maildir "/" (car auto-fdrs))))
          folders done)
    (while (not done)
      (let ((fname (read-file-name (format "Refile to folder (%s): "
                                       defaultf)
                               maildir 
                               defaultf
                               nil
                               nil)))
             (if (> (length fname) (1+ (length maildir)))
                 (let ((folder
                       (substring fname
                                  (1+ (length maildir)))))
                   (setq  folders (cons folder folders)))
               (setq done 1)
               )))
    folders))

(defun mhg-refile (&optional folder msg-list folders)
  "Refile from optional Mh FOLDER, the msgs in MSG-LIST to FOLDERS.
'inbox' is used if FOLDER is nil.  All messages in FOLDER are refiled if
MSG-LIST (list of msg number strings) is nil (interactively with a prefix
argument).  If FOLDERS is nil, 'mhg-user-refile-rules' is a
function of no arguments called to compute the list of FOLDERS for each
message. It is called within a buffer containing only the current
message."
  (interactive (list mhspool-current-directory
		     (if current-prefix-arg nil
		       (list (int-to-string (gnus-summary-article-number))))
		     (let ((auto-fdrs (set:remove
				       gnus-newsgroup-name
				       (mhg-user-refile-rules))))
		       (read-minibuffer "Refile to list of folders: "
					(if auto-fdrs
					    (prin1-to-string auto-fdrs)
					  "()"))
		       )))
  (or folder (setq folder "inbox"))
  (let* ((folder-path (concat mh-user-path folder))
	 (inform (and (interactive-p) (null msg-list)))
	 fdr-str fdr-list refile-cmd buf mkdir)
    (or msg-list (setq msg-list (directory-files folder-path nil "^[0-9]+$")))
    (if inform (message "Re-categorizing msgs from '+%s'..." folder))
    (mhg-send-command (concat "folder +" folder))
    (mapcar '(lambda (msg)
		(set-buffer
		 (setq buf (find-file-noselect (concat folder-path "/" msg))))
		(setq fdr-str
		      (mapconcat
		       '(lambda (fdr) (concat "+" fdr))
		       (setq fdr-list
				 (set:remove
				  folder
				  (or folders
				      (mhg-user-refile-rules))))
		       " ")
		      refile-cmd (concat "refile " fdr-str " " msg)
		      mkdir nil)
		(kill-buffer buf)
		(mapcar
		 '(lambda (fdr)
		    (setq fdr (concat mh-user-path fdr))
		    (or (file-directory-p fdr)
			(setq mkdir (concat mkdir "mkdir -p " fdr "; "))))
		 fdr-list)
		(mhg-send-command (concat mkdir refile-cmd))
		)
	     msg-list)
    (if inform (message "Re-categorizing msgs from '+%s'...Done" folder))
    ))

(defun mhg-refile-to1 (&optional folder msg-list folders)
  "Refile from optional Mh FOLDER, the msgs in MSG-LIST to FOLDERS.
'inbox' is used if FOLDER is nil.  All messages in FOLDER are refiled if
MSG-LIST (list of msg number strings) is nil (interactively with a prefix
argument).  If FOLDERS is nil, 'mhg-user-refile-rules' is a
function of no arguments called to compute the list of FOLDERS for each
message. It is called within a buffer containing only the current
message."
  (interactive (list mhspool-current-directory ;; gnus-newsgroup-name
		     (if current-prefix-arg nil
		       (list (int-to-string (gnus-summary-article-number))))
		     (let* ((article (gnus-summary-article-number))
                           (dummy 
                            (if (buffer-file-name) 'article
                                (progn 
                                  (gnus-summary-display-article article)
                                  (switch-to-buffer gnus-article-buffer))))
                           (auto-fdrs (set:remove
				       gnus-newsgroup-name
				       (mhg-user-refile-rules))))
                           (if (eq dummy 'article) () 
                         (switch-to-buffer gnus-summary-buffer))
                       (gnus-summary-mark-as-unread article)
                       (mhg-get-folder-names auto-fdrs))))
  (or folder (setq folder "inbox"))
  (let* ((folder-path (concat mh-user-path folder))
	 (inform (and (interactive-p) (null msg-list)))
	 fdr-str fdr-list refile-cmd buf mkdir)
    (or msg-list (setq msg-list (directory-files folder-path nil "^[0-9]+$")))
    (if inform (message "Re-categorizing msgs from '+%s'..." folder))
    (mhg-send-command (concat "folder +" folder))
    (mapcar '(lambda (msg)
		(set-buffer
		 (setq buf (find-file-noselect (concat folder-path "/" msg))))
		(setq fdr-str
		      (mapconcat
		       '(lambda (fdr) (concat "+" fdr))
		       (setq fdr-list
				 (set:remove
				  folder
				  (or folders
				      (mhg-user-refile-rules))))
		       " ")
		      refile-cmd (concat "refile " fdr-str " " msg)
		      mkdir nil)
		(kill-buffer buf)
		(mapcar
		 '(lambda (fdr)
		    (setq fdr (concat mh-user-path fdr))
		    (or (file-directory-p fdr)
			(setq mkdir (concat mkdir "mkdir -p " fdr "; "))))
		 fdr-list)
                (mhg-send-command (concat mkdir refile-cmd))
		)
	     msg-list)
    (if inform (message "Re-categorizing msgs from '+%s'...Done" folder))
    ))

(defun mhg-rmail-to-mh (folder)
  "Outputs from current Rmail message to last into an Mh FOLDER.
Ignores deleted messages.  FOLDER should not begin with a '+'.
The current buffer must contain a message file in rmail-mode."
  (interactive)
  (if (not (eq major-mode 'rmail-mode))
      (error "(rmail-to-mh): Read in an rmail file and move to first msg.")
    (require 'mh-e)
    (mh-find-path)
    (let* ((folder-path (concat mh-user-path folder))
	   (last 0))
      (while (/= last rmail-current-message)
	(setq last rmail-current-message)
	(rmail-output (format "%s/%d" folder-path rmail-current-message))
	(or rmail-delete-after-output (rmail-next-undeleted-message 1))
	))))

(defun mhg-where-refile ()
  "Prints to minibuffer list of folders to which msg would be auto-refiled."
  (interactive)
  (if (equal (buffer-name) gnus-summary-buffer)
      (let ((msg (gnus-summary-article-number)))
	(if (or (null gnus-current-article)
		(/= msg gnus-current-article))
	    (gnus-summary-display-article msg))
	(set-buffer gnus-article-buffer)))
  (call-interactively 'mhg-user-refile-rules))

;;; ************************************************************************
;;; Public util functions
;;; ************************************************************************

(defun rm (&optional server)
  "Incorporates new mail and reads under Gnus in Mh mail."
  (interactive)
  (if server () (setq server mhg-default-mail-server))
  (if (get-buffer "*Newsgroup*")
      (if (string= gnus-nntp-server server)
        (progn
          (switch-to-buffer "*Newsgroup*")
          (gnus-group-get-new-news))
        (mhg-read nil server))
    (mhg-read nil server)))

(defun rn (&optional prompt)
  "Reads news with Gnus."
  (interactive)
  (if (get-buffer "*Newsgroup*")
      (if (string= gnus-nntp-server mhg-default-nntp-server)
        (progn
          (switch-to-buffer "*Newsgroup*")
          (gnus-group-get-new-news))
        (mhg-news prompt))
    (mhg-news prompt)))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************


(defun mhg-file-to (&rest files)
  "Saves a copy or link to current mail msg in rest of args FILES."
  (setq mhg-filed-to (append mhg-filed-to files))
  )

(defun mhg-hdr-regexp (field &rest regexps)
  "Returns t if mail FIELD contains rest of args REGEXPS in any order."
  (and field
       (not (memq nil
		  (mapcar '(lambda (regexp) (string-match regexp field))
			  regexps)))))

(defun mhg-hdr-string (field &rest strings)
  "Returns t if mail FIELD contains rest of args STRINGS in any order."
  (and field
       (not (memq nil
		  (mapcar '(lambda (str)
			     (string-match (regexp-quote str) field))
			  strings)))))

;;; ************************************************************************
;;; Utils to make mhg run without hyperbole
;;; ************************************************************************


(defun mhg-mail-shell ()
  (interactive)
  (cond ((not (comint-check-proc "*Mail-shell*"))
	 (let* ((prog (or explicit-shell-file-name
			  (getenv "ESHELL")
			  (getenv "SHELL")
			  "/bin/sh"))		     
		(name (file-name-nondirectory prog))
		(startfile (concat "~/.emacs_" name))
		(xargs-name (intern-soft (concat "explicit-" name "-args"))))
	   (set-buffer (apply 'make-comint "Mail-shell" prog
			      (if (file-exists-p startfile) startfile)
			      (if (and xargs-name (boundp xargs-name))
				  (symbol-value xargs-name)
                                '("-i"))))
	   (shell-mode)))))

(defun mhg-send-command (command)
  (interactive "sstring to send: ")
  (let ((buffer (get-buffer-create "*Mail-shell*")))
    (save-window-excursion
      (pop-to-buffer buffer)
      (message (buffer-name))
      (mhg-mail-shell)
      (end-of-buffer)
      (insert-string command)
      (comint-send-input))))

;; Emacs 19
(defvar hyperb:emacs19-p nil)
(if hyperb:emacs19-p 
    (progn
      (setq menu-bar-final-items (cons 'mhg menu-bar-final-items))
      (defvar menu-bar-mhg-menu (make-sparse-keymap "MHG"))
      (define-key gnus-group-mode-map [menu-bar mhg]
        (cons "MHG" menu-bar-mhg-menu))
      (define-key gnus-summary-mode-map [menu-bar mhg]
        (cons "MHG" menu-bar-mhg-menu))
      (define-key gnus-article-mode-map [menu-bar mhg]
        (cons "MHG" menu-bar-mhg-menu))
      (define-key menu-bar-mhg-menu [vmbox] (cons "refile INBOX" 'inc-inbox))

      (defun inc-inbox ()
        (interactive)
        (require 'mhg)
        (mhg-send-command "inc +inbox -file ~/vmail/INBOX -truncate"))
      ))
(provide 'mhg)
