#ifndef	_SYS_DPVAR_H
#define	_SYS_DPVAR_H

#ifdef	__cplusplus
extern "C" {
#endif
/*
 * Per stream state information.
 *
 * For each open stream, a dpstr structure is allocated. It is freed
 * when the stream is closed.  It establishes a correspondence between
 * the stream and an attached dp interface.
 */

typedef struct dpstr {
    struct dpstr *ds_next;	/* Linked List of Open Streams */
    struct dpstr *ds_ppanext;	/* Linked List of Streams for PPA */
    struct dpstr *ds_sapnext;	/* Linked List of Streams for SAP */
    queue_t	 *ds_urq;	/* Pointer To Our Upper Read Q */
    queue_t	 *ds_uwq;	/* Pointer To Our Upper Write Q */
    struct dp	 *ds_dp;	/* Pointer To Attached DP Interface */
    ulong	 ds_state;	/* Current DL State */
    phdr_t	 ds_dlsap;	/* Bound DLSAP */
    ulong	 ds_flags;	/* Miscellaneous Flags */
    uint	 ds_ioc_id;	/* ID of pending ioctl */
    int		 ds_minor;	/* Minor Device Number */
    kmutex_t	 ds_lock;	/* Mutex to protect this structure */
} dpstr_t;

/*
 * Values for ds_flags
 */
#define	DSF_RAW		0x0001	/* Raw mode for input (include headers) */
#define	DSF_ENABLED	0x0002	/* Enable upper write queue */

/*
 * This structure is allocated for each bound DLSAP.
 * A number of these are formed in a linked list for each PPA.
 * They point back to the original stream that they were bound to.
 */
typedef struct dpsaplist {
    phdr_t		 dsl_dlsap;	/* Bound DLSAP */
    struct dpstr	*dsl_ds;	/* Associated stream */
    struct dpsaplist	*dsl_next;	/* Next in list for this PPA */
} dpsaplist_t;


#define	DP_DEF_ACTIVE	16

/*
 * Structure to hold state information about one tcp connection.
 */
typedef struct dptcpcon {
    struct dptcpcon *dc_next,
		    *dc_prev;
    int		     dc_active;
    struct in_addr   dc_ip_src,
		     dc_ip_dst;
    u_short	     dc_th_sport,
		     dc_th_dport;
} dptcpcon_t;

/*
 * Values for dc_active
 */
#define	DC_XMIT_ACTIVE	1
#define	DC_RECV_ACTIVE	2

/*
 * All information necessary to manage a set of connection states.
 */
typedef struct dptcptab {
    dptcpcon_t *dt_mru;
    dptcpcon_t dt_con[DP_DEF_ACTIVE];	/* Should this be sized at run time? */
} dptcptab_t;

	
/*
 * Per interface state information.
 *
 * This structure is allocated/deallocated by the special ioctls
 * DPIOCIFATTACH/DPIOCIFDETACH.
 */


/*
 * Values for dp_flags
 */
#define	DPF_DOACTIVE	0x0001	/* Track active TCP connections */
#define	DPF_DOTCP	0x0002	/* Track active TCP connections */
#define	DPF_TIMEO	0x0004	/* Timeout function installed */
/*
 * Values for dp_state
 */
#define	DPS_ACTIVE	0	/* Active */
#define	DPS_ACTIVE_C	1	/* Used for timeouts only */
#define	DPS_ACTIVE_U	2	/* Used for timeouts only */
#define	DPS_WAITING	3	/* Waiting for call completion */
#define	DPS_FAILCALL	4	/* After failed call */
#define	DPS_DISCON	5	/* No traffic, initiate call on traffic */
#define	DPS_DOWN	6	/* Don't bother trying to connect */

#define	DPS_NSTATES	7

typedef struct dp {
    struct dp	 *dp_next;	/* Next dp interface */
    int		  dp_ppa;	/* Physical Point of Attachment (Unit number) */
    dpstr_t	 *dp_ds;	/* Attached streams */
    krwlock_t	  dp_dslock;	/* Protect attached streams list */
    dpsaplist_t	 *dp_dsl;	/* Bound DLSAPs */
    krwlock_t	  dp_dsllock;	/* Protect Bound DLSAP list */
    queue_t	 *dp_lwq;	/* Pointer To Our Lower Write Q */
    ulong	  dp_mtu;	/* Maximum Transmission Unit */
    ushort	  dp_flags;	/* Miscelaneous Flags */
    ushort	  dp_state;	/* Dialup State */
    int		  dp_timer;	/* Activity timeout counter */
    int		  dp_ftimer;	/* Fast Timeout counter */
    int		  dp_timeid;	/* Timeout id (returned by timeout()) */
    int		  dp_timeout[DPS_NSTATES];	/* Timeouts for each state */
    int		  dp_nactive;	/* Number of active "connections" */
    kmutex_t	  dp_timerlock;	/* Mutex for timeout facility */
    dptcptab_t	  dp_tcptab;	/* Table of active TCP connections */
    kmutex_t	  dp_tcptablock;/* Mutex for TCP table */
#ifdef	KSTAT
    struct kstat *dp_ks;	/* Allocated Kernel Stats Info */
#endif

    ulong_t	  dp_ibytes;	  /* # bytes received */
    ulong_t	  dp_ipackets;	  /* # packets received */
    ulong_t	  dp_ierrors;	  /* # total input errors */
    ulong_t	  dp_obytes;	  /* # bytes sent */
    ulong_t	  dp_opackets;	  /* # packets sent */
    ulong_t	  dp_oerrors;	  /* # total output errors */
    ulong_t	  dp_fram;	  /* # receive framing errors */
    ulong_t	  dp_crc;	  /* # receive crc errors */
    ulong_t	  dp_inits;	  /* # driver inits */
    ulong_t	  dp_nocanput;	  /* # input canput() returned false */
    ulong_t	  dp_allocbfail;  /* # allocb failed */
    ulong_t	  dp_callincoming;/* # incoming calls */
    ulong_t	  dp_callcomplete;/* # completed calls (in or out) */
    ulong_t	  dp_callattempts;/* # attempted outgoing calls */
    ulong_t	  dp_callfails;	  /* # failed calls */
    ulong_t	  dp_nomodem;	  /* # failed calls due to no modems */
} dp_t;
/*
 * Defines for DLSAP
 */

#define	DP_DLSAP_SIZE	sizeof(phdr_t)
#define	DP_SAP_SIZE	sizeof(ushort)

/*
 * Timeout constants
 */

#define	DPT_ACTIVE		DPS_ACTIVE
#define	DPT_ACTIVE_C		DPS_ACTIVE_C
#define	DPT_ACTIVE_U		DPS_ACTIVE_U
#define	DPT_WAITING		DPS_WAITING
#define	DPT_FAILCALL		DPS_FAILCALL


#define	DPT_HZ			2

#ifdef	KSTAT
/*
 * Things that netstat wants to know about.
 */
typedef struct dpstat {
    struct kstat_named dps_ipackets;
    struct kstat_named dps_ierrors;
    struct kstat_named dps_opackets;
    struct kstat_named dps_oerrors;
    struct kstat_named dps_collisions;
} dpstat_t;

#endif

/*
 * Some defines for common streams operations.
 */

#define	DP_DB_BASE(mp)		((mp)->b_datap->db_base)
#define	DP_DB_LIM(mp)		((mp)->b_datap->db_lim)
#define	DP_DB_REF(mp)		((mp)->b_datap->db_ref)
#define	DP_DB_TYPE(mp)		((mp)->b_datap->db_type)

#define	DP_MBLK_LEN(mp)		((mp)->b_wptr - (mp)->b_rptr)
#define	DP_MBLK_SIZE(mp)	((mp)->b_datap->db_lim - (mp)->b_datap->db_base)

#ifdef	__cplusplus
}
#endif

#endif	/* _SYS_DPVAR_H */
