/*
 * process.c
 *
 * Interpreter loop and program control
 *
 */

#include "frotz.h"
#include "s5api.h"


void z_sound_effect (struct sg *g)
{
}

/*
 * load_operand
 *
 * Load an operand, either a variable or a constant.
 *
 */

void load_operand (struct sg *g, zbyte type)
{
    zword value;

    if (type & 2) { 			/* variable */

	zbyte variable;

	CODE_BYTE (variable)

	if (variable == 0)
	    value = *(g->sp)++;
	else if (variable < 16)
	    value = *(g->fp - variable);
	else {
	    zword addr = g->h_globals + 2 * (variable - 16);
	    LOW_WORD (addr, value)
	}

    } else if (type & 1) { 		/* small constant */

	zbyte bvalue;

	CODE_BYTE (bvalue)
	value = bvalue;

    } else CODE_WORD (value) 		/* large constant */

    g->zargs[g->zargc++] = value;
}/* load_operand */

/*
 * load_all_operands
 *
 * Given the operand specifier byte, load all (up to four) operands
 * for a VAR or EXT opcode.
 *
 */

void load_all_operands (struct sg *g,zbyte specifier)
{
    short i;

    for (i = 6; i >= 0; i -= 2) {

	zbyte type = (specifier >> i) & 0x03;

	if (type == 3)
	    break;

	load_operand (g,type);

    }

}/* load_all_operands */

/*
 * interpret
 *
 * Z-code interpreter main loop
 *
 */

void interpret (struct sg *g)
{

    do {

	zbyte opcode;

	CODE_BYTE (opcode)

	g->zargc = 0;

	if (opcode < 0x80) {			/* 2OP opcodes */

	    load_operand (g,(zbyte) (opcode & 0x40) ? 2 : 1);
	    load_operand (g,(zbyte) (opcode & 0x20) ? 2 : 1);

	    g->var_opcodes[opcode & 0x1f] (g);

	} else if (opcode < 0xb0) {		/* 1OP opcodes */

	    load_operand (g,(zbyte) (opcode >> 4));

	    g->op1_opcodes[opcode & 0x0f] (g);

	} else if (opcode < 0xc0) {		/* 0OP opcodes */

	    g->op0_opcodes[opcode - 0xb0] (g);

	} else {				/* VAR opcodes */

	    zbyte specifier1;
	    zbyte specifier2;

	    if (opcode == 0xec || opcode == 0xfa) {	/* opcodes 0xec */
		CODE_BYTE (specifier1)                  /* and 0xfa are */
		CODE_BYTE (specifier2)                  /* call opcodes */
		load_all_operands (g,specifier1);		/* with up to 8 */
		load_all_operands (g,specifier2);         /* arguments    */
	    } else {
		CODE_BYTE (specifier1)
		load_all_operands (g,specifier1);
	    }

	    g->var_opcodes[opcode - 0xc0] (g);

	}

    } while (g->finished == 0);

    g->finished--;

}/* interpret */

/*
 * call
 *
 * Call a subroutine. Save PC and FP then load new PC and initialise
 * new stack frame. Note that the caller may legally provide less or
 * more arguments than the function actually has. The call type "ct"
 * can be 0 (z_call_s), 1 (z_call_n) or 2 (direct call).
 *
 */

void call (struct sg *g,zword routine, short argc, zword *args, short ct)
{
    long pc;
    zword value;
    zbyte count;
    short i;

    if (g->sp - g->stack < 4)
	runtime_error (g,"Stack overflow");

    GET_PC (pc)

    *--(g->sp) = (zword) (pc >> 9);		/* for historical reasons */
    *--(g->sp) = (zword) (pc & 0x1ff);	/* Frotz keeps its stack  */
    *--(g->sp) = (zword) (g->fp - g->stack - 1);	/* format compatible with */
    *--(g->sp) = (zword) (argc | (ct << 8));	/* Mark Howell's Zip      */

    g->fp = g->sp;

    /* Calculate byte address of routine */

    if (g->h_version <= V3)
	pc = (long) routine << 1;
    else if (g->h_version <= V5)
	pc = (long) routine << 2;
    else if (g->h_version <= V7)
	pc = ((long) routine << 2) + ((long) g->h_functions_offset << 3);
    else /* h_version == V8 */
	pc = (long) routine << 3;

    if (pc >= g->story_size)
	runtime_error (g,"Call to illegal address");

    SET_PC (pc)

    /* Initialise local variables */

    CODE_BYTE (count)

    if (count > 15)
	runtime_error (g,"Call to non-routine");
    if (g->sp - g->stack < count)
	runtime_error (g,"Stack overflow");

    value = 0;

    for (i = 0; i < count; i++) {

	if (g->h_version <= V4)		/* V1 to V4 games provide default */
	    CODE_WORD (value)		/* values for all local variables */

	*--(g->sp) = (zword) ((argc-- > 0) ? args[i] : value);

    }

    /* Start main loop for direct calls */

    if (ct == 2)
	interpret (g);

}/* call */

/*
 * ret
 *
 * Return from the current subroutine and restore the previous stack
 * frame. The result may be stored (0), thrown away (1) or pushed on
 * the stack (2). In the latter case a direct call has been finished
 * and we must exit the interpreter loop.
 *
 */

void ret (struct sg *g,zword value)
{
    long pc;
    short ct;

    if (g->sp > g->fp)
	runtime_error (g,"Stack underflow");

    g->sp = g->fp;

    ct = *(g->sp)++ >> 8;
    g->fp = g->stack + 1 + *(g->sp)++;
    pc = *(g->sp)++;
    pc = ((long) *(g->sp)++ << 9) | pc;

    SET_PC (pc)

    /* Handle resulting value */

    if (ct == 0)
	store (g,value);
    if (ct == 2)
	*--(g->sp) = value;

    /* Stop main loop for direct calls */

    if (ct == 2)
	g->finished++;

}/* ret */

/*
 * branch
 *
 * Take a jump after an instruction based on the flag, either true or
 * false. The branch can be short or long; it is encoded in one or two
 * bytes respectively. When bit 7 of the first byte is set, the jump
 * takes place if the flag is true; otherwise it is taken if the flag
 * is false. When bit 6 of the first byte is set, the branch is short;
 * otherwise it is long. The offset occupies the bottom 6 bits of the
 * first byte plus all the bits in the second byte for long branches.
 * Uniquely, an offset of 0 means return false, and an offset of 1 is
 * return true.
 *
 */

void branch (struct sg *g,short flag)
{
    long pc;
    zword offset;
    zbyte specifier;
    zbyte off1;
    zbyte off2;

    CODE_BYTE (specifier)

    off1 = specifier & 0x3f;

    if (!flag)
	specifier ^= 0x80;

    if (!(specifier & 0x40)) {		/* it's a long branch */

	if (off1 & 0x20)		/* propagate sign bit */
	    off1 |= 0xc0;

	CODE_BYTE (off2)

	offset = (off1 << 8) | off2;

    } else offset = off1;		/* it's a short branch */

    if (specifier & 0x80)

	if (offset > 1) {		/* normal branch */

	    GET_PC (pc)
	    pc += (short) offset - 2;
	    SET_PC (pc)

	} else ret (g,offset);		/* special case, return 0 or 1 */

}/* branch */

/*
 * store
 *
 * Store an operand, either as a variable or pushed on the stack.
 *
 */

void store (struct sg *g,zword value)
{
    zbyte variable;

    CODE_BYTE (variable)
    if (variable == 0)
	*--(g->sp) = value;
    else if (variable < 16)
	*(g->fp - variable) = value;
    else {
	zword addr = g->h_globals + 2 * (variable - 16);
	SET_WORD (addr, value)
    }

}/* store */

/*
 * direct_call
 *
 * Call the interpreter loop directly. This is necessary when
 *
 * - a sound effect has been finished
 * - a read instruction has timed out
 * - a newline countdown has hit zero
 *
 * The interpreter returns the result value on the stack.
 *
 */

short direct_call (struct sg *g,zword addr)
{
    zword saved_zargs[8];
    short saved_zargc;
    short i;

    /* Calls to address 0 return false */

    if (addr == 0)
	return 0;

    /* Save operands and operand count */

    for (i = 0; i < 8; i++)
	saved_zargs[i] = g->zargs[i];

    saved_zargc = g->zargc;

    /* Call routine directly */

    call (g,addr, 0, 0, 2);

    /* Restore operands and operand count */

    for (i = 0; i < 8; i++)
	g->zargs[i] = saved_zargs[i];

    g->zargc = saved_zargc;

    /* Resulting value lies on top of the stack */

    return (short) *(g->sp)++;

}/* direct_call */

/*
 * __extended__
 *
 * Load and execute an extended opcode.
 *
 */

void __extended__ (struct sg *g)
{
    zbyte opcode;
    zbyte specifier;

    CODE_BYTE (opcode)
    CODE_BYTE (specifier)

    load_all_operands (g,specifier);

    if (opcode < 0x1d)			/* extended opcodes from 0x1d on */
	g->ext_opcodes[opcode] (g);		/* are reserved for future spec' */

}/* __extended__ */

/*
 * __illegal__
 *
 * Exit game because an unknown opcode has been hit.
 *
 */

void __illegal__ (struct sg *g)
{

    runtime_error (g,"Illegal opcode");

}/* __illegal__ */

/*
 * z_catch, store the current stack frame for later use with z_throw.
 *
 *	no zargs used
 *
 */

void z_catch (struct sg *g)
{

    store (g,(zword) (g->fp - g->stack));

}/* z_catch */

/*
 * z_throw, go back to the given stack frame and return the given value.
 *
 *	zargs[0] = value to return
 *	zargs[1] = stack frame
 *
 */

void z_throw (struct sg *g)
{

    if (g->zargs[1] > STACK_SIZE)
	runtime_error (g,"Bad stack frame");

    g->fp = g->stack + g->zargs[1];

    ret (g,g->zargs[0]);

}/* z_throw */

/*
 * z_call_n, call a subroutine and discard its result.
 *
 * 	zargs[0] = packed address of subroutine
 *	zargs[1] = first argument (optional)
 *	...
 *	zargs[7] = seventh argument (optional)
 *
 */

void z_call_n (struct sg *g)
{

    if (g->zargs[0] != 0)
	call (g,g->zargs[0], g->zargc - 1, g->zargs + 1, 1);

}/* z_call_n */

/*
 * z_call_s, call a subroutine and store its result.
 *
 * 	zargs[0] = packed address of subroutine
 *	zargs[1] = first argument (optional)
 *	...
 *	zargs[7] = seventh argument (optional)
 *
 */

void z_call_s (struct sg *g)
{

    if (g->zargs[0] != 0)
	call (g,g->zargs[0], g->zargc - 1, g->zargs + 1, 0);
    else
	store (g,0);

}/* z_call_s */

/*
 * z_check_arg_count, branch if subroutine was called with >= n arg's.
 *
 * 	zargs[0] = number of arguments
 *
 */

void z_check_arg_count (struct sg *g)
{

    if (g->fp == g->stack + STACK_SIZE)
	branch (g,g->zargs[0] == 0);
    else
	branch (g,g->zargs[0] <= (*(g->fp) & 0xff));

}/* z_check_arg_count */

/*
 * z_jump, jump unconditionally to the given address.
 *
 *	zargs[0] = PC relative address
 *
 */

void z_jump (struct sg *g)
{
    long pc;

    GET_PC (pc)

    pc += (short)(g->zargs[0]) - 2;

    if (pc >= g->story_size)
	runtime_error (g,"Jump to illegal address");

    SET_PC (pc)

}/* z_jump */

/*
 * z_nop, no operation.
 *
 *	no zargs used
 *
 */

void z_nop (struct sg *g)
{

    /* Do nothing */

}/* z_nop */

/*
 * z_quit, stop game and exit interpreter.
 *
 *	no zargs used
 *
 */

void z_quit (struct sg *g)
{

    g->finished = 9999;

}/* z_quit */

/*
 * z_ret, return from a subroutine with the given value.
 *
 *	zargs[0] = value to return
 *
 */

void z_ret (struct sg *g)
{

    ret (g,g->zargs[0]);

}/* z_ret */

/*
 * z_ret_popped, return from a subroutine with a value popped off the stack.
 *
 *	no zargs used
 *
 */

void z_ret_popped (struct sg *g)
{

    ret (g,*(g->sp)++);

}/* z_ret_popped */

/*
 * z_rfalse, return from a subroutine with false (0).
 *
 * 	no zargs used
 *
 */

void z_rfalse (struct sg *g)
{

    ret (g,0);

}/* z_rfalse */

/*
 * z_rtrue, return from a subroutine with true (1).
 *
 * 	no zargs used
 *
 */

void z_rtrue (struct sg *g)
{

    ret (g,1);

}/* z_rtrue */
