## -*-Tcl-*-
 # ###################################################################
 #  Vince's Additions --- an extension package for Alpha
 # 
 #  FILE: "fileManipulation.tcl"
 #                                    created: 24/2/98 {1:57:08 pm} 
 #                                last update: 1999-09-06T22:40:07Z 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Mostly Copyright (c) 1998-1999  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # ###################################################################
 ##

namespace eval file {}

proc file::ensureStandardPath {n} {
    if {$n == ""} {return ""}
    set n [file nativename $n]
    # So we can handle relative path names
    if {[file pathtype $n] == "relative"} {
	set n [file join [pwd] $n]
    }
    return $n
}

proc file::showInFinder {{f {}}} {
    if {$f == ""} {set f [win::Current]}
    if {![file exists "$f"]} {
	message "ERROR: FILE NOT FOUND: \"$f\""
	return
    }
    switchTo Finder
    AEBuild Finder misc mvis "----" [makeAlis $f]
}

proc file::tryToOpen {{fname ""}} {
    if {$fname == ""} {set fname [getSelect]}
    set f [file join [file dirname [win::Current]] $fname]
    if {[file exists $f]} {
	file::openQuietly $f
    } else {
	alertnote "Sorry, I couldn't find that file.  You could install\
	  Vince's Additions which includes better include-path handling."
    }
}

proc file::ensureDirExists {dir} {
    if {![file exists $dir]} {
	set parent [file dirname $dir]
	if {$dir == "" || ($parent == $dir)} {
	    error "Can't create the folder because the disk doesn't exist."
	}
	file::ensureDirExists [file dirname $dir]
	file mkdir $dir
	return 1
    }
    return 0
}

proc file::openAny {file} {
    getFileInfo $file a
    if {![info exists a(type)] || ($a(type) == "TEXT")} {
	edit $file
	return
    } else {
	sendOpenEvent -noreply Finder "${file}"
    }
}

proc file::renameTo {} {
    set c [win::Current]
    if {![file exists $c]} { alertnote "Not a file window!" ; return }
    set new [prompt "New name for file:" [file tail $c]]
    if {[file exists [set to [file join [file dirname $c] $new]]]} {
	alertnote "Already exists!"
	return
    }
    killWindow
    file rename $c $to
    edit $to
}

proc file::standardFind {f} {
    global HOME auto_path PREFS tclExtensionsFolder file::separator
    set dirs $auto_path
    lappend dirs [file join $HOME Tcl Completions] $PREFS \
      [file join $HOME Help] [file join $HOME Tools]
    if {[info exists tclExtensionsFolder]} { lappend dirs $tclExtensionsFolder }
    foreach dir $dirs {
	if {[file exists [file join ${dir} ${f}]]} {
	    return [file join ${dir} ${f}]
	}
    }
    if {[regexp -- ${file::separator} $f]} {
	foreach dir $dirs {
	    if {[file exists [file join [file dirname $dir] $f]]} {
		return [file join [file dirname $dir] $f]
	    }
	}
    }
    error "Not found"	
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::hyperOpen" --
 # 
 #  Called by embedded hyperlinks; we look through an installation
 #  directory (and subdirs) if it is known, then the prefs directory, 
 #  then all of the auto_path.  If the file is of type TEXT, we open
 #  it, else we ask the finder to open it.
 # -------------------------------------------------------------------------
 ##
proc file::hyperOpen { name } {
    global PREFS tclExtensionsFolder auto_path file::separator
    set currD [list [file dirname [win::Current]]]
    set dirs [glob -t d -dir $currD -nocomplain -- *]
    foreach d $dirs {
	lappend currD [string trimright $d ${file::separator}]
    }
    lappend currD $PREFS 
    if {[info exists tclExtensionsFolder]} { lappend currD $tclExtensionsFolder }
    foreach d [concat $currD $auto_path] {
	if {[file exists [file join $d $name]]} {
	    file::openAny [file join $d $name]
	    return
	}
    }
    beep
    message "Sorry, couldn't find $name"
}
## 
 # -------------------------------------------------------------------------
 # 
 # "file::hyperHelpOpen" --
 # 
 #  Called by embedded hyperlinks; we look through an installation
 #  directory (and subdirs) if it is known, then the prefs directory, 
 #  then all of the auto_path.
 # -------------------------------------------------------------------------
 ##
proc file::hyperHelpOpen { name } {
    global HOME auto_path file::separator
    set currD [list [file dirname [win::Current]]]
    set dirs [glob -t d -dir $currD -nocomplain -- *]
    foreach d $dirs {
	lappend currD [string trimright $d ${file::separator}]
    }
    lappend currD [file join $HOME Help]
    foreach d [concat $currD $auto_path] {
	foreach n [glob -dir [file join $d $name] -nocomplain -- *] {
	    if {[regexp -nocase "help" [file tail $n]]} {
		edit $n
		return
	    }
	}
    }
    beep
    message "Sorry, couldn't find a help file for $name"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::jumpToCode" --
 # 
 #  It creates a hyperlink to a specific string in a code file, without
 #  requiring a mark to be defined there. It was handy for identifying places 
 #  in other packages that potentially collide with my key-bindings.
 #  
 #  Author: Jon Guyer.
 # -------------------------------------------------------------------------
 ##
proc file::jumpToCode {text file code} {
    set hyper {edit -c }
    append hyper $file
    append hyper { ; set pos [search -f 1 -r 1 "}
    append hyper $code
    append hyper {"] ; select [lindex $pos 0] [lindex $pos 1]}
    file::searchAndHyperise $text $hyper 0 3
}


proc file::sameModifiedDate {a b} {
    getFileInfo $a infoa
    getFileInfo $b infob
    # bigger = newer
    set ma $infoa(modified)
    set mb $infob(modified)
    return [expr {$ma == $mb ? 1 : 0}]
}

proc file::secondIsOlder {a b} {
    getFileInfo [stripNameCount $a] infoa
    getFileInfo [stripNameCount $b] infob
    # bigger = newer
    set ma $infoa(modified)
    set mb $infob(modified)
    return [expr {$ma > $mb ? 1 : 0}]
}

proc file::replaceSecondIfOlder {a b {complain 1} {backup ""}} {
    if {![file exists $a]} { error "file::replaceSecondIfOlder -- first file '$a' does not exist!" }
    if {[file exists $b]} {
	if {[file::secondIsOlder $a $b]} {
	    file::remove [file dirname $b] [list [file tail $b]] $backup
	    file copy $a $b
	    install::log "Copied [file tail $a] to $b"
	    return 1
	} elseif {[file::secondIsOlder $b $a]} {
	    install::log "The pre-existing [file tail $a] is newer than the one which was to be installed."
	}
    } elseif {$complain} { 
	error "file::replaceSecondIfOlder -- second file '$b' does not exist!"
    } else {
	file copy $a $b
	install::log "Copied [file tail $a] to $b"
    } 
    return 0
}

proc file::removeCheckingWins {f} {
    install::log "Removed $f"
    if {[set i [lsearch -regexp [winNames -f] "^[quote::Regfind $f]( <\d+>)?$"]] != -1} {
	bringToFront [lindex [winNames -f] $i]
	killWindow
	file delete $f
	return 1
    }
    file delete $f
    return 0
}

proc file::remove {to files {backup ""}} {
    foreach f $files {
	if {[file exists [file join $to $f]]} {
	    file::removeOne [file join $to $f] $backup
	}
    }
}

proc file::removeOne {f {backup ""}} {
    set ff [file tail $f]
    message "Removing old '$ff'"
    if {${backup} != ""} {
	if {![file exists $backup]} { file mkdir $backup }
	set i ""
	while {[file exists [file join $backup $ff$i]]} {
	    if {$i == ""} { set i 0}
	    incr i
	}
	file copy $f [file join ${backup} $ff$i]
    }
    file::removeCheckingWins $f
}

proc file::getSig {f} {
    if {[catch {getFileInfo $f arr}]} { return "" }
    return $arr(creator)
}


## 
 # ----------------------------------------------------------------------
 #	 
 #	"file::searchAndHyperise"	--
 #	
 #	 Scans through an entire file for a	particular string or
 #	 regexp, and attaches a	hyperlink of the specified form
 #	 (regsub'ed	if desired)	to the original	string.
 #			
 #	Side effects:
 #	 Many hyperlinks will be embedded in your file
 #	
 #	Arguments:
 #	 Look for 'text', replace with 'link', doing both with a regexp
 #	 if signified (regexp = 1), using colour 'col', and offsetting
 #	 the link start and end by 'startoff' and 'endoff' respectively.
 #	 This last bit is so you can search for a large pattern, but only
 #	 embed a link in a smaller part of it.
 #	 
 #	Examples: 
 #	 see 'proc install::hyperiseUrls'
 # ----------------------------------------------------------------------
 ##
proc file::searchAndHyperise { text link {regexp 0} {col 3} {startoff 0} {endoff 0}} {
    set pos [minPos]
    catch {
	while 1 {
	    set inds [search -s -f 1 -r $regexp -- $text $pos]
	    set from [lindex $inds 0]
	    set to [lindex $inds 1]
	    set realfrom $from
	    set realto $to
	    set realfrom [pos::math $realfrom + $startoff]
	    set realto [pos::math $realto + $endoff]
	    text::color $realfrom $realto $col
	    if {$link != ""} {
		if {$regexp} {
		    regsub -- $text [getText $from $to] "$link" llink
		} else {
		    set llink $link
		}
		text::hyper $realfrom $realto $llink
	    }
	    set pos $to
	}	
    }
    refresh
}
proc file::multiSearchAndHyperise {args} {
    while 1 {
	set text [lindex $args 0]
	set link [lindex $args 1]
	set args [lrange $args 2 end]
	if {$text == ""} {return}
	file::searchAndHyperise $text $link
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::findAllInstances" --
 # 
 #  Returns all instances of a given pattern in a file.  This is a regexp
 #  search, and the pattern must match all the way to the end of the 
 #  file.  Here is an example usage:
 #  
 #  	set pat2 {^.*\\(usepackage|RequirePackage)\{([^\}]+)\}(.*)$}
 #  	set subpkgs [file::findAllInstances $filename $pat2 1]
 #  
 #  Notice the pattern ends in '(.*)$', this is important.
 #  Notice that since there is one extra '()' pair in the regexp,
 #  we give '1' as the last argument.
 #  
 #  WARNING:  Calling this procedure incorrectly can easily result
 #  in an infinite loop.  This will tend to crash Alpha and is hard
 #  to debug using trace-dumps, because Alpha will tend to crash
 #  whilst tracing too!  To debug, modify the 'while' loop so that it
 #  also increments a counter, and stops after a few iterations.
 # -------------------------------------------------------------------------
 ##
proc file::findAllInstances {filename searchString {extrabrackets 0}} {
    # Get the text of the file to be searched:
    if {[lsearch [winNames -f] $filename] >= 0} {
	set fileText [getText -w $filename [minPos] [maxPos -w $filename]]
    } elseif {[file exists $filename]} {
	set fd [open $filename]
	set fileText [read $fd]
	close $fd
    } else {
	return ""
    }
    # Search the text for the search string:
    while {[string length $fileText]} {
	set dmy [lrange "d d d d d d" 0 $extrabrackets]
	if {[eval regexp -- [list $searchString] [list $fileText] $dmy match fileText]} {
	    lappend matches $match
	} else {
	    break
	}
    }
    if {[info exists matches]} {
	return $matches
    } else {
	return ""
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"file::getModeForFile" --
 #	
 #	 This is an	adaptation of Tom Pollard's	emacs mode setting facility.
 #	 I call	it from	activateHook, which	means it takes effect before
 #	 the window	yet	exists,	so you don't get a double redraw.
 #	 Here are Tom's	comments from the original:
 #	   
 #	   # Emacs-style mode selection	using first	nonblank line of file
 #	   #
 #	   # Checks	for	interpreter	line "#!/dir/subdir/command	...", or
 #	   # explicit major	mode election "-*-Mode:	vars ...-*-".
 #	   #
 #	   # "command" or "Mode" is	compared (case-insensitively) to Alpha mode
 #	   # names and first matching mode is used for the file.
 #	   #
 #	   # Author:   Tom Pollard	<pollard@chem.columbia.edu>
 #	   # Modified: 9/11/95
 #	
 #	Note: this proc actually opens the file for reading.  It _must_ close
 #	the file before exiting.  If you modify this proc, make sure that
 #	happens!
 #	
 #  To Do: I currently use 'file exists' to catch activation of non-file 
 #		   windows such as '*tcl shell*'. There may be a better way.
 #
 # --Version--Author------------------Changes-------------------------------  
 #	  1.0	  <vince@santafe.edu> first modification from Tom Pollard's
 #    1.1     <vince@santafe.edu> copes with a common Tcl/Tk exec trick.
 #    1.2     <vince@santafe.edu> can map creators if desired.
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.0} {
proc file::getModeForFile {name {winname ""}} {
    # if it doesn't exist as a file it's probably a funny window, so return
    if {![file exists "$name"]} { 
	if {[string first "* Trace" $name] == "0" } {
	    zoom
	    toggleScrollbar
	    return Tcl
	}
	return
    }
    global modeCreator file::config
    if {[info exists modeCreator([set sig [getFileSig $name]])]} {
	return $modeCreator($sig)
    }
    if {[catch [list open "$name" r] fid]} { return }
    # find first non-empty line. Return if we fail
    for { set line "" } { [string trim $line] == "" } {} {
	if { [gets $fid line] == -1} { close $fid ; return }
    }
    set ll $line
    if {$winname == ""} {set winname $name}
    while {[regexp "\\((\[a-zA-Z\]+):(\[^\\)\])\\)(.*)" $ll "" var val ll]} {
	lappend file::config($winname) [list $var $val]
    }
    if {[regexp -nocase {^[^\n\r]*[-# \(]install($|[- \)])} $line]} {
	global HOME
	if {![string match [file join ${HOME} Tcl *] $name]} {
	    if {[catch {file readlink [file join ${HOME} Tcl]} link] || ([string first $link $name] != 0)} {
		close $fid
		return "Inst"
	    }
	}
    }
    if {[regexp {^#![ 	]*([^ 	\n\r]+)} $line dmy mtch] } {
	if {[regexp {([^/]+)$} $mtch majorMode]} { 
	    # remove trailing version number
	    set majorMode [string trimright $majorMode "01234567890."]
	    if {$majorMode == "sh"} {
		# need to check if we're using a common unix trick
		if {[gets $fid ll] != -1} {
		    while {[string index [string trimleft $ll] 0] == "#"} {
			if {[gets $fid ll] == -1} { close $fid ; return }
		    } 
		} else {
		    if {[regexp {[\n\r][ \t]*[^#][^\r\n]*[\r\n]} $line ll]} {
			set ll [string trimleft $ll]
		    } else {
			set ll ""
		    }
		}
		if {[regexp {^exec +([^ ]+) } $ll dummy ll]} {
		    regexp {([^/]+)$} [string trimright $ll "01234567890."] majorMode
		}
	    }		
	} else {
	    close $fid
	    return 
	}
    } elseif {[regexp -- {-\*- *(Mode:)? *([^ 	:;]+).*-\*-} $line "" "" majorMode]} {
	# do nothing
    } else {
	close $fid
	return
    }
    close $fid
    
    global unixMode
    set majorMode [string tolower $majorMode]
    if {[info exists unixMode($majorMode)]} {
	return $unixMode($majorMode)
    } else {
	global mode::features
	set m [array names mode::features]
	if {[set i [lsearch [string tolower $m] $majorMode]] != -1} {
	    return [lindex $m $i]
	}
    }
    return 
}
} else {
    proc file::getModeForFile {name {winname ""}} {
	# if it doesn't exist as a file it's probably a funny window, so return
	if {![file exists "$name"]} { 
	    if {[string first "* Trace" $name] == "0" } {
		zoom
		toggleScrollbar
		return Tcl
	    }
	    return
	}
	global modeCreator
	if {[info exists modeCreator([set sig [getFileSig $name]])]} {
	    return $modeCreator($sig)
	}
	if {[catch [list ::open "$name" r] fid]} { return }
	# find first non-empty line. Return if we fail
	for { set line "" } { [string trim $line] == "" } {} {
	    if { [gets $fid line] == -1} { ::close $fid ; return }
	}
	set ll $line
	if {$winname == ""} {set winname $name}
	variable config
	while {[regexp "\\((\[a-zA-Z\]+):(\[^\\)\])\\)(.*)" $ll "" var val ll]} {
	    lappend config($winname) [list $var $val]
	}
	if {[regexp -nocase {^[^\n\r]*[-# \(]install($|[- \)])} $line]} {
	    global HOME
	    if {![file::pathStartsWith $name $HOME]} {
		::close $fid
		return "Inst"
	    }
	}
	if {[regexp {^#![	]*([^	\n\r]+)} $line dmy mtch] } {
	    if {[regexp {([^/]+)$} $mtch majorMode]} { 
		# remove trailing version number
		set majorMode [string trimright $majorMode "01234567890."]
		if {$majorMode == "sh"} {
		    # need to check if we're using a common unix trick
		    if {[gets $fid ll] != -1} {
			while {[string index [string trimleft $ll] 0] == "#"} {
			    if {[gets $fid ll] == -1} { ::close $fid ; return }
			} 
		    } else {
			if {[regexp {[\n\r][ \t]*[^#][^\r\n]*[\r\n]} $line ll]} {
			    set ll [string trimleft $ll]
			} else {
			    set ll ""
			}
		    }
		    if {[regexp {^exec +([^ ]+) } $ll dummy ll]} {
			regexp {([^/]+)$} [string trimright $ll "01234567890."] majorMode
		    }
		}		
	    } else {
		::close $fid
		return 
	    }
	} elseif {[regexp -- {-\*- *(Mode:)? *([^	:;]+).*-\*-} $line "" "" majorMode]} {
	    # do nothing
	} else {
	    ::close $fid
	    return
	}
	::close $fid
	
	global unixMode
	set majorMode [string tolower $majorMode]
	if {[info exists unixMode($majorMode)]} {
	    return $unixMode($majorMode)
	} else {
	    global mode::features
	    set m [array names mode::features]
	    if {[set i [lsearch [string tolower $m] $majorMode]] != -1} {
		return [lindex $m $i]
	    }
	}
	return    
    
    }
}
    
# These are mappings required by the above proc.  If you need to extend this 
# list to include a mode you are writting, place a statement like the following 
# in your alpha::mode body
set unixMode(matlab) {MATL}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::whichModeForWin" --
 # 
 #  Copes with trailing '<2>', .orig, copy, '~',...
 # -------------------------------------------------------------------------
 ##
proc file::whichModeForWin {name} {
    regsub { <[0-9]+>$} $name "" nm
    if {[set m [file::getModeForFile $nm $name]] != ""} { return $m }
    global ModeSuffixes
    set nm [file tail $name]    
    regsub {( copy|~[0-9]*|.orig)+$} $nm "" nm
    case $nm in $ModeSuffixes
    return $winMode
}

proc file::pathEndsWith {name filelist {optionalchar ""}} {
    # This stuff is necessary on Windows where there can be a
    # variety of possible file separators (back and forwards slashes).
    # On other platforms we should find the correct file first time
    # through the loop.
    global file::separator
    lappend separators ${file::separator}
    if {${file::separator} != [file nativename ${file::separator}]} {
	lappend separators [file nativename ${file::separator}]
    }
    foreach n [list $name [file nativename $name]] {
	foreach s $separators {
	    set flist $filelist
	    if {$optionalchar != ""} {
		set reg "[quote::Regfind $s$n]${optionalchar}?$"
	    } else {
		set reg "[quote::Regfind $s$n]$"
	    }
	    while {[set ind [lsearch -regexp $flist $reg]] >= 0} {
		set f [lindex $flist $ind]
		if {[file exists $f]} {
		    return $f
		}
		set flist [lrange $flist [incr ind] end]
	    }
	}
    }
    return ""
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::pathStartsWith" --
 # 
 #  This proc must not throw an error!
 #  
 #  Checks in cross-platform way whether the given file $name lies
 #  in the given directory.  Complicated by Windows.
 # -------------------------------------------------------------------------
 ##
proc file::pathStartsWith {name prepath} {
    # to ensure forward slashes on windows
    set name [file join $name]
    if {[string match [file join $prepath *] $name]} {
	return 1
    } elseif {![catch {file readlink $prepath} link] && ([string first $link $name] == 0)} {
	return 1
    }
    return 0
}


# Below:
#		Expanded version of old 'DblClickAux.tcl'
# 
# Authors: Tom Pollard <pollard@chem.columbia.edu>
#	  Tom Scavo   <trscavo@syr.edu>
#	  Vince Darley <vince@santafe.edu>
# 
#  modified by  rev reason
#  -------- --- --- -----------
#  9/97     VMD 1.0 reorganised for new alpha distribution.
# ###################################################################
##

#############################################################################
# Take any valid Macintosh filespec as input, and return the
# corresponding absolute filespec.  Filenames without an explicit
# folder are resolved relative to the folder of the current document.
#
proc file::absolutePath {filename} {
    set	name [file tail	$filename]
    set	subdir [file dirname $filename]
    if { [string length $subdir] > 0 && [string index $subdir 0] != ":" } {
	set dir ""
    } else {
	set dir [file dirname [lindex [winNames	-f] 0]]
    }
    return [file join $dir$subdir $name]
}

#############################################################################
# Open the file specified by the full pathname "$filename"
# If it's already open, just switch to it without any fuss.
#
proc file::openQuietly {filename} {
    edit -c -w $filename
}

if {[info tclversion] < 8.0} {
#############################################################################
# Searches $filename for the given pattern $searchString.  If the 
# search is successful, returns the matched string; otherwise returns
# the empty string.  If the flag 'indices' is true and the search is
# successful, returns a list of two pos giving the indices of the
# found string; otherwise returns the list '-1 -1'.
#
proc file::searchFor {filename searchString {indices 0}} {
    # Get the text of the file to be searched:
    if {[lsearch [winNames -f] $filename] >= 0} {
	set fileText [getText -w $filename [minPos] [maxPos -w $filename]]
    } elseif {[file exists $filename]} {
	set fd [open $filename]
	set fileText [read $fd]
	close $fd
    } else {
	if { $indices } {
	    return [list -1 -1]
	} else {
	    return ""
	}
    }
    # Search the text for the search string:
    if { $indices } {
	if {[regexp -indices $searchString $fileText mtch]} {
	    # Fixes an apparent bug in 'regexp':
	    return [list [lindex $mtch 0] [expr {[lindex $mtch 1] + 1}]]
	} else {		
	    return [list -1 -1]
	}
    } else {
	if {[regexp -- $searchString $fileText mtch]} {
	    return $mtch
	} else {		
	    return ""
	}
    }
}

#############################################################################
#  Read and return the complete contents of the specified file.
#
proc file::readAll {fileName} {
    if {[file exists $fileName] && [file readable $fileName]} {
	set fileid [open $fileName "r"]
	set contents [read $fileid]
	close $fileid
	return $contents
    } else {
	error "No readable file found"
    }
}


#############################################################################
#  Save $text in $filename.  If $text is null, create an empty file.
#  Overwrite if {$overwrite} is true or the file does not exist; 
#  otherwise, prompt the user.
#
proc file::writeAll {filename {text {}} {overwrite 0}} {
    if { $overwrite || ![file exists $filename] } {
	message "Saving $filename"
	set fd [open $filename "w"]
	puts $fd $text
	close $fd
    } else {
	if {[dialog::yesno "File $filename exists!  Overwrite?"]} {
	    file::writeAll $filename $text 1
	} else {
	    message "No file written"
	}
    }
}
} else {
    #############################################################################
    # Searches $filename for the given pattern $searchString.  If the 
    # search is successful, returns the matched string; otherwise returns
    # the empty string.  If the flag 'indices' is true and the search is
    # successful, returns a list of two pos giving the indices of the
    # found string; otherwise returns the list '-1 -1'.
    #
    proc file::searchFor {filename searchString {indices 0}} {
	# Get the text of the file to be searched:
	if {[lsearch [winNames -f] $filename] >= 0} {
	    set fileText [getText -w $filename [minPos] [maxPos -w $filename]]
	} elseif {[file exists $filename]} {
	    set fd [::open $filename]
	    set fileText [::read $fd]
	    ::close $fd
	} else {
	    if { $indices } {
		return [list -1 -1]
	    } else {
		return ""
	    }
	}
	# Search the text for the search string:
	if { $indices } {
	    if {[regexp -indices -- $searchString $fileText mtch]} {
		# Fixes an apparent bug in 'regexp':
		return [list [pos::math [minPos] + [lindex $mtch 0]] \
		  [pos::math [minPos] + [expr {[lindex $mtch 1] + 1}]]]
	    } else {		
		return [list -1 -1]
	    }
	} else {
	    if {[regexp -- $searchString $fileText mtch]} {
		return $mtch
	    } else {		
		return ""
	    }
	}
    }
    
#############################################################################
    #  Read and return the complete contents of the specified file.
    #
    proc file::readAll {fileName} {
	if {[file exists $fileName] && [file readable $fileName]} {
	    set fileid [::open $fileName "r"]
	    set contents [::read $fileid]
	    ::close $fileid
	    return $contents
	} else {
	    error "No readable file found"
	}
    }
    

    #############################################################################
    #  Save $text in $filename.  If $text is null, create an empty file.
    #  Overwrite if {$overwrite} is true or the file does not exist; 
    #  otherwise, prompt the user.
    #
    proc file::writeAll {filename {text {}} {overwrite 0}} {
	if { $overwrite || ![file exists $filename] } {
	    message "Saving $filename"
	    set fd [::open $filename "w"]
	    puts $fd $text
	    ::close $fd
	} else {
	    if {[dialog::yesno "File $filename exists!  Overwrite?"]} {
		file::writeAll $filename $text 1
	    } else {
		message "No file written"
	    }
	}
    }
}
    

#############################################################################
#  Highlight (select) a particular line in the designated file, opening the
#  file if necessary.  Returns the full name of the buffer containing the
#  opened file.  If provided, a message is displayed on the status line.
#
proc file::gotoLine {fname line {mesg {}}} {
    if {[lsearch [winNames -f] "*$fname"] >= 0} {
	bringToFront $fname
    } elseif {[lsearch [winNames] "*$fname"] >= 0} {
	bringToFront $fname
    } elseif {[file exists $fname]} {
	edit $fname
	catch {shrinkWindow 2}
    } else {
	alertnote "File \" $fname \" not found."
	return
    }
    set pos [rowColToPos $line 0]
    select [lineStart $pos] [nextLineStart $pos]
    if {[string length $mesg]} { message $mesg }
    return [win::Current]
}

#############################################################################
# Return a list of all subfolders found within $folder,
# down to some maximum recursion depth.  The top-level
# folder is not included in the returned list.
#
proc file::hierarchy {folder {depth 3}} {
    set folders {}
    if {$depth > 0} {
	global file::separator
	incr depth -1
	if {[string length [file tail $folder]] > 0} {
	    set folder "$folder${file::separator}"
	}
	foreach m [glob -nocomplain -path $folder -- *] {
	    if {[file isdirectory $m]} {
		set folders [concat $folders [list $m]]
		set folders [concat $folders [file::hierarchy ${m}${file::separator} $depth]]
	    }
	}
    }
    return $folders
}

proc file::recurse {dir} {
    if {![file exists $dir]} {return {}}
    set files {}
    foreach f [glob -nocomplain -dir $dir -- *] {
	if {[file isdir $f]} {
	    eval lappend files [file::recurse $f]
	} else {
	    lappend files $f
	}
    }
    return $files
}

proc file::touch {f {depth 3}} {
    if {[file isfile $f]} {
	setFileInfo $f modified [now]
	return
    }
    if {$depth == 0} {return}
    foreach ff [glob -nocomplain -dir $f *] {
	file::touch $ff [expr {$depth -1}]
    }
}

proc file::revertThese {args} {
    foreach f $args {
	foreach w [winNames -f] {
	    set ww $w
	    regsub { <[0-9]+>$} $w {} w
	    if {$f == $w} {
		bringToFront $ww
		revert
	    }
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::completeFromDir" --
 # 
 #  Here's a good example:
 # 
 #    set filename [prompt::statusLineComplete "Open which header" \
 #       [list file::completeFromDir $universalHeadersFolder] \
 #        -nocache -tryuppercase]
 #  
 #  Returns the list of files in '$dir' which start with '$f'.
 # -------------------------------------------------------------------------
 ##
proc file::completeFromDir {dir f} {
    set old [pwd]
    cd $dir
    set res [glob -nocomplain -dir $f *]
    cd $old
    return $res
}

proc file::translatePathDelimiters {path} {
    if {[file exists $path]} {
	return [file nativename $path]
    }
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    regsub -all "/" $path ":" path
	}
	"windows" {
	    if {[regexp -nocase {[a-z]:} $path vol]} {
		regsub -all ":" [string range $path 2 end] "/" path
		set path $vol$path
	    } else {
		regsub -all ":" $path "/" path
	    }
	}
	"unix" {
	    regsub -all ":" $path "/" path
	}
    }
    return $path
}

if {$tcl_platform(platform) == "macintosh"} {
    proc file::decompress {file} {
	global unstuffSig
	set downloadFolder [file dirname $file]
	set file [file tail $file]
	# install
	set filepre [lindex [split $file .] 0]
	# decode the downloaded file (this may happen automatically)
	set f_encoded [glob -nocomplain -path [file join $downloadFolder $filepre] *{.hqx,.bin}]
	set f_stuffed [glob -nocomplain -path [file join $downloadFolder $filepre] *.sit]
	if {[llength $f_encoded] == 1} {
	    if {[llength $f_stuffed] == 1} {
		# downloader was set to decode automatically --- we must wait
		set ff [lindex $f_stuffed 0]
		while {![file writable $ff]} {
		    switchTo '$unstuffSig'
		}
		switchTo 'ALFA'
	    } else {
		# downloader not set to decode automatically
		set ff [lindex $f_encoded 0]
		message "Decoding [file tail $ff]"
		set name [file tail [app::launchFore $unstuffSig]]
		sendOpenEvent -r '$unstuffSig' $ff
	    }
	}
	# decompress the downloaded file (this may happen automatically)
	set f_stuffed [glob -nocomplain -path [file join $downloadFolder $filepre] *.sit]
	set f_results [glob -t TEXT -nocomplain -path [file join $downloadFolder $filepre] *]
	eval lappend f_results [glob -nocomplain -path [file join $downloadFolder $filepre] -t d *]
	set f_results [eval lremove [list $f_results] $f_stuffed $f_encoded]
	if {[llength $f_results] == 0} {
	    # we didn't decompress automatically
	    set ff [lindex $f_stuffed 0]
	    message "Decompressing [file tail $ff]"
	    set name [file tail [app::launchFore $unstuffSig]]
	    sendOpenEvent -r '$unstuffSig' $ff		
	}
    }
} else {
    proc file::decompress {file} {
	app::runScript unstuff "Unstuffing application (Stuffit Expander 5)" "" 1 0 $file
    }
}


