package InfluxDB::Client::Response::Meta;
# ABSTRACT: meta information of responses stored in an L<InfluxDB::Client::Response> instance

use strict;
use warnings;

sub new {
    my ( $class, %args ) = @_;
    
    # make sure that we only store data we want and need
    my $self = bless {
        _influxdb_version => $args{influxdb_version},
        _request_id       => $args{request_id},
        _server           => $args{server}
    } => $class;
    
    return $self;
}

sub influxdb_version {
    my ( $self, $version ) = @_;
    $self->{_influxdb_version} = $version if ($version);
    return $self->{_influxdb_version};
}

sub request_id {
    my ( $self, $id ) = @_;
    $self->{_request_id} = $id if ($id);
    return $self->{_request_id};
}


sub server {
    my ( $self, $server ) = @_;
    $self->{_server} = $server if ($server);
    return $self->{_server};
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

InfluxDB::Client::Response::Meta - meta information of responses stored in an L<InfluxDB::Client::Response> instance

=head1 VERSION

version 0.1

=head1 METHODS

=head2 new(%args)

creates a new L<InfluxDB::Client::Response::Meta> instance

The arguments are:

=over 4

=item *

C<influxdb_version> the version of the remote InfluxDB

=item *

C<request_id> the unique request id

=item *

C<server> the address and port of our remote server

=back

=head2 influxdb_version()

B<returns:> the version of the remote InfluxDB

=head2 influxdb_version($version)

Sets the InfluxDB version to C<$version>

B<returns:> the new version of the remote InfluxDB

=head2 request_id()

B<returns:> the unique request id

=head2 request_id($id)

Sets the request ID C<$id>

B<returns:> the new unique request id

=head2 server()

B<returns:> the address and port of our remote server

=head2 server($sever)

sets the remote server information to C<$server>

B<returns:> the new address and port of our remote server

=head1 AUTHOR

Thomas Berger <thomas.berger@1und1.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2017 by 1&1 Telecommunication SE.

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut
