#!/usr/bin/perl

use Modern::Perl;

use Test::More tests => 28;
use Data::Dumper;

use FindBin;
use lib "$FindBin::Bin/lib";
use T::Discovery;
use T::OverDrive;

my $EMAIL = 'nobody@nowhere.com';
#use WebService::ILS::OverDrive;
#$WebService::ILS::OverDrive::DEBUG = 1;

use_ok('WebService::ILS::OverDrive::Patron');

SKIP: {
    skip "Not testing OverDrive::Patron API, WEBSERVICE_ILS_TEST_OVERDRIVE_PATRON not set", 27
      unless $ENV{WEBSERVICE_ILS_TEST_OVERDRIVE_PATRON};

    my $od_id     = $ENV{OVERDRIVE_TEST_CLIENT_ID}
        or BAIL_OUT("Env OVERDRIVE_TEST_CLIENT_ID not set");
    my $od_secret = $ENV{OVERDRIVE_TEST_CLIENT_SECRET}
        or BAIL_OUT("Env OVERDRIVE_TEST_CLIENT_SECRET not set");
    my $od_website_id = $ENV{OVERDRIVE_TEST_WEBSITE_ID}
        or BAIL_OUT("Env OVERDRIVE_TEST_WEBSITE_ID not set");
    my $od_authorization_name = $ENV{OVERDRIVE_TEST_AUTHORIZATION_NAME} || 'odapilibrary';
    my $od_user_id = $ENV{OVERDRIVE_TEST_USER_ID}
        or BAIL_OUT("Env OVERDRIVE_TEST_USER_ID not set");
    my $od_password = $ENV{OVERDRIVE_TEST_USER_PASSWORD};

    my $od = WebService::ILS::OverDrive::Patron->new({
        test => 1,
        client_id => $od_id,
        client_secret => $od_secret,
    });

    ok($od->auth_by_user_id($od_user_id, $od_password, $od_website_id, $od_authorization_name), "auth_by_user_id()");
    die $od->auth_token;

    my $patron = T::OverDrive::patron($od);
    my ($hold_limit, $checkout_limit, $active);
    if ($patron) {
        $hold_limit = $patron->{hold_limit};
        $checkout_limit = $patron->{checkout_limit};
        $active = $patron->{active};
    }
    #BAIL_OUT("Patron not active") unless $active;

    clear($od) if $ENV{OVERDRIVE_TEST_CLEAR};

    my $init_checkouts = $od->checkouts;
    my $init_holds = $od->holds;

    my ($total_init_holds);
    if ($init_holds) {
        $total_init_holds = $init_holds->{total};
    }
    ok(defined $total_init_holds, "Holds") or diag(Dumper($init_holds));

    my ($total_init_checkouts);
    if ($init_checkouts) {
        $total_init_checkouts = $init_checkouts->{total};
    }
    ok(defined $total_init_checkouts, "Checkouts") or diag(Dumper($init_checkouts));

    my $items = T::Discovery::search_all_random_page( $od );
    BAIL_OUT("No items in search results, cannot test circulation") unless $items && @$items;

    if ($total_init_holds >= $hold_limit) {
        my $item = $init_holds->{items}[0];
        test_remove_hold($od, $item->{id});
        test_place_hold($od, $init_checkouts, $init_holds, [$item]);
    }
    else {
        my $item_id = test_place_hold($od, $init_checkouts, $init_holds, $items);
        SKIP: {
            skip "Could not place hold", 2 unless $item_id;

            test_remove_hold($od, $item_id);
        }
    }

    if ($total_init_checkouts >= $checkout_limit - 1) {
        foreach(@{ $init_checkouts->{items} }) {
            next if $_->{format};

            $od->return($_->{id}) or BAIL_OUT("Checkouts at full capacity and cannot return");
            $total_init_checkouts--;

            last if $total_init_checkouts < $checkout_limit - 1;
        }
        $init_checkouts = $od->checkouts;
    }
    BAIL_OUT("Checkouts at full capacity and no item can be returned")
      if $total_init_checkouts >= $checkout_limit;

    test_checkout($od, $init_checkouts, $items);
    $total_init_checkouts++;

    SKIP: {
        skip "Not testing checkout with format locking, OVERDRIVE_TEST_LOCK_FORMAT not set", 4
          unless $ENV{OVERDRIVE_TEST_LOCK_FORMAT};

        SKIP: {
            skip "Checkouts at full capacity, cannot test checkout with locked format", 4
              if $total_init_checkouts >= $checkout_limit;

            $init_checkouts = $od->checkouts;
            test_checkout_with_format($od, $init_checkouts, $items);
        }
    }

    subtest "Standard search" => sub { T::OverDrive::search( $od ) };

    $patron = $od->native_patron;
    ok($patron && $patron->{patronId}, "Native patron")
      or diag(Dumper($patron));

    subtest "Native search"   => sub { T::OverDrive::native_search( $od ) };
}

sub test_place_hold {
    my ($od, $init_checkouts, $init_holds, $pool_items) = @_;

    my $item = pick_unused_item($od, [@{ $init_checkouts->{items} }, @{ $init_holds->{items} }], $pool_items);
    unless ($item) {
        diag( Dumper($init_checkouts, $init_holds, $pool_items) );
        BAIL_OUT("Cannot find appropriate item to place hold");
    }

    my $item_id = $item->{id};
    my ($hold, $cannot_place_hold);
    {
        local $@;
        $hold = eval { $od->place_hold($item_id, $EMAIL, "AUTO_CHECKOUT") };
        if ($@) {
            diag("$@\n".Dumper($item));
            $cannot_place_hold = ($@ =~ m/not allowed to place a hold on this title/io);
        }
    }
    my ($hold_item_id, $hold_item_id_uc, $total_holds, $ok);
    SKIP: {
        skip "Cannot place hold", 1 if $cannot_place_hold;

        if ($hold) {
            $hold_item_id = uc $hold->{id};
            $hold_item_id_uc = uc $hold_item_id;
            $total_holds = $hold->{total};
        }
#       $ok = ok($hold_item_id_uc eq uc($item_id) && $total_holds == scalar(@$hold_items) + 1, "Place hold")
        $ok = ok($hold_item_id_uc && $hold_item_id_uc eq uc($item_id), "Place hold")
          or diag(Dumper($init_checkouts, $init_holds, $item, $hold));
    }

    SKIP: {
        skip "Cannot place hold", 2 unless $ok;

        my $holds = $od->holds;
        my $found;
        foreach (@{ $holds->{items} }) {
            if (uc($_->{id}) eq $hold_item_id_uc) {
                $found = 1;
                last;
            }
        }
        ok ($found && $holds->{total} == $init_holds->{total} + 1, "Hold in the list")
          or diag(Dumper($hold, $holds, $init_holds));

        my $same_hold = $od->place_hold($item_id, $EMAIL, "AUTO_CHECKOUT");
        ok(
            $same_hold->{id} eq $hold->{id} &&
            $same_hold->{placed_datetime} eq $hold->{placed_datetime},
            "Place same hold")
          or diag(Dumper($same_hold, $hold));
    }

    return $hold_item_id;
}

sub test_remove_hold {
    my ($od, $item_id) = @_;

    ok( $od->remove_hold($item_id), "Remove hold" );
    ok( $od->remove_hold($item_id), "Remove hold again");
}

sub test_checkout {
    my ($od, $init_checkouts, $pool_items) = @_;

    my $item = pick_unused_item($od, $init_checkouts->{items}, $pool_items, 'AVAILABLE_ONLY');
    unless ($item) {
        diag( Dumper($init_checkouts, $pool_items) );
        BAIL_OUT("Cannot find appropriate item to checkout");
    }

    my $item_id = $item->{id};
    my $checkout;
    {
        local $@;
        $checkout = eval { $od->checkout($item_id) };
        diag("$@\n".Dumper($item)) if $@;
    }
    my ($checkout_item_id, $checkout_item_id_uc);
    if ($checkout) {
        $checkout_item_id = $checkout->{id};
        $checkout_item_id_uc = uc $checkout_item_id;
    }
    my $ok = ok($checkout_item_id_uc && $checkout_item_id_uc eq uc($item_id), "Checkout")
      or diag(Dumper($init_checkouts, $item, $checkout));

    SKIP: {
        skip "Cannot checkout", 1 unless $ok;

        my $checkouts = $od->checkouts;
        my $found;
        foreach (@{ $checkouts->{items} }) {
            if (uc($_->{id}) eq $checkout_item_id_uc) {
                $found = $_;
                last;
            }
        }
#       ok ($found && $checkouts->{total} == $init_checkouts->{total} + 1, "Checkout in the list")
#       Sometimes API loses marbles when it comes to counting
        ok ($found, "Checkout in the list")
          or diag(Dumper($checkout, $checkouts, $init_checkouts));

        SKIP: {
            skip "Checkout not found", 5 unless $found;

            my $formats = $od->checkout_formats($item_id);
            $ok = $formats && scalar(@$formats);
            ok ($ok, "Checkout formats")
              or diag(Dumper($formats));

            my $same_checkout = $od->checkout($item_id);
            ok(
                $same_checkout->{id} eq $checkout->{id} &&
                $same_checkout->{checkout_datetime} eq $checkout->{checkout_datetime},
                "Same checkout not locked")
              or diag(Dumper($same_checkout, $checkout));

            ok( $od->return($item_id), "Return" );
            # This is a bug in OverDrive API
            $ok = eval { $od->return($item_id) };
            if ($@) {
                diag("Return again: $@\nPassing nevertheless");
                $ok = 1;
            }
            ok( $ok, "Return again");


            my @all_formats = (@{$formats || []}, @{ $item->{formats} || [] });
            my $format;
            SKIP: {
                skip "Not testing format locking, OVERDRIVE_TEST_LOCK_FORMAT not set", 3
                  unless $ENV{OVERDRIVE_TEST_LOCK_FORMAT};

                foreach my $f (@all_formats) {
                    next if grep { $_ eq $f } @{ $od->NON_LOCKABLE_FORMATS };

                    $format = $f;
                    last;
                }

                SKIP: {
                    skip "Checkout formats @all_formats cannot be locked in", 2
                      unless $format;

                    $checkout = $od->checkout($item_id);

                    my $res = $od->lock_format($item_id, $format);
                    ok($res eq $format, "Lock format $format") or diag("Format: $res");

                    my $same_checkout = $od->checkout($item_id);
                    ok(
                        $same_checkout->{id} eq $checkout->{id} &&
                        $same_checkout->{checkout_datetime} eq $checkout->{checkout_datetime} &&
                        $same_checkout->{format} eq $format,
                        "Same checkout format locked")
                      or diag(Dumper($same_checkout, $checkout, $format));

                    test_download_url($od, $item_id, $format);
                }
            }

            SKIP: {
                skip "Download url already tested", 1 if $format;

                foreach my $f (@all_formats) {
                    next unless grep { $_ eq $f } @{ $od->NON_LOCKABLE_FORMATS };

                    $format = $f;
                    last;
                }

                SKIP: {
                    skip "No non-lockable checkout formats @all_formats", 1
                      unless $format;

                    $checkout = $od->checkout($item_id);

                    test_download_url($od, $item_id, $format);

                    $od->return($item_id);
                }
            }
        }
    }

    return $checkout_item_id;
}

sub test_checkout_with_format {
    my ($od, $init_checkouts, $pool_items) = @_;

    my $item = pick_unused_item($od, $init_checkouts->{items}, $pool_items, 'AVAILABLE_ONLY');
    unless ($item) {
        diag( Dumper($init_checkouts, $pool_items) );
        BAIL_OUT("Cannot find appropriate item to checkout with locked format");
    }

    my $item_id = $item->{id};
    my ($checkout, $format, @all_formats);
    foreach my $f (@{ $item->{formats} || []}) {
        push @all_formats, $f;
        next if grep { $_ eq $f } @{ $od->NON_LOCKABLE_FORMATS };

        $format = $f;
        local $@;
        $checkout = eval { $od->checkout($item_id, $format) };
        # next if $@ && $@ =~ m//o;
        diag("$@\n".Dumper($item)) if $@;
        last if $checkout;
    }
    SKIP: {
        skip "Checkout formats @all_formats cannot be locked in", 4
          unless $format;

        my ($checkout_item_id, $checkout_item_id_uc, $checkout_item_format, $checkout_item_format_lc);
        if ($checkout) {
            $checkout_item_id = $checkout->{id};
            $checkout_item_id_uc = uc $checkout_item_id;
            $checkout_item_format = $checkout->{format};
            $checkout_item_format_lc = $checkout_item_format ? lc ($checkout_item_format) : "";
        }
        my $ok = ok(
            $checkout_item_id_uc && $checkout_item_id_uc eq uc($item_id) &&
            $checkout_item_format_lc eq lc($format),
            "Checkout with locked format $format"
        ) or diag(Dumper($init_checkouts, $item, $checkout));

        SKIP: {
            skip "Cannot checkout with locked format $format", 3 unless $ok;

            my $checkouts = $od->checkouts;
            my $found;
            foreach (@{ $checkouts->{items} }) {
                if (uc($_->{id}) eq $checkout_item_id_uc) {
                    $found = $_;
                    last;
                }
            }
            ok (
                $found && $found->{format} && lc($found->{format}) eq $checkout_item_format_lc,
                "Checkout in the list"
            ) or diag(Dumper($checkout, $checkouts, $init_checkouts));

            SKIP: {
                skip "Checkout not found", 2 unless $found;

                my $same_checkout = $od->checkout($item_id);
                ok(
                    uc($same_checkout->{id}) eq $checkout_item_id_uc &&
                    $same_checkout->{checkout_datetime} eq $checkout->{checkout_datetime} &&
                    $same_checkout->{format} && lc($same_checkout->{format}) eq $checkout_item_format_lc,
                    "Same checkout without specified format")
                  or diag(Dumper($same_checkout, $checkout));

                $same_checkout = $od->checkout($item_id, $format);
                ok(
                    uc($same_checkout->{id}) eq $checkout_item_id_uc &&
                    $same_checkout->{checkout_datetime} eq $checkout->{checkout_datetime} &&
                    $same_checkout->{format} && lc($same_checkout->{format}) eq $checkout_item_format_lc,
                    "Same checkout with specified format")
                  or diag(Dumper($same_checkout, $checkout));
            }
        }

        return $checkout_item_id;
    }

    return;
}

sub test_download_url  {
    my ($od, $item_id, $format) = @_;

    my $download_url = $od->checkout_download_url(
        $item_id,
        $format,
        "http://wherever.com/failure",
        "http://wherever.com/success"
    );
    ok($download_url, "Download url");
}

sub pick_unused_item {
    my ($od, $used_items, $pool_items, $available_only) = @_;

    my ($item, $non_lockable_item);
    POOL_ITEMS_LOOP:
    foreach my $pi (@$pool_items) {
        if ($used_items) {
            my $id_uc = uc $pi->{id};
            foreach (@$used_items) {
                next POOL_ITEMS_LOOP if uc($_->{id}) eq $id_uc;
            }
        }

        next POOL_ITEMS_LOOP
          if $available_only && !($pi->{formats} && $od->is_item_available($pi->{id}));

        foreach my $f (@{ $pi->{formats} }) {
            next POOL_ITEMS_LOOP if $f eq 'periodicals-nook';

            next if grep { $_ eq $f } @{ $od->NON_LOCKABLE_FORMATS };

            $item = $pi;
            last POOL_ITEMS_LOOP;
        }
        $non_lockable_item = $pi;
    }
    return $item || $non_lockable_item;
}

sub clear {
    my ($od) = @_;

    my $checkouts = $od->checkouts;
    eval { $od->return($_->{id}) } foreach @{$checkouts->{items}};

    my $holds = $od->holds;
    $od->remove_hold($_->{id}) foreach @{$holds->{items}};
}
