package Hex::Parser;
use strict;
use warnings;

use parent 'Exporter';
our @EXPORT_OK = qw(
    parse_intel_hex
    parse_srec_hex);

our $VERSION = '0.01';

use Hex;

sub parse_intel_hex {
    my ( $file ) = @_;

    open(my $fh, '<', $file) || die "could not open $file: $!";

    my $hex           = Hex->new;
    my $addr_high_dec = 0;

    my @hex_parts;
    while ( my $line = <$fh> ){
        $line =~ m{
		  : # intel hex start
		   [[:xdigit:]]{2}  # bytecount
		  ([[:xdigit:]]{4}) # addr
		  ([[:xdigit:]]{2}) # type
		  ([[:xdigit:]] * ) # databytes
	       	   [[:xdigit:]]{2}  # checksum
	      }ix or next;

        my $intel_type_dec = $2;
        my @bytes_hex      = unpack( '(A2)*', $3 );

        # data line?
        if ( $intel_type_dec == 0 ) {
            push @hex_parts, [ $addr_high_dec + hex($1), \@bytes_hex ];
        }

        # extended linear address type?
        elsif ( $intel_type_dec == 4 ) {
            $addr_high_dec = hex( join '', @bytes_hex ) << 16;
        }

        # extended segment address type?
        elsif ( $intel_type_dec == 2 ) {
            $addr_high_dec = hex( join '', @bytes_hex ) << 4;
        }
    }


    $hex->_set_merged_parts([ sort { $a->[0] <=> $b->[0] } @hex_parts ] );
    return $hex;
}

my %_address_length_of_srec_type = (
    0 => '4',
    1 => '4',
    2 => '6',
    3 => '8',
    4 => undef,
    5 => '4',
    6 => '6',
    7 => '8',
    8 => '6',
    9 => '4',
);

sub parse_srec_hex {
    my ( $file ) = @_;

    open my $fh, '<', $file || die "could not open file: $!";

    my $hex = Hex->new;

    my @hex_parts;
    while ( my $line = <$fh> ){
        next unless substr( $line, 0, 1 ) =~ m{s}i;

        my $type = substr $line, 1, 1;

        my $addr_length = $_address_length_of_srec_type{$type};

        $line =~ m{
		      s #srec hex start
		  ([[:xdigit:]]{1})             #type
		   [[:xdigit:]]{2}              #bytecount
		  ([[:xdigit:]]{$addr_length})  #addr
		  ([[:xdigit:]] * )             #databytes
		   [[:xdigit:]]{2}              #checksum
	      }ix or next;

        #data line?
        if ( $1 == 0 || $1 == 1 || $1 == 2 || $1 == 3) {
            push @hex_parts, [ hex $2, [ unpack '(A2)*', $3 ] ];
        }
    }

    $hex->_set_merged_parts( [ sort { $a->[0] <=> $b->[0] } @hex_parts ] );

    #sort the bytes of the record
    return $hex;
}

1;


=head1 NAME

Hex::Parser - parse intel and srec hex records

=head1 SYNOPSIS

    use Hex::Parser qw(parse_intel_hex parse_srec_hex);

    # for intel hex record
    my $hex = parse_intel_hex( 'intel.hex' );n

    # for srec hex record
    my $hex = parse_srec_hex( 'srec.hex' );

    # get 100 bytes ( hex format ) starting at address 0x100
    # every single byte that is not found is returned as undef
    my $bytes_ref = $hex->get( 0x100, 10 );

    # remove 100 bytes strting at address 0x100
    $hex->remove( 0x100, 10 );

    # write/overwrite 3 bytes starting at address 0x100
    $hex->write( 0x100, [ 'AA', 'BB', 'CC' ] );

    # dump as intel hex ( will use extended linear addresses of 32 bit addresses )
    # maximum of 10 bytes in data field
    $hex->as_intel_hex( 10, $file_handle );

    # dump as srec hex ( always tries to use smallest address )
    # maximum of 10 bytes in data field
    $hex->as_screc_hex( 10, $file_handle );

=head1 DESCRIPTION

Manipulate intel/srec hex files.

=head2 Functions

=over 12

=item C<parse_intel_hex( $intel_hex_file_name )>

Exported by Hex::Parser
Parses intel hex file. Returns Hex object.

=item C<parse_srec_hex( $srec_hex_file_name )>

Exported by Hex::Parser
Parses srec hex file. Returns Hex object.

=back

=head2 Methods of Hex

=over 12

=item C<get( $from, $count )>

Returns $count hex bytes in array reference starting at address $from.
If byte is not found, undef instead.

C<[ 'AA', '00', undef, undef, 'BC', undef ]>

=item C<remove( $from, $count )>

Removes $count bytes starting at address $from.

=item C<write( $from, $bytes_ref )>

(Over)Writes bytes starting at address $from with bytes in $bytes_ref.

=item C<as_intel_hex( $bytes_hex_a_line, $file_handle )>

Writes data as intel hex into file hanlde. Maximum of $hytes_hex_a_line in data field.
Extended linear addresses as offset are used if needed.
Extended segment addresses are not supported.

=item C<as_srec_hex( $bytes_hex_a_line, $file_handle )>

Writes data as srec into file hanlde. Maximum of $hytes_hex_a_line in data field.
Tries to use the smallest address field.

=back

=head1 LICENSE

This is released under the Artistic License.

=head1 AUTHOR

spebern <bernhard@specht.net>

=cut




