package XML::Toolkit::Builder;
use Moose;
use XML::SAX::Writer;
use XML::Filter::Moose::Class;
use XML::SAX::ParserFactory;

has namespace => (
    isa        => 'Str',
    is         => 'ro',
    lazy_build => 1,
);

sub _build_namespace { 'MyApp' }

has template => (
    isa => 'Str',
    is  => 'ro',
);

has output => (
    isa        => 'ArrayRef',
    is         => 'ro',
    default    => sub { [] },
    lazy       => 1,
    auto_deref => 1,
);

has filter => (
    isa        => 'XML::Filter::Moose::Class',
    is         => 'ro',
    lazy_build => 1,
    handles    => [qw(render)],
);

sub _build_filter {
    my %params = ( namespace => $_[0]->namespace, );
    $params{template} = $_[0]->template if defined $_[0]->template;
    XML::Filter::Moose::Class->new(%params);
}

has parser => (
    is         => 'ro',
    lazy_build => 1,
    handles    => [qw(parse_uri parse_file parse_string)]
);

sub _build_parser {
    XML::SAX::ParserFactory->parser( Handler => $_[0]->filter );
}

no Moose;
1;
__END__


=head1 NAME

XML::Toolkit::Builder - A set of tools for Building XML Classes via XML::SAX Streams

=head1 SYNOPSIS

    use XML::Toolkit::Builder;
    my $builder = XML::Toolkit::Builder->new( namespace => 'MyApp' );
    $self->builder->parse_file( $file );
    say $builder->render;

=head1 ATTRIBUTES

=over

=item namespace - The namespace for your output 

This will define the root level namespace for the classes generated by the
Builder. This defaults to "MyApp".

=item filter - An XML::SAX Filter

The render method is required. This is the class that renders the parsed
events into a set of Moose classes. XML::Filter::Moose::Class documented
elsewhere in this distribution is the default implementation.

=item template - Optionally a template to pass to the XML::Filter::Moose::Class

This is to allow customization of the output template for the generated Moose
classes.

=item output - Output Buffer

An ArrayRef that contains all of the output.

=item parser - A Parser Object

The methods parse_uri, parse_file and parse_string are required. This defaults
to a XML::SAX::ParserFactory parser.

=back 

=head1 INCOMPATIBILITIES

None reported.

=head1 BUGS AND LIMITATIONS

No bugs have been reported.

Please report any bugs or feature requests to
C<bug-xml-toolkit@rt.cpan.org>, or through the web interface at
L<http://rt.cpan.org>.


=head1 AUTHOR

Chris Prather  C<< <chris@prather.org> >>

=head1 LICENCE AND COPYRIGHT

Copyright (c) 2008, Chris Prather C<< <chris@prather.org> >>. Some rights reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.
