#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "spvm_dynamic_array.h"
#include "spvm_util_allocator.h"
#include "spvm_compiler.h"

SPVM_DYNAMIC_ARRAY* SPVM_DYNAMIC_ARRAY_new(int32_t capacity) {
  
  assert(capacity >= 0);
  
  SPVM_DYNAMIC_ARRAY* array = SPVM_UTIL_ALLOCATOR_safe_malloc_zero(sizeof(SPVM_DYNAMIC_ARRAY));
  array->length = 0;
  
  if (capacity == 0) {
    array->capacity = 128;
  }
  else {
    array->capacity = capacity;
  }
  
  int64_t values_byte_size = (int64_t)array->capacity * (int64_t)sizeof(void*);
  void** values = SPVM_UTIL_ALLOCATOR_safe_malloc_zero(values_byte_size);
  
  array->values = values;
  
  return array;
}

void SPVM_DYNAMIC_ARRAY_maybe_extend(SPVM_DYNAMIC_ARRAY* array) {
  
  assert(array);
  
  int32_t length = array->length;
  int32_t capacity = array->capacity;
  
  if (length >= capacity) {
    int32_t new_capacity = capacity * 2;
    
    int64_t new_values_byte_size = (int64_t)new_capacity * (int64_t)sizeof(void*);
    void** new_values = SPVM_UTIL_ALLOCATOR_safe_malloc_zero(new_values_byte_size);
    memcpy(new_values, array->values, capacity * sizeof(void*));
    free(array->values);
    array->values = new_values;
    
    array->capacity = new_capacity;
  }
}

void SPVM_DYNAMIC_ARRAY_free(SPVM_DYNAMIC_ARRAY* array) {
  
  free(array->values);
  free(array);
}

void SPVM_DYNAMIC_ARRAY_push(SPVM_DYNAMIC_ARRAY* array, void* value) {
  
  SPVM_DYNAMIC_ARRAY_maybe_extend(array);
  
  int32_t length = array->length;
  
  *(void**)&array->values[length] = value;
  array->length++;
}

void* SPVM_DYNAMIC_ARRAY_fetch(SPVM_DYNAMIC_ARRAY* array, int32_t index) {
  
  assert(array);
  assert(index >= 0);
  assert(index < array->length);
  
  return *(void**)&array->values[index];
}

void SPVM_DYNAMIC_ARRAY_store(SPVM_DYNAMIC_ARRAY* array, int32_t index, void* value) {
  
  assert(array);
  assert(index >= 0);
  assert(index < array->length);
  
  *(void**)&array->values[index] = value;
}

void* SPVM_DYNAMIC_ARRAY_pop(SPVM_DYNAMIC_ARRAY* array) {
  
  assert(array->length >= 0);
  
  if (array->length == 0) {
    return NULL;
  }
  else {
    array->length--;
    return *(void**)&array->values[array->length];
  }
}
