package DateTime;

use strict;

use vars qw($VERSION);

BEGIN
{
    $VERSION = '0.12';

    my $loaded = 0;
    unless ( $ENV{PERL_DATETIME_PP} )
    {
        if ( $] >= 5.006 )
        {
            require XSLoader;
            XSLoader::load( 'DateTime', $DateTime::VERSION );
        }
        else
        {
            require DynaLoader;
            @DateTime::ISA = 'DynaLoader';
            DateTime->bootstrap( $DateTime::VERSION );
        }

        $loaded = 1;
    }

    require DateTimePP unless $loaded;
}

use DateTime::Duration;
use DateTime::Language;
use DateTime::TimeZone;
use DateTime::LeapSecond;
use Params::Validate qw( validate SCALAR BOOLEAN HASHREF OBJECT );
use Time::Local ();

# for some reason, overloading doesn't work unless fallback is listed
# early.
#
# 3rd parameter ( $_[2] ) means the parameters are 'reversed'.
# see: "Calling conventions for binary operations" in overload docs.
#
use overload ( 'fallback' => 1,
               '<=>' => '_compare_overload',
               'cmp' => '_compare_overload',
               '-' => '_subtract_overload',
               '+' => '_add_overload',
             );

# Have to load this after overloading is defined, after BEGIN blocks
# or else weird crashes ensue
require DateTime::Infinite;

use constant MAX_NANOSECONDS => 1_000_000_000;  # 1E9 = almost 32 bits

use constant INFINITY     =>       100 ** 100 ** 100 ;
use constant NEG_INFINITY => -1 * (100 ** 100 ** 100);

my( @MonthLengths, @LeapYearMonthLengths );

BEGIN
{
    @MonthLengths =
        ( 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 );

    @LeapYearMonthLengths = @MonthLengths;
    $LeapYearMonthLengths[1]++;
}

{
    # I'd rather use Class::Data::Inheritable for this, but there's no
    # way to add the module-loading behavior to an accessor it
    # creates, despite what its docs say!
    my $DefaultLanguage;
    sub DefaultLanguage
    {
        my $class = shift;

        if (@_)
        {
            my $lang = shift;

            DateTime::Language->load($lang);

            $DefaultLanguage = $lang;
        }

        return $DefaultLanguage;
    }
}
__PACKAGE__->DefaultLanguage('English');

sub new
{
    my $class = shift;
    my %p = validate( @_,
                      { year   => { type => SCALAR },
                        month  => { type => SCALAR, default => 1 },
                        day    => { type => SCALAR, default => 1 },
                        hour   => { type => SCALAR, default => 0 },
                        minute => { type => SCALAR, default => 0 },
                        second => { type => SCALAR, default => 0 },
                        nanosecond => { type => SCALAR, default => 0 },
                        fractional_second =>
                        { type => SCALAR, default => undef },
                        language  => { type => SCALAR | OBJECT,
                                       default => $class->DefaultLanguage },
                        time_zone => { type => SCALAR | OBJECT,
                                       default => 'floating' },
                      }
                    );

    my $self = {};

    if ( ref $p{language} )
    {
        $self->{language} = $p{language};
    }
    else
    {
        $self->{language} = DateTime::Language->new( language => $p{language} );
    }

    $self->{tz} =
        ( ref $p{time_zone} ?
          $p{time_zone} :
          DateTime::TimeZone->new( name => $p{time_zone} )
        );

    $self->{local_rd_days} =
        $class->_ymd2rd( @p{ qw( year month day ) } );

    $self->{local_rd_secs} =
        $class->_time_as_seconds( @p{ qw( hour minute second ) } );

    $self->{rd_nanosecs} = $p{nanosecond};

    if ( defined $p{fractional_second} )
    {
        my $int = int( $p{fractional_second} );
        $self->{local_rd_secs} += $int;
        $self->{rd_nanosecs} += ( $p{fractional_second} - $int ) * MAX_NANOSECONDS;
    }

    bless $self, $class;

    $self->_normalize_nanoseconds( $self->{local_rd_secs}, $self->{rd_nanosecs} );

    # Set this explicitly since it can't be calculated accurately
    # without knowing our time zone offset, and it's possible that the
    # offset can't be calculated without having at least a rough guess
    # of the datetime's year.  This year need not be correct, as long
    # as its equal or greater to the correct number, so we fudge by
    # adding one to the local year given to the constructor.
    $self->{utc_year} = $p{year} + 1;

    $self->_calc_utc_rd;
    $self->_calc_local_rd;

    return $self;
}

sub _calc_utc_rd
{
    my $self = shift;

    delete $self->{utc_c};

    if ( $self->{tz}->is_utc || $self->{tz}->is_floating )
    {
        $self->{utc_rd_days} = $self->{local_rd_days};
        $self->{utc_rd_secs} = $self->{local_rd_secs};
    }
    else
    {
        $self->{utc_rd_days} = $self->{local_rd_days};
        $self->{utc_rd_secs} =
            $self->{local_rd_secs} - $self->_offset_from_local_time;
    }

    if ( $self->{tz}->is_floating )
    {
        $self->_normalize_seconds( $self->{utc_rd_days}, $self->{utc_rd_secs} );
    }
    else
    {
        $self->_normalize_leap_seconds( $self->{utc_rd_days}, $self->{utc_rd_secs} );
    }
}

sub _normalize_leap_seconds
{
    # args: 0 => days, 1 => seconds
    my $delta_days;

    # rough adjust - can adjust many days
    if ( $_[2] < 0 )
    {
        $delta_days = int( ($_[2] - 86399) / 86400 );
    }
    else
    {
        $delta_days = int( $_[2] / 86400 );
    }

    my $new_day = $_[1] + $delta_days;
    my $delta_seconds = 86400 * ( $new_day - $_[1] ) +
                        DateTime::LeapSecond::leap_seconds( $new_day ) -
                        DateTime::LeapSecond::leap_seconds( $_[1] );

    $_[2] -= $delta_seconds;
    $_[1] = $new_day;

    # fine adjust - up to 1 day
    my $day_length = DateTime::LeapSecond::day_length( $new_day );
    if ( $_[2] >= $day_length )
    {
        $_[2] -= $day_length;
        $_[1]++;
    }
    elsif ( $_[2] < 0 )
    {
        $day_length = DateTime::LeapSecond::day_length( $new_day - 1 );
        $_[2] += $day_length;
        $_[1]--;
    }
}

sub _calc_local_rd
{
    my $self = shift;

    delete $self->{local_c};

    # We must short circuit for UTC times or else we could end up with
    # loops between DateTime.pm and DateTime::TimeZone
    if ( $self->{tz}->is_utc || $self->{tz}->is_floating )
    {
        $self->{local_rd_days} = $self->{utc_rd_days};
        $self->{local_rd_secs} = $self->{utc_rd_secs};
    }
    else
    {
        $self->{local_rd_days} = $self->{utc_rd_days};
        $self->{local_rd_secs} = $self->{utc_rd_secs} + $self->offset;

        $self->_normalize_seconds( $self->{local_rd_days}, $self->{local_rd_secs} );
    }

    $self->_calc_local_components;
}

sub _calc_local_components
{
    my $self = shift;

    @{ $self->{local_c} }{ qw( year month day day_of_week
                               day_of_year quarter day_of_quarter) } =
        $self->_rd2ymd( $self->{local_rd_days}, 1 );

    @{ $self->{local_c} }{ qw( hour minute second ) } =
        $self->_seconds_as_components( $self->{local_rd_secs}, $self->{utc_rd_secs} );
}

sub _calc_utc_components
{
    my $self = shift;

    $self->_calc_utc_rd unless defined $self->{utc_rd_days};

    @{ $self->{utc_c} }{ qw( year month day ) } =
        $self->_rd2ymd( $self->{utc_rd_days} );

    @{ $self->{utc_c} }{ qw( hour minute second ) } =
        $self->_seconds_as_components( $self->{utc_rd_secs} );
}

sub _utc_ymd
{
    my $self = shift;

    $self->_calc_utc_components unless exists $self->{utc_c}{year};

    return @{ $self->{utc_c} }{ qw( year month day ) };
}

sub _utc_hms
{
    my $self = shift;

    $self->_calc_utc_components unless exists $self->{utc_c}{hour};

    return @{ $self->{utc_c} }{ qw( hour minute second ) };
}

sub from_epoch
{
    my $class = shift;
    my %p = validate( @_,
                      { epoch => { type => SCALAR },
                        language   => { type => SCALAR | OBJECT, optional => 1 },
                        time_zone  => { type => SCALAR | OBJECT, optional => 1 },
                      }
                    );

    my %args;
    # Note, for very large negative values this may give a blatantly
    # wrong answer.
    @args{ qw( second minute hour day month year ) } =
        ( gmtime( delete $p{epoch} ) )[ 0..5 ];
    $args{year} += 1900;
    $args{month}++;

    my $self = $class->new( %p, %args, time_zone => 'UTC' );

    $self->set_time_zone( $p{time_zone} ) if exists $p{time_zone};

    return $self;
}

# use scalar time in case someone's loaded Time::Piece
sub now { shift->from_epoch( epoch => (scalar time), @_ ) }

sub today { shift->now(@_)->truncate( to => 'day' ) }

sub from_object
{
    my $class = shift;
    my %p = validate( @_,
                      { object => { type => OBJECT,
                                    can => 'utc_rd_values',
                                  },
                        language   => { type => SCALAR | OBJECT, optional => 1 },
                      },
                    );

    my $object = delete $p{object};

    my ( $rd_days, $rd_secs ) = $object->utc_rd_values;

    my %args;
    @args{ qw( year month day ) } = $class->_rd2ymd($rd_days);
    @args{ qw( hour minute second ) } = $class->_seconds_as_components($rd_secs);

    my $new = $class->new( %p, %args, time_zone => 'UTC' );

    $new->set( nanosecond => $object->nanosecond )
        if $object->can('nanosecond');

    $new->set_time_zone( $object->time_zone )
        if $object->can('time_zone');

    return $new;
}

sub last_day_of_month
{
    my $class = shift;
    my %p = validate( @_,
                      { year   => { type => SCALAR },
                        month  => { type => SCALAR },
                        hour   => { type => SCALAR, optional => 1 },
                        minute => { type => SCALAR, optional => 1 },
                        second => { type => SCALAR, optional => 1 },
                        language   => { type => SCALAR | OBJECT, optional => 1 },
                        time_zone  => { type => SCALAR | OBJECT, optional => 1 },
                      }
                    );

    my $day = ( $class->_is_leap_year( $p{year} ) ?
                $LeapYearMonthLengths[ $p{month} - 1 ] :
                $MonthLengths[ $p{month} - 1 ]
              );

    return $class->new( %p, day => $day );
}

sub clone { bless { %{ $_[0] } }, ref $_[0] }

sub year    { $_[0]->{local_c}{year} }

sub ce_year { $_[0]->{local_c}{year} <= 0 ?
              $_[0]->{local_c}{year} - 1 :
              $_[0]->{local_c}{year} }

sub month   { $_[0]->{local_c}{month} }
*mon = \&month;

sub month_0 { $_[0]->{local_c}{month} - 1 };
*mon_0 = \&month_0;

sub month_name { $_[0]->{language}->month_name( $_[0] ) }

sub month_abbr { $_[0]->{language}->month_abbreviation( $_[0] ) }

sub day_of_month { $_[0]->{local_c}{day} }
*day  = \&day_of_month;
*mday = \&day_of_month;

sub quarter {$_[0]->{local_c}{quarter} };

sub day_of_month_0 { $_[0]->{local_c}{day} - 1 }
*day_0  = \&day_of_month_0;
*mday_0 = \&day_of_month_0;

sub day_of_week { $_[0]->{local_c}{day_of_week} }
*wday = \&day_of_week;
*dow  = \&day_of_week;

sub day_of_week_0 { $_[0]->{local_c}{day_of_week} - 1 }
*wday_0 = \&day_of_week_0;
*dow_0  = \&day_of_week_0;

sub day_name { $_[0]->{language}->day_name( $_[0] ) }

sub day_abbr { $_[0]->{language}->day_abbreviation( $_[0] ) }

sub day_of_quarter { $_[0]->{local_c}{day_of_quarter} }
*doq = \&day_of_quarter;

sub day_of_quarter_0 { $_[0]->day_of_quarter - 1 }
*doq_0 = \&day_of_quarter_0;

sub day_of_year { $_[0]->{local_c}{day_of_year} }
*doy = \&day_of_year;

sub day_of_year_0 { $_[0]->{local_c}{day_of_year} - 1 }
*doy_0 = \&day_of_year_0;

sub ymd
{
    my ( $self, $sep ) = @_;
    $sep = '-' unless defined $sep;

    return sprintf( "%0.4d%s%0.2d%s%0.2d",
                    $self->year, $sep,
                    $self->{local_c}{month}, $sep,
                    $self->{local_c}{day} );
}
*date = \&ymd;

sub mdy
{
    my ( $self, $sep ) = @_;
    $sep = '-' unless defined $sep;

    return sprintf( "%0.2d%s%0.2d%s%0.4d",
                    $self->{local_c}{month}, $sep,
                    $self->{local_c}{day}, $sep,
                    $self->year );
}

sub dmy
{
    my ( $self, $sep ) = @_;
    $sep = '-' unless defined $sep;

    return sprintf( "%0.2d%s%0.2d%s%0.4d",
                    $self->{local_c}{day}, $sep,
                    $self->{local_c}{month}, $sep,
                    $self->year );
}

sub hour   { $_[0]->{local_c}{hour} }

sub minute { $_[0]->{local_c}{minute} }
*min = \&minute;

sub second { $_[0]->{local_c}{second} }
*sec = \&second;

sub fractional_second { $_[0]->second + $_[0]->nanosecond / MAX_NANOSECONDS }

sub nanosecond { $_[0]->{rd_nanosecs} }

sub millisecond { _round( $_[0]->{rd_nanosecs} / 1000000 ) }

sub microsecond { _round( $_[0]->{rd_nanosecs} / 1000 ) }

sub _round
{
    my $val = shift;
    my $int = int $val;

    return $val - $int >= 0.5 ? $int + 1 : $int;
}

sub hms
{
    my ( $self, $sep ) = @_;
    $sep = ':' unless defined $sep;

    return sprintf( "%0.2d%s%0.2d%s%0.2d",
                    $self->{local_c}{hour}, $sep,
                    $self->{local_c}{minute}, $sep,
                    $self->{local_c}{second} );
}
# don't want to override CORE::time()
*DateTime::time = \&hms;

sub iso8601 { join 'T', $_[0]->ymd('-'), $_[0]->hms(':') }
*datetime = \&iso8601;

sub is_leap_year { $_[0]->_is_leap_year( $_[0]->year ) }

sub week
{
    my $self = shift;

    unless ( defined $self->{local_c}{week_year} )
    {
        # This algorithm was taken from Date::Calc's DateCalc.c file
        my $jan_one_dow_m1 =
            ( ( $self->_ymd2rd( $self->year, 1, 1 ) + 6 ) % 7 );

        $self->{local_c}{week_number} =
            int( ( ( $self->day_of_year - 1 ) + $jan_one_dow_m1 ) / 7 );
        $self->{local_c}{week_number}++ if $jan_one_dow_m1 < 4;

        if ( $self->{local_c}{week_number} == 0 )
        {
            $self->{local_c}{week_year} = $self->year - 1;
            $self->{local_c}{week_number} =
                $self->_weeks_in_year( $self->{local_c}{week_year} );
        }
        elsif ( $self->{local_c}{week_number} == 53 &&
                $self->_weeks_in_year( $self->year ) == 52 )
        {
            $self->{local_c}{week_number} = 1;
            $self->{local_c}{week_year} = $self->year + 1;
        }
        else
        {
            $self->{local_c}{week_year} = $self->year;
        }
    }

    return @{ $self->{local_c} }{ 'week_year', 'week_number' }
}

# Also from DateCalc.c
sub _weeks_in_year
{
    my $self = shift;
    my $year = shift;

    my $jan_one_dow =
        ( ( $self->_ymd2rd( $year, 1, 1 ) + 6 ) % 7 ) + 1;
    my $dec_31_dow =
        ( ( $self->_ymd2rd( $year, 12, 31 ) + 6 ) % 7 ) + 1;

    return $jan_one_dow == 4 || $dec_31_dow == 4 ? 53 : 52;
}

sub week_year   { ($_[0]->week)[0] }
sub week_number { ($_[0]->week)[1] }

sub time_zone { $_[0]->{tz} }

sub offset { $_[0]->{tz}->offset_for_datetime( $_[0] ) }
sub _offset_from_local_time { $_[0]->{tz}->offset_for_local_datetime( $_[0] ) }

sub is_dst { $_[0]->{tz}->is_dst_for_datetime( $_[0] ) }

sub time_zone_short_name { $_[0]->{tz}->short_name_for_datetime( $_[0] ) }

sub language { $_[0]->{language} }

sub utc_rd_values { @{ $_[0] }{ 'utc_rd_days', 'utc_rd_secs' } }

# NOTE: no nanoseconds, no leap seconds
sub utc_rd_as_seconds   { ( $_[0]->{utc_rd_days} * 86400 ) + $_[0]->{utc_rd_secs} }

# NOTE: no nanoseconds, no leap seconds
sub local_rd_as_seconds { ( $_[0]->{local_rd_days} * 86400 ) + $_[0]->{local_rd_secs} }

# RD 1 is JD 1,721,424.5 - a simple offset
sub jd
{
    my $self = shift;

    my $jd = $self->{utc_rd_days} + 1_721_424.5;

    my $day_length = DateTime::LeapSecond::day_length( $self->{utc_rd_days} );

    return ( $jd +
             ( $self->{utc_rd_secs} / $day_length )  +
             ( $self->{rd_nanosecs} / $day_length / MAX_NANOSECONDS )
           );
}

sub mjd { $_[0]->jd - 2_400_000.5 }

my %formats =
    ( 'a' => sub { $_[0]->day_abbr },
      'A' => sub { $_[0]->day_name },
      'b' => sub { $_[0]->month_abbr },
      'B' => sub { $_[0]->month_name },
#      'c' => sub { $_[0]->strftime( $_[0]->{language}->preferred_datetime_format ) },
      'C' => sub { int( $_[0]->year / 100 ) },
      'd' => sub { sprintf( '%02d', $_[0]->day_of_month ) },
      'D' => sub { $_[0]->strftime( '%m/%d/%y' ) },
      'e' => sub { sprintf( '%2d', $_[0]->day_of_month ) },
      'F' => sub { $_[0]->ymd('-') },
      'g' => sub { substr( $_[0]->week_year, -2 ) },
      'G' => sub { $_[0]->week_year },
      'H' => sub { sprintf( '%02d', $_[0]->hour ) },
      'I' => sub { sprintf( '%02d', $_[0]->_hour_12 ) },
      'j' => sub { $_[0]->day_of_year },
      'k' => sub { sprintf( '%2d', $_[0]->hour ) },
      'l' => sub { sprintf( '%2d', $_[0]->_hour_12 ) },
      'm' => sub { sprintf( '%02d', $_[0]->month ) },
      'M' => sub { sprintf( '%02d', $_[0]->minute ) },
      'n' => sub { "\n" }, # should this be OS-sensitive?
      'N' => \&_format_nanosecs,
      'p' => sub { $_[0]->{language}->am_pm( $_[0] ) },
      'P' => sub { lc $_[0]->{language}->am_pm( $_[0] ) },
      'r' => sub { $_[0]->strftime( '%I:%M:%S %p' ) },
      'R' => sub { $_[0]->strftime( '%H:%M' ) },
      's' => sub { $_[0]->epoch },
      'S' => sub { sprintf( '%02d', $_[0]->second ) },
      't' => sub { "\t" },
      'T' => sub { $_[0]->strftime( '%H:%M:%S' ) },
      'u' => sub { $_[0]->day_of_week },
      # algorithm from Date::Format::wkyr
      'U' => sub { my $dow = $_[0]->day_of_week;
                   $dow = 0 if $dow == 7; # convert to 0-6, Sun-Sat
                   my $doy = $_[0]->day_of_year - 1;
                   return int( ( $doy - $dow + 13 ) / 7 - 1 )
                 },
      'w' => sub { my $dow = $_[0]->day_of_week;
                   return $dow % 7;
                 },
      'W' => sub { my $dow = $_[0]->day_of_week;
                   my $doy = $_[0]->day_of_year - 1;
                   return int( ( $doy - $dow + 13 ) / 7 - 1 )
                 },
#      'x' => sub { $_[0]->strftime( $_[0]->{language}->preferred_date_format ) },
#      'X' => sub { $_[0]->strftime( $_[0]->{language}->preferred_time_format ) },
      'y' => sub { sprintf( '%02d', substr( $_[0]->year, -2 ) ) },
      'Y' => sub { return $_[0]->year },
      'z' => sub { DateTime::TimeZone::offset_as_string( $_[0]->offset ) },
      'Z' => sub { $_[0]->{tz}->short_name_for_datetime( $_[0] ) },
      '%' => sub { '%' },
    );

$formats{h} = $formats{b};

sub strftime
{
    my $self = shift;
    # make a copy or caller's scalars get munged
    my @formats = @_;

    my @r;
    foreach my $f (@formats)
    {
        # regex from Date::Format - thanks Graham!
        $f =~ s/
	        %([%a-zA-Z])
	       /
                $formats{$1} ? $formats{$1}->($self) : $1
               /sgex;

        #  %3N 
        $f =~ s/
                %(\d+)N
               /
                $formats{N}->($self, $1)
               /sgex;

        return $f unless wantarray;

        push @r, $f;
    }

    return @r;
}

sub _format_nanosecs
{
    my $self = shift;
    my $precision = shift;

    my $ret = sprintf( "%09d", $self->{rd_nanosecs} );
    return $ret unless $precision;   # default = 9 digits

    # rd_nanosecs might contain a fractional separator
    my ( $int, $frac ) = split /[.,]/, $self->{rd_nanosecs};
    $ret .= $frac if $frac;

    return substr( $ret, 0, $precision );
}

sub _hour_12 { my $h = $_[0]->hour % 12; return $h ? $h : 12 }

# The Time::Local included with 5.00503 doesn't have timegm_nocheck,
# but its timegm doesn't do boundary checking
my $sub =
    ( defined &Time::Local::timegm_nocheck ?
      \&Time::Local::timegm_nocheck :
      \&Time::Local::timegm
    );
sub epoch
{
    my $self = shift;

    return $self->{utc_c}{epoch}
        if exists $self->{utc_c}{epoch};

    my ( $year, $month, $day ) = $self->_utc_ymd;
    my @hms = $self->_utc_hms;

    $self->{utc_c}{epoch} =
        eval { $sub->( ( reverse @hms ),
                       $day,
                       $month - 1,
                       $year - 1900,
                     ) };

    return $self->{utc_c}{epoch};
}

sub is_finite { 1 }
sub is_infinite { 0 }

# added for benefit of DateTime::TimeZone
sub utc_year { $_[0]->{utc_year} }

sub add { return shift->add_duration( DateTime::Duration->new(@_) ) }

sub subtract { return shift->subtract_duration( DateTime::Duration->new(@_) ) }

sub subtract_duration { return $_[0]->add_duration( $_[1]->inverse ) }

sub subtract_datetime
{
    my $self = shift;
    my $dt = shift;

    my $is_floating = $self->time_zone->is_floating &&
                      $dt->time_zone->is_floating;

    # We only want a negative duration if $dt > $self.  If just the
    # seconds are greater (but the days are equal or less), then
    # returning a negative duration is wrong.

    if ( $self->{utc_rd_days} == $dt->{utc_rd_days} )
    {
        return
            DateTime::Duration->new
                ( days        => $self->{utc_rd_days} - $dt->{utc_rd_days},
                  seconds     => $self->{utc_rd_secs} - $dt->{utc_rd_secs},
                  nanoseconds => $self->{rd_nanosecs} - $dt->{rd_nanosecs},
                );
    }
    elsif ( $self->{utc_rd_days} > $dt->{utc_rd_days} &&
            $self->{utc_rd_secs} < $dt->{utc_rd_secs} )
    {
        my $days = $self->{utc_rd_days} - 1;
        my $secs;
        if ( $is_floating )
        {
            $secs = $self->{utc_rd_secs} + 86400;
        }
        else
        {
            $secs = $self->{utc_rd_secs} + DateTime::LeapSecond::day_length( $days );
        }
        return DateTime::Duration->new
            ( days        => $days - $dt->{utc_rd_days},
              seconds     => $secs - $dt->{utc_rd_secs},
              nanoseconds => $self->{rd_nanosecs} - $dt->{rd_nanosecs},
            );
    }
    elsif ( $dt->{utc_rd_days} > $self->{utc_rd_days} &&
            $dt->{utc_rd_secs} < $self->{utc_rd_secs} )
    {
        my $days = $dt->{utc_rd_days} - 1;
        my $secs;
        if ( $is_floating )
        {
            $secs = $self->{utc_rd_secs} + 86400;
        }
        else
        {
            $secs = $dt->{utc_rd_secs} + DateTime::LeapSecond::day_length( $days );
        }
        return DateTime::Duration->new
            ( days    => $self->{utc_rd_days} - $days,
              seconds => $self->{utc_rd_secs} - $secs,
              nanoseconds => $self->{rd_nanosecs} - $dt->{rd_nanosecs},
            );
    }
    else
    {
        return DateTime::Duration->new
            ( days    => $self->{utc_rd_days} - $dt->{utc_rd_days},
              seconds => $self->{utc_rd_secs} - $dt->{utc_rd_secs},
              nanoseconds => $self->{rd_nanosecs} - $dt->{rd_nanosecs},
            );
    }
}

sub _add_overload
{
    my ( $dt, $dur, $reversed ) = @_;

    if ($reversed)
    {
        ( $dur, $dt ) = ( $dt, $dur );
    }

    # how to handle non duration objects?

    my $new = $dt->clone;
    $new->add_duration($dur);
    return $new;
}

sub _subtract_overload
{
    my ( $date1, $date2, $reversed ) = @_;

    if ($reversed)
    {
        ( $date2, $date1 ) = ( $date1, $date2 );
    }

    if ( UNIVERSAL::isa( $date2, 'DateTime::Duration' ) )
    {
        my $new = $date1->clone;
        $new->add_duration( $date2->inverse );
        return $new;
    }
    else
    {
        return $date1->subtract_datetime($date2);
    }
    # handle other cases?
}

sub add_duration
{
    my ( $self, $dur ) = @_;

    my %deltas = $dur->deltas;

    foreach my $val ( values %deltas )
    {
        my $inf;
        if ( $val == INFINITY )
        {
            $inf = DateTime::Infinite::Future->new;
        }
        elsif ( $val == NEG_INFINITY )
        {
            $inf = DateTime::Infinite::Past->new;
        }

        if ($inf)
        {
            %$self = %$inf;
            bless $self, ref $inf;

            return $self;
        }
    }

    $self->{local_rd_days} += $deltas{days} if $deltas{days};

    if ( $deltas{months} )
    {
        # For preserve mode, if it is the last day of the month, make
        # it the 0th day of the following month (which then will
        # normalize back to the last day of the new month).
        my ($y, $m, $d) = ( $dur->is_preserve_mode ?
                            $self->_rd2ymd( $self->{local_rd_days} + 1 ) :
                            $self->_rd2ymd( $self->{local_rd_days} )
                          );

        $d -= 1 if $dur->is_preserve_mode;

        if ( ! $dur->is_wrap_mode && $d > 28 )
        {
            # find the rd for the last day of our target month
            $self->{local_rd_days} = $self->_ymd2rd( $y, $m + $deltas{months} + 1, 0 );

            # what day of the month is it? (discard year and month)
            my $last_day = ($self->_rd2ymd( $self->{local_rd_days} ))[2];

            # if our original day was less than the last day,
            # use that instead
            $self->{local_rd_days} -= $last_day - $d if $last_day > $d;
        }
        else
        {
            $self->{local_rd_days} = $self->_ymd2rd( $y, $m + $deltas{months}, $d );
        }
    }

    if ( $deltas{days} || $deltas{months} )
    {
        # Again, we fudge the year so that the calculations being done
        # have something to work with.
        $self->{utc_year} += int( $deltas{months} / 12 ) + 1;

        $self->_calc_utc_rd;
        $self->_calc_local_rd;
    }

    if ( $deltas{minutes} )
    {
        $self->{utc_rd_secs} += $deltas{minutes} * 60;

        $self->_normalize_seconds( $self->{utc_rd_days}, $self->{utc_rd_secs} );
    }

    # We add seconds to the UTC time because if someone adds 24 hours,
    # we want this to be _different_ from adding 1 day when crossing
    # DST boundaries.
    if ( $deltas{seconds} || $deltas{nanoseconds})
    {
        $self->{utc_rd_secs} += $deltas{seconds};

        if ( $deltas{nanoseconds} )
        {
            $self->{rd_nanosecs} += $deltas{nanoseconds};
            $self->_normalize_nanoseconds( $self->{utc_rd_secs}, $self->{rd_nanosecs} );
        }

        # must always normalize seconds, because a nanosecond change
        # might cause a day change

        if ( $self->time_zone->is_floating )
        {
            $self->_normalize_seconds( $self->{utc_rd_days}, $self->{utc_rd_secs} );
        }
        else
        {
            $self->_normalize_leap_seconds( $self->{utc_rd_days}, $self->{utc_rd_secs} );
        }
    }

    if ( $deltas{minutes} || $deltas{seconds} || $deltas{nanoseconds})
    {
        delete $self->{utc_c};
        $self->_calc_local_rd;
    }

    return $self;
}

sub _compare_overload
{
    # note: $_[1]->compare( $_[0] ) is an error when $_[1] is not a
    # DateTime (such as the INFINITY value)
    return $_[2] ? - $_[0]->compare( $_[1] ) : $_[0]->compare( $_[1] );
}

sub compare
{
    shift->_compare( @_, 0 );
}

sub compare_ignore_floating
{
    shift->_compare( @_, 1 );
}

sub _compare
{
    my ( $class, $dt1, $dt2, $consistent ) = ref $_[0] ? ( undef, @_ ) : @_;

    return undef unless defined $dt2;

    if ( ! ref $dt2 && ( $dt2 == INFINITY || $dt2 == NEG_INFINITY ) )
    {
        return $dt1->{utc_rd_days} <=> $dt2;
    }

    die "Cannot compare a datetime to a regular scalar"
        unless ( UNIVERSAL::can( $dt1, 'utc_rd_values' ) &&
                 UNIVERSAL::can( $dt2, 'utc_rd_values' ) );

    if ( ! $consistent &&
         UNIVERSAL::can( $dt1, 'time_zone' ) &&
         UNIVERSAL::can( $dt2, 'time_zone' )
       )
    {
        my $is_floating1 = $dt1->time_zone->is_floating;
        my $is_floating2 = $dt2->time_zone->is_floating;

        if ( $is_floating1 && ! $is_floating2 )
        {
            $dt1 = $dt1->clone->set_time_zone( $dt2->time_zone );
        }
        elsif ( $is_floating2 && ! $is_floating1 )
        {
            $dt2 = $dt2->clone->set_time_zone( $dt1->time_zone );
        }
    }

    my ($days1, $secs1) = $dt1->utc_rd_values;
    my ($days2, $secs2) = $dt2->utc_rd_values;

    return $days1 <=> $days2 if $days1 != $days2;

    return $secs1 <=> $secs2 if $secs1 != $secs2;

    return $dt1->nanosecond <=> $dt2->nanosecond;
}

sub _normalize_nanoseconds
{
    # seconds, nanoseconds
    if ( $_[2] < 0 )
    {
        my $overflow = 1 + int( $_[2] / MAX_NANOSECONDS );
        $_[2] += $overflow * MAX_NANOSECONDS;
        $_[1] -= $overflow;
    }
    elsif ( $_[2] >= MAX_NANOSECONDS )
    {
        my $overflow = int( $_[2] / MAX_NANOSECONDS );
        $_[2] -= $overflow * MAX_NANOSECONDS;
        $_[1] += $overflow;
    }
}


sub set
{
    my $self = shift;
    my %p = validate( @_,
                      { year     => { type => SCALAR, optional => 1 },
                        month    => { type => SCALAR, optional => 1 },
                        day      => { type => SCALAR, optional => 1 },
                        hour     => { type => SCALAR, optional => 1 },
                        minute   => { type => SCALAR, optional => 1 },
                        second   => { type => SCALAR, optional => 1 },
                        nanosecond => { type => SCALAR, optional => 1 },
                        language => { type => SCALAR, optional => 1 },
                      } );

    my %old_p =
        ( map { $_ => $self->$_() }
          qw( year month day hour minute second nanosecond language time_zone )
        );

    my $new_dt = (ref $self)->new( %old_p, %p );

    %$self = %$new_dt;

    return $self;
}

sub truncate
{
    my $self = shift;
    my %p = validate( @_,
                      { to =>
                        { regex => qr/^(?:year|month|day|hour|minute|second)$/ },
                      },
                    );

    my %new = ( language  => $self->{language},
                time_zone => $self->{tz},
              );

    foreach my $f ( qw( year month day hour minute second ) )
    {
        $new{$f} = $self->$f();

        last if $p{to} eq $f;
    }

    my $new_dt = (ref $self)->new(%new);

    %$self = %$new_dt;

    return $self;
}

sub set_time_zone
{
    my ( $self, $tz ) = @_;

    # This is a bit of a hack but it works because time zone objects
    # are singletons, and if it doesn't work all we lose is a little
    # bit of speed.
    return if $self->{tz} eq $tz;

    my $was_floating = $self->{tz}->is_floating;

    $self->{tz} = ref $tz ? $tz : DateTime::TimeZone->new( name => $tz );

    # if it either was or now is floating (but not both)
    if ( $self->{tz}->is_floating xor $was_floating )
    {
        $self->_calc_utc_rd;
    }
    elsif ( ! $was_floating )
    {
        $self->_calc_local_rd;
    }

    return $self;
}


1;

__END__

=head1 NAME

DateTime - A date and time object

=head1 SYNOPSIS

  use DateTime;

  $dt = DateTime->new( year   => 1964,
                       month  => 10,
                       day    => 16,
                       hour   => 16,
                       minute => 12,
                       second => 47,
                       nanosecond => 500000000,
                       time_zone => 'Asia/Taipei',
                     );

  $dt = DateTime->from_epoch( epoch => $epoch );
  $dt = DateTime->now; # same as ( epoch => time() )

  $year   = $dt->year;
  $month  = $dt->month;         # 1-12 - also mon

  $day    = $dt->day;           # 1-31 - also day_of_month, mday

  $dow    = $dt->day_of_week;   # 1-7 (Monday is 1) - also dow, wday

  $hour   = $dt->hour;          # 0-23
  $minute = $dt->minute;        # 0-59 - also min

  $second = $dt->second;        # 0-61 (leap seconds!) - also sec

  $doy    = $dt->day_of_year    # 1-366 (leap years) - also doy

  $doq    = $dt->day_of_quarter # 1.. - also doq

  $qtr    = $dt->quarter        # 1-4

  # all of the start-at-1 methods above have correponding start-at-0
  # methods, such as $dt->day_of_month_0, $dt->month_0 and so on

  $ymd    = $dt->ymd            # 2002-12-06
  $ymd    = $dt->ymd('/')       # 2002/12/06 - also date

  $mdy    = $dt->mdy            # 12-06-2002
  $mdy    = $dt->mdy('/')       # 12/06/2002

  $dmy    = $dt->dmy            # 06-12-2002
  $dmy    = $dt->dmy('/')       # 06/12/2002

  $hms    = $dt->hms            # 14:02:29
  $hms    = $dt->hms('!')       # 14!02!29 - also time

  $is_leap  = $dt->is_leap_year;

  # these are localizable, see LANGUAGES section
  $month_name  = $dt->month_name # January, February, ...
  $month_abbr  = $dt->month_abbr # Jan, Feb, ...
  $day_name    = $dt->day_name   # Monday, Tuesday, ...
  $day_abbr    = $dt->day_abbr   # Mon, Tue, ...

  $epoch_time  = $dt->epoch;
  # may return undef if the datetime is outside the range that is
  # representable by your OS's epoch system.

  $dt2 = $dt + $duration_object;

  $dt3 = $dt - $duration_object;

  $duration_object = $dt - $dt2;

  $dt->set( year => 1882 );

  $dt->set_time_zone( 'America/Chicago' );

=head1 DESCRIPTION

DateTime is a class for the representation of date/time combinations,
and is part of the Perl DateTime project.  For details on this project
please see L<http://datetime.perl.org/>.

It represents the Gregorian calendar, extended backwards in time
before its creation (in 1582).  This is sometimes known as the
"proleptic Gregorian calendar".  In this calendar, the first day of
the calendar (the epoch), is the first day of year 1, which
corresponds to the date which was (incorrectly) believed to be the
birth of Jesus Christ.

The calendar represented does have a year 0, and in that way differs
from how dates are often written using "BCE/CE" or "BC/AD".

For infinite datetimes, please see the
L<DateTime::Infinite|DateTime::Infinite> module.

=head1 USAGE

=head2 0-based Versus 1-based Numbers

The DateTime.pm module follows a simple consistent logic for
determining whether or not a given number is 0-based or 1-based.

Month, day of month, day of week, and day of year are 1-based.  Any
method that is 1-based also has an equivalent 0-based method ending in
"_0".  So for example, this class provides both C<day_of_week()> and
C<day_of_week_0()> methods.

The C<day_of_week_0()> method still treats Monday as the first day of
the week.

All I<time>-related numbers such as hour, minute, and second are
0-based.

Years are neither, as they can be both positive or negative, unlike
any other datetime component.  There I<is> a year 0.

There is no C<quarter_0()> method.

=head2 Error Handling

Some errors may cause this module to die with an error string.  This
can only happen when calling constructor methods or methods that
change the object, such as C<set()>.  Methods that retrieve
information about the object, such as C<strftime()>, will never die.

=head2 Languages

Some methods are localizable.  This is done by setting the language
when constructing a DateTime object.  There is also a
C<DefaultLanguage()> class method which may be used to set the default
language for all DateTime objects created.  If this is not set, then
"English" is used.

Additional language subclasses are welcome.  See the Perl DateTime
project page at http://perl-date-time.sf.net/ for more details.

Some languages may return data as Unicode.  When using Perl 5.6.0 or
greater, this will be a native Perl Unicode string.  When using older
Perls, this will be a sequence of bytes representing the Unicode
character.

=head2 Methods

=head3 Constructors

All constructors can die when invalid parameters are given.

=over 4

=item * new( ... )

This class method accepts parameters for each date and time component:
"year", "month", "day", "hour", "minute", "second", "nanosecond".
Additionally, it accepts "fractional_second", "language",
and "time_zone" parameters.

  my $dt = DateTime->new( day    => 25,
                          month  => 10,
                          year   => 1066,
                          hour   => 7,
                          minute => 15,
                          second => 47,
                          nanosecond => 500000000,
                          time_zone  => 'America/Chicago',
                        );

If a "fractional_second" parameter is given, then the "nanosecond"
parameter will be ignored.

The behavior of this module when given parameters outside proper
boundaries (like a minute parameter of 72) is not defined, though
future versions may die.

Invalid parameter types (like an array reference) will cause the
constructor to die.

All of the parameters are optional except for "year".  The "month" and
"day" parameters both default to 1, while the "hour", "minute", and
"second", and "nanosecond" parameters all default to 0.

The language parameter should be a string matching one of the valid
languages.  See the L<DateTime::Language|DateTime::Language>
documentation for details.

The time_zone parameter can be either a scalar or a
C<DateTime::TimeZone> object.  A string will simply be passed to the
C<< DateTime::TimeZone->new >> method as its "name" parameter.  This
string may be an Olson DB time zone name ("America/Chicago"), an
offset string ("+0630"), or the words "floating" or "local".  See the
C<DateTime::TimeZone> documentation for more details.

The default time zone is "floating".

=head4 Ambiguous Local Times

Because of Daylight Saving Time, it is possible to specify a local
time that is ambiguous.  For example, in the US in 2003, the
transition from to saving to standard time occurs on October 26, at
02:00:00 local time.  The local clock changes from 01:59:59 (saving
time) to 01:00:00 (standard time).  This means that the hour from
01:00:00 through 01:59:59 actually occurs twice, though the UTC time
continues to move forward.

If you specify an ambiguous time, then the latest UTC time is always
used, in effect always choosing saving time.  In this case, you can
simply subtract an hour to the object in order to move to standard time,
for example:

  # This object represent 01:30:00 standard time
  my $dt = DateTime->new( year   => 2003,
                          month  => 10,
                          day    => 26,
                          hour   => 1,
                          minute => 30,
                          second => 0,
                          time_zone => 'America/Chicago',
                        );

  print $dt->hms;  # prints 01:30:00

  # Now the object represent 01:30:00 saving time
  $dt->subtract( hours => 1 );

  print $dt->hms;  # still prints 01:30:00

Alternately, you could create the object with the UTC time zone, and
then call the C<set_time_zone()> method to change the time zone.  This
would allow you to unambiguously specify the datetime.

=head4 Invalid Local Times

Another problem introduced by Daylight Saving Time is that certain
local times just do not exist.  For example, in the US in 2003, the
transition from standard to saving time occurred on April 6, at the
change to 2:00:00 local time.  The local clock changes from 01:59:59
(standard time) to 03:00:00 (saving time).  This means that there is
no 02:00:00 through 02:59:59 on April 6!

Attempting to create an invalid time currently causes a fatal error.
This may change in future version of this module.

=item * from_epoch( epoch => $epoch, ... )

This class method can be used to construct a new DateTime object from
an epoch time instead of components.  Just as with the C<new()>
method, it accepts "time_zone" and "language" parameters.

=item * now( ... )

This class method is equivalent to calling C<from_epoch()> with the
value returned from Perl's C<time()> function.

=item * today( ... )

This class method is equivalent to:

  DateTime->now->truncate( to => 'day' );

=item * from_object( object => $object, ... )

This class method can be used to construct a new DateTime object from
any object that implements the C<utc_rd_values()> method.  All
C<DateTime::Calendar> modules must implement this method in order to
provide cross-calendar compatibility.  This method accepts a
"language" parameter

If the object passed to this method has a C<time_zone()> method, that
is used to set the time zone of the newly created C<DateTime.pm>
object.  Otherwise UTC is used.

=item * last_day_of_month( ... )

This constructor takes the same arguments as can be given to the
C<new()> method, except for "day".  Additionally, both "year" and
"month" are required.

=item * clone

This object method returns a replica of the given object.

=back

=head3 "Get" Methods

This class has many methods for retrieving information about an
object.

=over 4

=item * year

Returns the year.

=item * ce_year

Returns the year according to the BCE/CE numbering system.  The year
before year 1 in this system is year -1, aka "1 BCE".

=item * month

Returns the month of the year, from 1..12.

=item * month_name

Returns the name of the current month.  See the
L<LANGUAGES|/LANGUAGES> section for more details.

=item * month_abbr

Returns the abbreviated name of the current month.  See the
L<LANGUAGES|/LANGUAGES> section for more details.

=item * day_of_month, day, mday

Returns the day of the month, from 1..31.

=item * day_of_week, wday, dow

Returns the day of the week as a number, from 1..7, with 1 being
Monday and 7 being Sunday.

=item * day_name

Returns the name of the current day of the week.  See the
L<LANGUAGES|/LANGUAGES> section for more details.

=item * day_abbr

Returns the abbreviated name of the current day of the week.  See the
L<LANGUAGES|/LANGUAGES> section for more details.

=item * day_of_year, doy

Returns the day of the year.

=item * day_of_quarter, doq

Returns the day of the quarter.

=item * quarter

Returns the quarter of the year, from 1..4.

=item * ymd( $optional_separator ), date

=item * mdy( $optional_separator )

=item * dmy( $optional_separator )

Each method returns the year, month, and day, in the order indicated
by the method name.  Years are zero-padded to four digits.  Months and
days are 0-padded to two digits.

By default, the values are separated by a dash (-), but this can be
overridden by passing a value to the method.

=item * hour

Returns the hour of the day, from 0..23.

=item * minute, min

Returns the minute of the hour, from 0..59.

=item * second, sec

Returns the second, from 0..61.  The values 60 and 61 are used for
leap seconds.

=item * fractional_second

Returns the second, as a real number from 0.0 until 61.999999999

The values 60 and 61 are used for leap seconds.

=item * millisecond

Returns the fractional part of the second as milliseconds (1E-3 seconds).

Half a second is 500 milliseconds.

=item * microsecond

Returns the fractional part of the second as microseconds (1E-6
seconds).  This value will be rounded to an integer.

Half a second is 500_000 microseconds.  This value will be rounded to
an integer.

=item * nanosecond

Returns the fractional part of the second as nanoseconds (1E-9 seconds).

Half a second is 500_000_000 nanoseconds.

=item * hms( $optional_separator ), time

Returns the hour, minute, and second, all zero-padded to two digits.
If no separator is specified, a colon (:) is used by default.

=item * datetime, iso8601

This method is equivalent to:

  $dt->ymd('-') . 'T' . $dt->hms(':')

=item * is_leap_year

This method returns a true or false indicating whether or not the
datetime object is in a leap year.

=item * week

 ($week_year, $week_number) = $dt->week

Returns information about the calendar week which contains this
datetime object. The values returned by this method are also available
separately through the week_year and week_number methods.

The first week of the year is defined by ISO as the one which contains
the fourth day of January, which is equivalent to saying that it's the
first week to overlap the new year by at least four days.

Typically the week year will be the same as the year that the object
is in, but dates at the very begining of a calendar year often end up
in the last week of the prior year, and similarly, the final few days
of the year may be placed in the first week of the next year.

=item * week_year

Returns the year of the week.

=item * week_number

Returns the week of the year, from 1..53.

=item * jd, mjd

These return the Julian Day and Modified Julian Day, respectively.
The value returned is a floating point number.  The fractional portion
of the number represents the time portion of the datetime.

=item * time_zone

This returns the C<DateTime::TimeZone> object for the datetime object.

=item * offset

This returns the offset, in seconds, of the datetime object according
to the time zone.

=item * is_dst

Returns a boolean indicating whether or not the datetime object is
currently in Daylight Saving Time or not.

=item * time_zone_short_name

This method returns the time zone abbreviation for the current time
zone, such as "PST" or "GMT".  These names are B<not> definitive, and
should not be used in any application intended for general use by
users around the world.

=item * strftime( $format, ... )

This method implements functionality similar to the C<strftime()>
method in C.  However, if given multiple format strings, then it will
return multiple scalars, one for each format string.

See the L<strftime Specifiers|/strftime Specifiers> section for a list
of all possible format specifiers.

=item * epoch

Return the UTC epoch value for the datetime object.  Internally, this
is implemented using C<Time::Local>, which uses the Unix epoch even on
machines with a different epoch (such as MacOS).  Datetimes before the
start of the epoch will be returned as a negative number.

Since the epoch does not account for leap seconds, the epoch time for
1971-12-31T23:59:60 (UTC) is exactly the same as that for
1972-01-01T00:00:00.

Since epoch times cannot represent many dates on most platforms, this
method may simply return undef in some cases.

Using your system's epoch time may be error-prone, since epoch times
have such a limited range on 32-bit machines.  Additionally, the fact
that different operating systems have different epoch beginnings is
another source of possible bugs.

=item * is_finite, is_infinite

These methods allow you to distinguish normal datetime objects from
infinite ones.  Infinite datetime objects are documented in
L<DateTime::Infinite|DateTime::Infinite>.

=item * utc_rd_values

Returns the current UTC Rata Die days and seconds as a two element
list.  This exists primarily to allow other calendar modules to create
objects based on the values provided by this object.

=item * utc_rd_as_seconds

Returns the current UTC Rata Die days and seconds purely as seconds.
This number ignores any fractional seconds stored in the object,
as well as leap seconds.

=item * local_rd_as_seconds

Returns the current local Rata Die days and seconds purely as seconds.
This number ignores any fractional seconds stored in the object,
as well as leap seconds.

=back

=head3 "Set" Methods

The remaining methods provided by C<DateTime.pm>, except where otherwise
specified, return the object itself, thus making method chaining
possible. For example:

  my $dt = DateTime->now->set_time_zone( 'Australia/Sydney' );

  my $first = DateTime
		->last_day_of_month( year => 2003, month => 3 )
		->add( days => 1 )
                ->subtract( seconds => 1 );

=over 4

=item * set( .. )

This method can be used to change the local components of a date time,
or its language.  This method accepts any parameter allowed by the
C<new()> method except for "time_zone".  Time zones may be set using
the C<set_time_zone()> method.

=item * truncate( to => ... )

This method allows you to reset some of the local time components in
the object to their "zero" values.  The "to" parameter is used to
specify which values to truncate, and it may be one of "year",
"month", "day", "hour", "minute", or "second".  For example, if
"month" is specified, then the local day becomes 1, and the hour,
minute, and second all become 0.

=item * set_time_zone( $tz )

This method accepts either a time zone object or a string that can be
passed as the "name" parameter to C<< DateTime::TimeZone->new() >>.
If the new time zone's offset is different from the old time zone,
then the I<local> time is adjusted accordingly.

For example:

  my $dt = DateTime->new( year => 2000, month => 5, day => 10,
                          hour => 15, minute => 15,
                          time_zone => '-0600', );

  print $dt->hour; # prints 15

  $dt->set_time_zone( '-0400' );

  print $dt->hour; # prints 17

If the old time zone was a floating time zone, then no adjustments to
the local time are made, except to account for leap seconds.  If the
new time zone is floating, then the I<UTC> time is adjusted in order
to leave the local time untouched.

Fans of Tsai Ming-Liang's films will be happy to know that this does
work:

  my $dt = DateTime::TimeZone->new( ..., time_zone => 'Asia/Taipei' );

  $dt->set_time_zone( 'Europe/Paris' );

Yes, now we can know "ni3 na1 bian1 ji3dian2?"

=item * add_duration( $duration_object )

This method adds a C<DateTime::Duration> to the current datetime.  See
the L<DateTime::Duration|DateTime::Duration> docs for more details.

=item * add( DateTime::Duration->new parameters )

This method is syntactic sugar around the C<add_duration()> method.  It
simply creates a new C<DateTime::Duration> object using the parameters
given, and then calls the C<add_duration()> method.

=item * subtract_duration( $duration_object )

When given a C<DateTime::Duration> object, this method simply calls
C<invert()> on that object and passes that new duration to the
C<add_duration> method.

=item * subtract( DateTime::Duration->new parameters )

Like C<add()>, this is syntactic sugar for the C<subtract_duration()>
method.

=item * subtract_datetime( $datetime )

This method returns a new C<DateTime::Duration> object representing
the difference between the two dates.  The duration object will only
have deltas for day and seconds.

=back

=head3 Class Methods

=over 4

=item * DefaultLanguage( $language )

This can be used to specify the default language to be used when
creating DateTime objects.  If unset, then "English" is used.

=item * compare

=item * compare_ignore_floating

  $cmp = DateTime->compare( $dt1, $dt2 );

  $cmp = DateTime->compare_ignore_floating( $dt1, $dt2 );

Compare two DateTime objects.  The semantics are compatible with
Perl's C<sort()> function; it returns -1 if $a < $b, 0 if $a == $b, 1
if $a > $b.

If one of the two DateTime objects has a floating time zone, it will
first be converted to the time zone of the other object.  This is what
you want most of the time, but it can lead to inconsistent results
when you compare a number of DateTime objects, some of which are
floating, and some of which are in other time zones.

If you want to have consistent results (because you want to sort a
number of objects, for example), you can use the
C<compare_ignore_floating()> method:

  @dates = sort { DateTime->compare_ignore_floating($a, $b) } @dates;

In this case, objects with a floating time zone will be sorted as if
they were UTC times.

Since DateTime objects overload comparison operators, this:

  @dates = sort @dates;

is equivalent to this:

  @dates = sort { DateTime->compare($a, $b) } @dates;

=back

=head2 How Date Math is Done

It's important to have some understanding of how date math is
implemented in order to effectively use this module and
C<DateTime::Duration>.

The parts of a duration can be broken down into four parts.  These are
months, days, minutes, and seconds.  Adding one month to a date is
different than adding 4 weeks or 28, 29, 30, or 31 days.  Similarly,
due to DST and leap seconds, adding a day can be different than adding
86,400 seconds, and adding a minute is not exactly the same as 60
seconds.

C<DateTime.pm> always adds (or subtracts) days, then months, minutes,
and then seconds.  If there are any boundary overflows, these are
normalized at each step.

This means that adding one month and one day to February 28, 2003 will
produce the date April 1, 2003, not March 29, 2003.

=head3 Leap Seconds and Date Math

The presence of leap seconds can cause some strange anomalies in date
math.  For example, the following is a legal datetime:

  my $dt = DateTime->new( year => 1971, month => 12, day => 31,
                          hour => 23, minute => 59, second => 60,
                          time_zone => 'UTC' );

If we do the following:

 $dt->add( months => 1 );

Then the datetime is now "1972-02-01 00:00:00", because there is no
23:59:60 on 1972-01-31.

Leap seconds also force us to distinguish between minutes and seconds
during date math.  Given the following datetime:

  my $dt = DateTime->new( year => 1971, month => 12, day => 31,
                          hour => 23, minute => 59, second => 30,
                          time_zone => 'UTC' );

we will get different results when adding 1 minute than we get if we
add 60 seconds.  This is because in this case, the last minute of the
day, beginning at 23:59:00, actually contains 61 seconds.

=head3 Local vs. UTC and 24 hours vs. 1 day

When doing date math, you are changing the I<local> datetime.  This is
generally the same as changing the UTC datetime, except when a change
crosses a daylight saving boundary.  The net effect of this is that 24
hours is not always the same as 1 day.

Specifically, if you do this:

  my $dt = DateTime->new( year => 2003, month => 4, day => 5,
                          hour => 2,
                          time_zone => 'America/Chicago',
                        );
  $dt->add( days => 1 );

then you will produce an I<invalid> local time, and therefore an
exception will be thrown.

However, this works:

  my $dt = DateTime->new( year => 2003, month => 4, day => 5,
                          hour => 2,
                          time_zone => 'America/Chicago',
                        );
  $dt->add( hours => 24 );

and produces a datetime with the local time of "03:00".

Another way of thinking of this is that when doing date math, each of
the seconds, minutes, days, and months components is added separately
to the local time.

So when we add 1 day to "2003-02-22 12:00:00" we are incrementing day
component, 22, by one in order to produce 23.  If we add 24 hours,
however, we're adding "24 * 60" minutes to the time component, and
then normalizing the result (because there is no "36:00:00").

If all this makes your head hurt, there is a simple alternative.  Just
convert your datetime object to the "UTC" time zone before doing date
math on it, and switch it back to the local time zone afterwards.
This avoids the possibility of having date math throw an exception,
and makes sure that 1 day equals 24 hours.  Of course, this may not
always be desirable, so caveat user!

=head2 Overloading

This module explicitly overloads the addition (+), subtraction (-),
string and numeric comparison operators.  This means that the
following all do sensible things:

  my $new_dt = $dt + $duration_obj;

  my $new_dt = $dt - $duration_obj;

  my $duration_obj = $dt - $new_dt;

  foreach my $dt ( sort @dts ) { ... }

Additionally, the fallback parameter is set to true, so other
derivable operators (+=, -=, etc.) will work properly.  Do not expect
increment (++) or decrement (--) to do anything useful.

=head2 strftime Specifiers

The following specifiers are allowed in the format string given to the
C<strftime()> method:

=over 4

=item * %a

The abbreviated weekday name.

=item * %A

The full weekday name.

=item * %b

The abbreviated month name.

=item * %B

The full month name.

=item * %C

The century number (year/100) as a 2-digit integer.

=item * %d

The day of the month as a decimal number (range 01 to 31).

=item * %D

Equivalent to %m/%d/%y.  This is not a good standard format if you
have want both Americans and Europeans to understand the date!

=item * %e

Like %d, the day of the month as a decimal number, but a leading zero
is replaced by a space.

=item * %F

Equivalent to %Y-%m-%d (the ISO 8601 date format)

=item * %G

The ISO 8601 year with century as a decimal number.  The 4-digit year
corresponding to the ISO week number (see %V).  This has the same
format and value as %y, except that if the ISO week number belongs to
the previous or next year, that year is used instead. (TZ)

=item * %g

Like %G, but without century, i.e., with a 2-digit year (00-99).

=item * %h

Equivalent to %b.

=item * %H

The hour as a decimal number using a 24-hour clock (range 00 to 23).

=item * %I

The hour as a decimal number using a 12-hour clock (range 01 to 12).

=item * %j

The day of the year as a decimal number (range 001 to 366).

=item * %k

The hour (24-hour clock) as a decimal number (range 0 to 23); single
digits are preceded by a blank. (See also %H.)

=item * %l

The hour (12-hour clock) as a decimal number (range 1 to 12); single
digits are preceded by a blank. (See also %I.)

=item * %m

The month as a decimal number (range 01 to 12).

=item * %M

The minute as a decimal number (range 00 to 59).

=item * %n

A newline character.

=item * %N

The fractional seconds digits. Default is 9 digits (nanoseconds).

  %3N   milliseconds (3 digits)
  %6N   microseconds (6 digits)
  %9N   nanoseconds  (9 digits)

=item * %p

Either `AM' or `PM' according to the given time value, or the
corresponding strings for the current locale.  Noon is treated as `pm'
and midnight as `am'.

=item * %P

Like %p but in lowercase: `am' or `pm' or a corresponding string for
the current locale.

=item * %r

The time in a.m.  or p.m. notation.  In the POSIX locale this is
equivalent to `%I:%M:%S %p'.

=item * %R

The time in 24-hour notation (%H:%M). (SU) For a version including the
seconds, see %T below.

=item * %s

The number of seconds since the epoch.

=item * %S

The second as a decimal number (range 00 to 61).

=item * %t

A tab character.

=item * %T

The time in 24-hour notation (%H:%M:%S).

=item * %u

The day of the week as a decimal, range 1 to 7, Monday being 1.  See
also %w.

=item * %U

The week number of the current year as a decimal number, range 00 to
53, starting with the first Sunday as the first day of week 01. See
also %V and %W.

=item * %V

The ISO 8601:1988 week number of the current year as a decimal number,
range 01 to 53, where week 1 is the first week that has at least 4
days in the current year, and with Monday as the first day of the
week. See also %U and %W.

=item * %w

The day of the week as a decimal, range 0 to 6, Sunday being 0.  See
also %u.

=item * %W

The week number of the current year as a decimal number, range 00 to
53, starting with the first Monday as the first day of week 01.

=item * %y

The year as a decimal number without a century (range 00 to 99).

=item * %Y

The year as a decimal number including the century.

=item * %z

The time-zone as hour offset from UTC.  Required to emit
RFC822-conformant dates (using "%a, %d %b %Y %H:%M:%S %z").

=item * %Z

The time zone or name or abbreviation.

=item * %%

A literal `%' character.

=back

=head1 SUPPORT

Support for this module is provided via the datetime@perl.org email
list.  See http://lists.perl.org/ for more details.

=head1 AUTHOR

Dave Rolsky <autarch@urth.org>

However, please see the CREDITS file for more details on who I really
stole all the code from.

=head1 COPYRIGHT

Copyright (c) 2003 David Rolsky.  All rights reserved.  This program
is free software; you can redistribute it and/or modify it under the
same terms as Perl itself.

Portions of the code in this distribution are derived from other
works.  Please see the CREDITS file for more details.

The full text of the license can be found in the LICENSE file included
with this module.

=head1 SEE ALSO

datetime@perl.org mailing list

http://datetime.perl.org/

=cut
