package Archive::Har::Entry::Request::PostData::Params;

use warnings;
use strict;

our $VERSION = 0.07;

sub new {
	my ($class, $params) = @_;
	my $self = {};
	bless $self, $class;
	if (defined $params) {
		$self->name($params->{name});
		$self->value($params->{value});
		if (defined $params->{fileName}) {
			$self->file_name($params->{fileName});
		}
		if (defined $params->{contentType}) {
			$self->content_type($params->{contentType});
		}
		if (defined $params->{comment}) {
			$self->comment($params->{comment});
		}
		foreach my $key (sort { $a cmp $b } keys %{$params}) {
			if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
				if (defined $params->{$key}) {
					$self->$key($params->{$key});
				}
			}
		}
	}
	return $self;
}

sub name {
	my ($self, $new) = @_;
	my $old = $self->{name};
	if (@_ > 1) {
		$self->{name} = $new;
	}
	return $old;
}

sub value {
	my ($self, $new) = @_;
	my $old = $self->{value};
	if (@_ > 1) {
		$self->{value} = $new;
	}
	return $old;
}

sub file_name {
	my ($self, $new) = @_;
	my $old = $self->{fileName};
	if (@_ > 1) {
		$self->{fileName} = $new;
	}
	return $old;
}

sub content_type {
	my ($self, $new) = @_;
	my $old = $self->{contentType};
	if (@_ > 1) {
		$self->{contentType} = $new;
	}
	return $old;
}

sub comment {
	my ($self, $new) = @_;
	my $old = $self->{comment};
	if (@_ > 1) {
		$self->{comment} = $new;
	}
	return $old;
}

sub AUTOLOAD {
	my ($self, $new) = @_;
	my $type = ref $self or Carp::croak("$self is not an object");

	my $name = $Archive::Har::Entry::Request::PostData::Params::AUTOLOAD;
	$name =~ s/.*://smx;   # strip fully-qualified portion

	if ($name =~ /^_[[:alnum:]]+$/smx) { # private fields
		my $old = $self->{$name};
		if (@_ > 1) {
			$self->{$name} = $new;
		}
		return $self->{$name};
	} else {
		Carp::croak("$name is not specified in the HAR 1.2 spec and does not start with an underscore");
	}
	return;
}

sub TO_JSON {
	my ($self) = @_;
	my $json = { };
	$json->{name} = $self->name();
	$json->{value} = $self->value();
	if (defined $self->comment()) {
		$json->{comment} = $self->comment();
	}
	foreach my $key (sort { $a cmp $b } keys %{$self}) {
		next if (!defined $self->{$key});
		if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
			$json->{$key} = $self->{$key};
		}
	}
	return $json;
}

1;
__END__

=head1 NAME

Archive::Har::Entry::Request::PostData::Params - Represents a single name/value pair from the query string for a request inside the HTTP Archive

=head1 VERSION

Version 0.07

=head1 SYNOPSIS

    use Archive::Har();

    my $http_archive_string = '"log": { "version": "1.1", .... ';
    my $har = Archive::Har->new();
    $har->string($http_archive_string);
    foreach my $entry ($har->entries()) {
        my $request = $entry->request();
	my $postData = $request->postData();
	if (defined $postData) {
		foreach my $param ($postData->params()) {
			print "Name: " . $element->name() . "\n";
			print "Value: " . $element->value() . "\n";
			print "File Name: " . $element->file_name() . "\n";
			print "Content Type: " . $element->content_type() . "\n";
			print "Comment: " . $element->comment() . "\n";
		}
	}
    }

=head1 SUBROUTINES/METHODS

=head2 name

returns the name of the parameter in the posted data

=head2 value

returns the value of the parameter in the posted data or content of the posted file

=head2 file_name

returns the name of the posted file

=head2 content_type

returns the content type of the posted file

=head2 comment

returns the comment about the parameters

