#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebDyne::Install.
#
#  WebDyne::Install is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: WebDyne_Install.pm,v 1.15 2006/05/17 08:41:57 aspeer Exp $



#  WebDyne Apache Install module
#
package WebDyne::Install;


#  Compiler Pragma
#
sub BEGIN	{ $^W=0 };
use strict	qw(vars);
use vars	qw($VERSION $REVISION);
use warnings;
no  warnings	qw(uninitialized);


#  Webmod Modules
#
use WebMod::Base qw(:all);
use WebDyne::Constant;
use WebDyne::Install::Constant;


#  External Modules
#
use File::Path;
use Text::Template;
use IO::File;
use Config;


#  Version information in a formate suitable for CPAN etc. Must be
#  all on one line
#
$VERSION = eval { require WebDyne::Install::VERSION; do $INC{'WebDyne/Install/VERSION.pm'}};


#  Revision information, auto maintained by CVS
#
$REVISION=(qw$Revision: 1.15 $)[1];


#  Debug
#
debug("%s loaded, version $VERSION, revision $REVISION", __PACKAGE__);


#  Quit if in RPM build, don't want to impact real files
#
sub BEGIN { $ENV{'RPM_BUILD_ROOT'} && exit 0 }


#  Var to hold class wide data
#
my %Class;


#  Init done.
#
1;


#------------------------------------------------------------------------------


sub import {


    #  Set silent, verbose or debug flags as indicated when class loaded
    #
    foreach my $import qw(silent debug) {
    	$Class{$import}++ if grep { $_ eq $import } @_[1..$#_];
    }
    $Class{'debug'} && ($ENV{'PERL_DEBUG_ALL'}=1);
    shift()->SUPER::import(@_);

}


sub _ {


    #  Hairy stuff, only print debug and verbose message, strip :D and :V of end of
    #  each respectively
    #
    return undef if $Class{'silent'};
    @_ || return $/;
    sprintf(join('[install] - ',  undef, ucfirst(shift())).$/, @_);


}


#  Uninstall
#
sub uninstall {

    #  Set flag and call install routine
    #
    $Class{'_uninstall'}++;
    shift()->install(@_);

}


#  Install the Apache conf params needed to run the system
#
sub install {


    #  Get class, other paths
    #
    my ($class, $prefix, $installbin)=@_;

    print _;
    print _"installation source directory '$prefix'." if $prefix;
    debug("class $class, prefix $prefix, installbin $installbin");


    #  Get package file name so we can look up in inc
    #
    (my $class_fn=$class.q(.pm))=~s/::/\//g;


    #  Get from INC
    #
    $class_fn=$INC{$class_fn} ||
	return err("unable to find location for $class in \%INC");


    #  Split
    #
    my $class_dn=(File::Spec->splitpath($class_fn))[1];
    my @class=split(/\:\:/, $class);


    #  Get config_hr
    #
    my $config_hr="${class}::Constant"->hashref();
    debug('config_hr %s', Dumper($config_hr));


    #  Cannot do anything without Apache binary
    #
    return err("unable to find apache binary") 
      unless $config_hr->{'HTTPD_BIN'};


    #  Update with prefix and installbin dirs
    #
    @{$config_hr}{qw(DIR_PREFIX DIR_INSTALLBIN)}=(
    	$prefix, $installbin);


    #  Get template file name
    #
    my $template_fn=File::Spec->catfile(
	$class_dn, $class[-1], $FILE_WEBDYNE_CONF_TEMPLATE);


    #  Open it as a template
    #
    my $template_or=Text::Template->new(

        type    =>  'FILE',
        source  =>  $template_fn,

       ) || return err("unable to open template $template_fn, $!");


    #  Fill in with out self ref as a hash
    #
    my $webdyne_conf=$template_or->fill_in(

        HASH	    =>  { self => \$class, config_hr=>\$config_hr, MP2=>\$MP2 },
	DELIMITERS  =>  [ '<!--', '-->' ], 

       ) || return err("unable to fill in template $template_fn, $Text::Template::ERROR");



    #  Get apache config dir
    #
    my $apache_conf_dn=$config_hr->{'DIR_APACHE_CONF'} ||
	return err('unable to determine Apache config directory');


    #  Work out config file name
    #
    my $webdyne_conf_fn=File::Spec->catfile(
	$apache_conf_dn, $FILE_WEBDYNE_CONF);


    #  Open, write webdyne config file unless in uninstall
    #
    unless ($Class{'_uninstall'}) {
        print _"writing Apache config file '$webdyne_conf_fn'.";
	my $webdyne_conf_fh=IO::File->new($webdyne_conf_fn, O_CREAT|O_WRONLY|O_TRUNC) ||
	    return err("unable to open file $webdyne_conf_fn, $!");
	print $webdyne_conf_fh $webdyne_conf;
	$webdyne_conf_fh->close();
    }


    #  Only modify config file if no conf.d dir, denoted by var below
    #
    unless ($config_hr->{'HTTPD_SERVER_CONFIG_SKIP'}) {


	#  Get Apache config file, append root if not absolute
	#
	my $apache_conf_fn=$config_hr->{'HTTPD_SERVER_CONFIG_FILE'} ||
	    return err("unable to determine main server config file");
	($apache_conf_fn=~/^\//) || (
	    $apache_conf_fn=File::Spec->catfile(
		$config_hr->{'HTTPD_ROOT'},$apache_conf_fn));
	my $apache_conf_fh=IO::File->new($apache_conf_fn, O_RDONLY) ||
	    return err("unable to open file $apache_conf_fn, $!");
	print _"Apache config file '$apache_conf_fn'";


	#  Setup delims looking for
	#
	my ($delim, @delim)=$config_hr->{'FILE_APACHE_CONF_DELIM'};


	#  Turn into array, search for delims
	#
	my ($index, @apache_conf);
	while (my $line=<$apache_conf_fh>) {
	    push @apache_conf, $line;
	    push(@delim, $index) if $line=~/\Q$delim\E/;
	    $index++;
	}


	#  Check found right number of delims
	#
	if (@delim!=2 and @delim!=0) {

	    return err(
		"found %s '$delim' delimiter%s in $apache_conf_fn at line%s %s, expected exactly 2 delimiters",
		scalar @delim,
		($#delim ? 's' : '') x 2,
		join(',', @delim)
	       );

	}


	#  Check if delim found, if not, make last line
	#
	unless (@delim) { @delim=($index, $index-1) }
	$delim[1]++;


	#  Close
	#
	$apache_conf_fh->close();


	#  Splice the lines between the delimiters out
	#
	splice(@apache_conf, $delim[0], $delim[1]-$delim[0]);


	#  Clean up end of conf file, remove excess CR's
	#
	my $lineno=$delim[0]-1;
	for (undef; $lineno > 0; $lineno--) {


	    #  We are going backwards through the file, as soon as we
	    #  see something we quit
	    #
	    my $line=$apache_conf[$lineno];
	    chomp($line);


	    #  Empty
	    #
	    if ($line=~/^\s*$/) {


		#  Yes, delete and continue
		#
		splice(@apache_conf, $lineno, 1);

	    }
	    else {


		#  No, quit after rewinding lineno to last val
		#
		$lineno++;
		last;

	    }
	}


	#  Only splice back in if not uninstalling
	#
	unless ($Class{'_uninstall'}) {


	    #  Get template we want to include in the config file
	    #
	    $template_fn=File::Spec->catfile(
		$class_dn, $class[-1], $FILE_APACHE_CONF_TEMPLATE);


	    #  Open it as a template
	    #
	    $template_or=Text::Template->new(

		type    =>  'FILE',
		source  =>  $template_fn,

	       ) || return err("unable to open template $template_fn, $!");


	    #  Fill in with out self ref as a hash
	    #
	    my $apache_conf=$template_or->fill_in(

		HASH	    =>  { self => \$class, config_hr=>\$config_hr },
		DELIMITERS  => [ '<!--', '-->' ], 

	       ) || return err("unable to fill in template $template_fn, $Text::Template::ERROR");


	    #  Splice in now, but write out at end of block
	    #
	    splice(@apache_conf, $lineno, undef, $apache_conf);


	}
	else {


	    #  In uninstall - get rid of conf file
	    #
	    foreach my $fn (grep {(-f $_)} $webdyne_conf_fn) {
		unlink($fn) && print _"unlink config file $fn\n";
	    }


	}


	#  Re-open httpd.conf for write out, unless uninstall and delims not found, which
	#  means nothing was changed.
	#
	unless ($Class{'_uninstall'} && ($delim[0] == $delim[1])) {
	    $apache_conf_fh=IO::File->new($apache_conf_fn, O_TRUNC|O_WRONLY) ||
		return err("unable to open file $apache_conf_fn, $!");
	    print $apache_conf_fh join('', @apache_conf);
	    $apache_conf_fh->close();
	    print _"Apache config file '$apache_conf_fn' updated.";
	}

    }
    else {

	#  No need to update Apache config file - using conf.d dir

    }



    #  All done now if uninstalling
    #
    $Class{'_uninstall'} && do {
    	print _"uninstall completed";
	return 1;
    };


    #  Create the cache and session/state dirs, Update cache dir only now
    #
    foreach my $dest_dn (grep {!-d $_} $WEBDYNE_CACHE_DN) {
    #foreach my $dest_dn ($WEBDYNE_CACHE_DN) {


	#  Make
	#
	print _"creating directory '$dest_dn'.";
	File::Path::mkpath($dest_dn, 1, 0755) || do {
	    return err("unable to create dir $dest_dn") unless (-d $dest_dn) };


	#  Chown
	#
	#print _"updating directory '$dest_dn' ownership";
	chown($APACHE_UID, $APACHE_GID, $dest_dn) ||
	    return err("unable to chown $dest_dn to $APACHE_UID.$APACHE_GID");


    }


    #  Add prefix to perl5lib store
    #
    print _"updating perl5lib config.";
    &perl5lib::add($prefix);


    # Done
    #
    print _"install completed.";
    print $/;
    return 1;

}
