use Data::Dumper;
use lib 'lib';
use Test::More tests => 23;
use Parse::BBCode;

my %tag_def_html = (

    code2 => {
#        class => 'block',
        parse => 0,
        code => sub {
            my $c = $_[2];
            $c = Parse::BBCode::escape_html($$c);
            "<code>$c</code>"
        },
    },
    perlmonks => '<a href="http://www.perlmonks.org/?node=%{uri|html}a" rel="nofollow">%{parse}s</a>',
);

my $bbc2html = Parse::BBCode->new({                                                              
        tags => {
            Parse::BBCode::HTML->defaults,
            %tag_def_html,
        },
    }
);

my @tests = (
    [ q#[B]bold? [test#,
        q#[B]bold? [test# ],
    [ q#[i=23]italic [b]bold italic <html>[/b][/i]# . $/,
        q#<i>italic <b>bold italic &lt;html&gt;</b></i><br># ],
    [ q#[U][noparse]<html>[u][c][/noparse][/u]# . $/,
        q#<u>&lt;html&gt;[u][c]</u><br># ],
    [ q#[img=foo.jpg]desc <html>[/img]#,
        q#<img src="foo.jpg" alt="[desc &lt;html&gt;]" title="desc &lt;html&gt;"># ],
    [ q#[url=javascript:alert(123)]foo <html>[i]italic[/i][/url]#,
        q#<a href="" rel="nofollow">foo &lt;html&gt;<i>italic</i></a># ],
    [ q#[url=http://foo]foo <html>[i]italic[/i][/url]#,
        q#<a href="http://foo" rel="nofollow">foo &lt;html&gt;<i>italic</i></a># ],
    [ q#[email=no"mail]mail [i]me[/i][/email]#,
        q#<a href="mailto:">mail <i>me</i></a># ],
    [ q#[email="test <foo@example.org>"]mail [i]me[/i][/email]#,
        q#<a href="mailto:foo@example.org">mail <i>me</i></a># ],
    [ q#[email]test <foo@example.org>[/email]#,
        q#<a href="mailto:foo@example.org">test &lt;foo@example.org&gt;</a># ],
    [ q#[size=7]big[/size]#,
        q#<span style="font-size: 7">big</span># ],
    [ q#[size=huge!]big[/size]#,
        q#<span style="font-size: 0">big</span># ],
    [ q{[color=#0000ff]blue[/color]},
        q{<span style="color: #0000ff">blue</span>} ],
    [ q{[color="no color!"]blue[/color]},
        q{<span style="color: inherit">blue</span>} ],
    [ q#[list=1][*]first[*]second[*]third[/list]#,
        q#<ul><li>first</li><li>second</li><li>third</li></ul># ],
    [ q#[quote=who]cite[/quote]#,
        q#<div class="bbcode_quote_header">who:<div class="bbcode_quote_body">cite</div</div># ],
    [ q#[code]use strict;[/code]#,
        q#<div class="bbcode_code_header">:<div class="bbcode_cote_body">use strict;</div></div># ],
    [ q#[perlmonks=123]foo <html>[i]italic[/i][/perlmonks]# . $/,
        q#<a href="http://www.perlmonks.org/?node=123" rel="nofollow">foo &lt;html&gt;<i>italic</i></a><br># ],
    [ q#[noparse]foo[b][/noparse]#,
        q#foo[b]# ],
    [ q#[code]foo[code]bar<html>[/code][/code]#,
        q#<div class="bbcode_code_header">:<div class="bbcode_cote_body">foo[code]bar&lt;html&gt;</div></div>[/code]# ],
    [ q#[i]italic [b]bold italic <html>[/i][/b]#,
        q#<i>italic [b]bold italic &lt;html&gt;</i>[/b]# ],
    [ q#[i]italic [b]bold italic <html>[/i][/b]#,
        q#[i]italic <b>bold italic &lt;html&gt;[/i]</b>#, 'i' ],
);
for my $test (@tests) {
    my ($text, $exp, $forbid) = @$test;
    if ($forbid) {
        $bbc2html->forbid($forbid);
    }
    my $parsed = $bbc2html->render($text);
    #warn __PACKAGE__.':'.__LINE__.": $parsed\n";
    s/[\r\n]//g for ($exp, $parsed);
    $text =~ s/[\r\n]//g;
    cmp_ok($parsed, 'eq', $exp, "parse '$text'");
    if ($forbid) {
        $bbc2html->permit($forbid);
    }
}

eval {
    my $parsed = $bbc2html->render();
};
my $error = $@;
#warn __PACKAGE__.':'.__LINE__.": <<$@>>\n";
cmp_ok($error, '=~', 'Missing input', "Missing input for render()");

$bbc2html->permit('foobar');
my $allowed = $bbc2html->get_allowed;
#warn __PACKAGE__.':'.__LINE__.$".Data::Dumper->Dump([\$allowed], ['allowed']);
ok(
    (!grep { $_ eq 'foobar' } @$allowed),
    "permit() an unsupported tag");
