
=head1 NAME

Blog::Spam::Plugin::httpbl - Lookup submitters in the HTTP;bl list

=cut

=head1 ABOUT

This plugin is designed to test the submitters of comments against the
project honeypot RBL - HTTP;bl.

An IP which is listed in the service will be refused the ability to
submit comments - and this result will be cached for a week.

=cut

=head1 DETAILS

B<NOTE>: You must have an API key to use this function, and that
key should be stored in /etc/blogspam/httpbl.key.

You can find further details of the Project Honeypot via
http://www.projecthoneypot.org/httpbl_configure.php


=cut

=head1 AUTHOR

=over 4

=item Steve Kemp

http://www.steve.org.uk/

=back

=cut

=head1 LICENSE

Copyright (c) 2008-2010 by Steve Kemp.  All rights reserved.

This module is free software;
you can redistribute it and/or modify it under
the same terms as Perl itself.
The LICENSE file contains the full text of the license.

=cut



package Blog::Spam::Plugin::httpbl;

use strict;
use warnings;

use File::Path;
use Socket;


=begin doc

Constructor.  Called when this plugin is instantiated.

This merely saves away the name of our plugin.

=end doc

=cut

sub new
{
    my ( $proto, %supplied ) = (@_);
    my $class = ref($proto) || $proto;

    my $self = {};
    $self->{ 'name' } = $proto;

    # verbose?
    $self->{ 'verbose' } = $supplied{ 'verbose' } || 0;

    bless( $self, $class );
    return $self;
}


=begin doc

Return the name of this plugin.

=end doc

=cut

sub name
{
    my ($self) = (@_);
    return ( $self->{ 'name' } );
}




=begin doc

Test whether the IP address submitting the comment is listed
in the blacklist.

=end doc

=cut

sub testComment
{
    my ( $self, %params ) = (@_);

    #
    #  IP is mandatory - we will always have it.
    #
    my $ip = $params{ 'ip' };

    #
    #  But we cannot cope with non-IPv4 addresses.
    #
    return "OK" unless ( $ip =~ /^([0-9]+)\.([0-9]+)\.([0-9]+)\.([0-9]+)$/ );

    #
    #  Get the state directory which we'll use as a cache.
    #
    my $state = $params{ 'parent' }->getStateDir();
    my $cdir  = $state . "/cache/httpbl/";

    #
    #  Is the result cached?
    #
    my $safe = $ip;
    $safe =~ s/[:\.]/-/g;
    if ( -e "$cdir/$safe" )
    {
        return ("SPAM:Cached from HTTP;bl");
    }

    #
    #  Reverse for lookup
    #
    my $rev_ip = join( ".", reverse split( /\./, $ip ) );

    #
    #  Now lookup.
    #
    my $httpbl_key  = "keykeykeykey";
    my $httpbl_zone = "dnsbl.httpbl.org";
    my $name        = "$httpbl_key.$rev_ip.$httpbl_zone";

    #
    #  Get the key
    #
    if ( -e "/etc/blogspam/httpbl.key" )
    {
        if ( open( FILE, "<", "/etc/blogspam/httpbl.key" ) )
        {
            $httpbl_key = <FILE> || "";
            chomp($httpbl_key);
            close(FILE);
        }
    }

    #
    #  Fail?
    #
    my @a = gethostbyname($name);
    unless ( $#a > 3 )
    {
        return "OK";
    }

    #
    #  Work out what is going on.
    #
    @a = map {inet_ntoa($_)} @a[4 .. $#a];
    my ( undef, $days, $threat, $type ) = split( /\./, $a[0] );

    unless ( $type & 7 )
    {
        return "OK";
    }

    #
    #  Blocked.
    #
    #  Cache the result
    #
    if ( !-d $cdir )
    {
        mkpath( $cdir, { verbose => 0 } );
    }
    open( FILE, ">", "$cdir/$safe" ) or
      die "Failed to open $cdir/$safe - $!";
    print FILE "\n";
    close(FILE);

    #
    #  Return spam result
    #
    return ("SPAM:Listed in HTTP;bl");
}



=begin doc

Expire our cached entries once a week.

=end doc

=cut

sub expire
{
    my ( $self, $parent, $frequency ) = (@_);

    if ( $frequency eq "weekly" )
    {
        $self->{ 'verbose' } && print "Cleaning HTTP;bl Cache\n";

        my $state = $parent->getStateDir();
        my $cdir  = $state . "/cache/httpbl/";

        foreach my $entry ( glob( $cdir . "/*" ) )
        {

            #
            #  We're invoked once per week, but we
            # only want to remove files which are themselves
            # older than a week.
            #
            if ( -M $entry > 7 )
            {
                $self->{ 'verbose' } && print "\tRemoving: $entry\n";
                unlink($entry);
            }
        }
    }
}

1;
